/* ***************************************************************************************
* FILE:          DefaultControlTemplateMap.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DefaultControlTemplateMap is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(DefaultControlTemplateMap_H)
#define DefaultControlTemplateMap_H

#include "ControlTemplateMap.h"

class ControlTemplateInstance;

/**
 * The DefaultControlTemplateMap implements a ControlTemplateMap and it used during control template cloning.
 * Only to be used by a ControlTemplateCloningStrategy.
 */
class DefaultControlTemplateMap : public ControlTemplateMap
{
   public:
      DefaultControlTemplateMap();
      virtual ~DefaultControlTemplateMap();

      /**
       * Initialize with the correct ControlTemplateInstance. Two-phase-construction is required to enable its use with in a CachedControlTemplateInstance.
       */
      void Init(ControlTemplateInstance* controlTemplateInstance = 0);

      virtual FeatStd::UInt32 GetOwnerId() const;
      virtual FeatStd::UInt32 GetIndex() const;
      virtual Candera::Node2D* ResolveNodeClone(Candera::Node2D* node) const;
      virtual void ResolveNodeCloneArray(Candera::ArrayProperty<Candera::Node2D*>& cloneNodes, const Candera::ArrayProperty<Candera::Node2D*>& originalNodes) const;
      virtual Candera::RenderNode* ResolveNodeClone(Candera::RenderNode* node) const;
      virtual Candera::Effect2D* ResolveEffectClone(Candera::Effect2D* effect) const;
      virtual Candera::WidgetBase* ResolveWidgetClone(Candera::WidgetBase* widget) const;
      virtual void CloneProperties(const Candera::Widget2D& widget, Candera::Widget2D& widgetClone) const;
      virtual void CloneProperties(const BaseWidget2D& widget, BaseWidget2D& widgetClone) const;

   private:
      friend class ControlTemplateCloneTraverser2D;
      friend class ControlTemplateInstance;
      friend class CloneMappingComparator;

      FEATSTD_MAKE_CLASS_UNCOPYABLE(DefaultControlTemplateMap);

      struct NodeCloneMapping
      {
         const Candera::Node2D* _node;
         Candera::Node2D* _nodeClone;
      };

      struct EffectCloneMapping
      {
         const Candera::Effect2D* _effect;
         Candera::Effect2D* _effectClone;
      };

      struct WidgetCloneMapping
      {
         Candera::Node2D* _nodeClone;
         Candera::WidgetBase* _widget;
         Candera::WidgetBase* _widgetClone;
      };

      class NodeCloneMappingComparator
      {
         public:
            bool operator()(const NodeCloneMapping& a, const NodeCloneMapping& b) const
            {
               FEATSTD_LINT_NEXT_EXPRESSION(946, "Relational or subtract operator applied to pointers [MISRA C++ Rule 5-0-15], [MISRA C++ Rule 5-0-17], [MISRA C++ Rule 5-0-18]: changing this would affect the algorithm ")
               return a._node < b._node;
            }
      };

      class EffectCloneMappingComparator
      {
         public:
            bool operator()(const EffectCloneMapping& a, const EffectCloneMapping& b) const
            {
               FEATSTD_LINT_NEXT_EXPRESSION(946, "Relational or subtract operator applied to pointers [MISRA C++ Rule 5-0-15], [MISRA C++ Rule 5-0-17], [MISRA C++ Rule 5-0-18]: changing this would affect the algorithm ")
               return a._effect < b._effect;
            }
      };

      class WidgetCloneMappingComparator
      {
         public:
            bool operator()(const WidgetCloneMapping& a, const WidgetCloneMapping& b) const
            {
               FEATSTD_LINT_NEXT_EXPRESSION(946, "Relational or subtract operator applied to pointers [MISRA C++ Rule 5-0-15], [MISRA C++ Rule 5-0-17], [MISRA C++ Rule 5-0-18]: changing this would affect the algorithm ")
               return a._widget < b._widget;
            }
      };

      CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, node, CANDERA_LINT_REASON_NONCONST)
      static Courier::Int NodeComparator(Candera::Node2D* node, const NodeCloneMapping& mappingEntry);

      CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, node, CANDERA_LINT_REASON_NONCONST)
      static Courier::Int EffectComparator(Candera::Effect2D* node, const EffectCloneMapping& mappingEntry);

      CANDERA_SUPPRESS_LINT_FOR_SYMBOL(818, widget, CANDERA_LINT_REASON_NONCONST)
      static Courier::Int WidgetComparator(Candera::WidgetBase* widget, const WidgetCloneMapping& mappingEntry);

      /**
      * Clones the widget properties from the original widgets. Only to be called by a ControlTemplateCloneTraverser2D or a ControlTemplateCloningStrategy.
      */
      bool ClonePropertiesInternal();

      ControlTemplateInstance* _controlTemplateInstance;
      Courier::Vector<NodeCloneMapping> _nodeCloneMap;
      Courier::Vector<EffectCloneMapping> _effectCloneMap;
      Courier::Vector<WidgetCloneMapping> _widgetCloneMap;
};


#endif
