/* ***************************************************************************************
* FILE:          ControlTemplateMap.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ControlTemplateMap is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef CONTROL_TEMPLATE_MAP_H
#define CONTROL_TEMPLATE_MAP_H

#include <CanderaAssetLoader/AssetLoaderBase/ArrayProperty.h>

namespace Candera {
class Node2D;
class RenderNode;
class Effect2D;
class WidgetBase;
class Widget2D;
}


class BaseWidget2D;

namespace hmibase {
namespace widget {

/**
 * The ControlTemplateMap provides a map of all original nodes/widgets to the cloned instances.
 * During cloning the map can be used in the ControlTemplateCloneableWidget::CloneForm method to resolve
 * the correct new instances of nodes and widgets.
 */
class WidgetCloneHelper
{
   public:
      virtual ~WidgetCloneHelper() { }

      /**
       * Returns the owner id of the associated item.
       */
      virtual FeatStd::UInt32 GetOwnerId() const = 0;

      /**
      * Returns the index of the associated item.
      */
      virtual FeatStd::UInt32 GetIndex() const = 0;

      /**
       * Resolves the correct new instance of given node or returns the given node if it has not been cloned.
       */
      virtual Candera::Node2D* ResolveNodeClone(Candera::Node2D* node) const = 0;

      /**
      * Resolves the correct new instance of given node array.
      */
      virtual void ResolveNodeCloneArray(Candera::ArrayProperty<Candera::Node2D*>& cloneNodes, const Candera::ArrayProperty<Candera::Node2D*>& originalNodes) const = 0;

      /**
       * Resolves the correct new instance of given node or returns the given node if it has not been cloned.
       */
      virtual Candera::RenderNode* ResolveNodeClone(Candera::RenderNode* node) const = 0;

      /**
      * Resolves the correct new instance of a given effect or returns the given effect if it has not been cloned.
      */
      virtual Candera::Effect2D* ResolveEffectClone(Candera::Effect2D* effect) const = 0;

      /**
       * Resolves the correct new instance of given widget or returns the given widget if it has not been cloned.
       */
      virtual Candera::WidgetBase* ResolveWidgetClone(Candera::WidgetBase* widget) const = 0;

      /**
       * This method will be called to clone the base properties.
       */
      virtual void CloneProperties(const Candera::Widget2D& widget, Candera::Widget2D& widgetClone) const = 0;

      /**
       * This method will be called to clone the base properties.
       */
      virtual void CloneProperties(const BaseWidget2D& widget, BaseWidget2D& widgetClone) const = 0;
};


}
}


typedef hmibase::widget::WidgetCloneHelper ControlTemplateMap;

#endif
