/* ***************************************************************************************
* FILE:          ControlTemplateInstance.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ControlTemplateInstance is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Courier/Visualization/ViewScene2D.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ControlTemplateInstance.cpp.trc.h"
#endif

using namespace Candera;

#include <Candera/System/Rtti/Rtti.h>

#include "ControlTemplateInstance.h"
#include "ControlTemplateCloningStrategy.h"
#include "ControlTemplate.h"
#include <Widgets/2D/Size/SizeWidget2D.h>
#include <Widgets/2D/Layout/MarginWidget2D.h>
#include "Widgets/2D/BaseWidget2D.h"
#include "Widgets/2D/Common/ListItemAnimationDynamicPropertyHost.h"
#include "AnimationCloner.h"
#include <Widgets/2D/List/generated/ListEnums.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateCloningStrategy.h>
#include "Candera/EngineBase/Animation/AnimationBlendedProperty.h"
#include "ControlTemplateInstanceDisposer.h"


FeatStd::Int ControlTemplateInstance::_scrollAnimationSensitivity = 100;

static int sCTIInstances = 0;
FEATSTD_RTTI_BASECLASS_DEFINITION(ControlTemplateInstance);
ControlTemplateInstance::ControlTemplateInstance() :
   _scrollAnimationInput(ScrollAnimationInput::IndexBased),
   _owner(0),
   _ownerId(0),
   _index(0),
   _itemRootNode(0),
   _templateNode(0),
   _animationOffset(0),
   _disposer(0),
   _expandPlaybackFinished(false)
{
   ++sCTIInstances;
   _expandAnimationsManager.SetListener(this);
   _controlTemplateMap.Init(this);
}


ControlTemplateInstance::~ControlTemplateInstance()
{
   CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1579, "Cleanup made by DisposeContent.")
   DisposeContent();
   --sCTIInstances;

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ControlTemplateInstance::~ControlTemplateInstance remaining %d instances", sCTIInstances));
}


void ControlTemplateInstance::Dispose()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ControlTemplateInstance::Dispose %p, disposer= %p", this, _disposer));
   if (0 != _disposer)
   {
      _disposer->Dispose(this);
   }
}


bool ControlTemplateInstance::IsItemIntersectingPoint(Courier::ViewScene2D* view2D, Candera::Node2D* node, Candera::UInt32 pointX, Candera::UInt32 pointY)
{
   if ((0 != node) && (0 != view2D))
   {
      const Courier::ViewScene2D::CameraPtrVector& cameras = view2D->GetCameraPtrVector();
      for (SizeType cameraIndex = 0; cameraIndex < cameras.Size(); ++cameraIndex)
      {
         Candera::Camera2D* camera2D = cameras[cameraIndex];
         if (camera2D->IsRenderingEnabled())
         {
            const Candera::Vector2 point(static_cast<Courier::Float>(pointX), static_cast<Courier::Float>(pointY));
            if (node->IsPickIntersectingBoundingRectangle(*camera2D, point))
            {
               return true;
            }
         }
      }
   }
   return false;
}


void ControlTemplateInstance::DisposeContent()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ControlTemplateInstance::DisposeContent %p, itemRootNode= %p, owner= %p", this, _itemRootNode, _owner));
   if ((0 != _itemRootNode) && (0 != _owner))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ControlTemplateInstance::DisposeContent disposing %d widgets, %d stringIds", _widgets.Size(), _stringIds.Size()));
      for (SizeType i = 0; i < _widgets.Size(); ++i)
      {
         _widgets[i]->Finalize();
         _widgets[i]->Dispose();
         _widgets[i] = 0;
      }

      for (SizeType i = 0; i < _stringIds.Size(); ++i)
      {
         FEATSTD_DELETE(_stringIds[i]);
         _stringIds[i] = 0;
      }
      _itemRootNode->Unload(Candera::ScopeMask(), Candera::Node2D::Deep);
      _itemRootNode->Dispose();
      _itemRootNode = 0;
   }

   _templateNode = 0;

   _widgets.Clear();

   _stringIds.Clear();

   _names.clear();

   _owner = 0;
   _ownerId = 0;
}


void ControlTemplateInstance::Update()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ControlTemplateInstance::Update %p widgetsSIze=%d", this, _widgets.Size()));

   for (SizeType i = 0; i < _widgets.Size(); ++i)
   {
      _widgets[i]->Update();
   }
}


void ControlTemplateInstance::PositionAnimate(Candera::Float itemPosition)
{
   if (_itemRootNode != 0)
   {
      for (AnimationControllers::iterator it(_scrollAnimations.begin()); it != _scrollAnimations.end(); ++it)
      {
         Candera::Animation::AnimationController::SharedPointer& animation(it->second);
         if (!animation.PointsToNull())
         {
            if (_scrollAnimationInput == Candera::ScrollAnimationInput::PixelPositionBased)
            {
               animation->Animate(static_cast<Candera::UInt32>(itemPosition) - _animationOffset);
            }
            else
            {
               animation->Animate(static_cast<Candera::UInt32>(itemPosition * 10 * _scrollAnimationSensitivity));
            }
         }
      }
   }

   for (SizeType i = 0; i < _widgetsToUpdateAfterScrollAnimation.Size(); i++)
   {
      if (0 != _widgetsToUpdateAfterScrollAnimation[i])
      {
         _widgetsToUpdateAfterScrollAnimation[i]->Update();
      }
   }
}


void ControlTemplateInstance::SpeedAnimate(Candera::UInt32 itemSpeed)
{
   if (_itemRootNode != 0)
   {
      for (AnimationControllers::iterator it(_speedAnimations.begin()); it != _speedAnimations.end(); ++it)
      {
         Candera::Animation::AnimationController::SharedPointer& animation(it->second);
         if (!animation.PointsToNull())
         {
            animation->Animate(itemSpeed);
         }
      }
   }

   for (SizeType i = 0; i < _widgetsToUpdateAfterScrollAnimation.Size(); i++)
   {
      if (0 != _widgetsToUpdateAfterScrollAnimation[i])
      {
         _widgetsToUpdateAfterScrollAnimation[i]->Update();
      }
   }
}


bool ControlTemplateInstance::OnMessage(const Courier::Message& msg)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ControlTemplateInstance::OnMessage msgPtr=%p, msgId=%p widgetsSIze=%d", &msg, msg.GetId(), _widgets.Size()));

   for (SizeType widgetIndex = 0; widgetIndex < _widgets.Size(); ++widgetIndex)
   {
      if (_widgets[widgetIndex]->OnMessage(msg))
      {
         return true;
      }
   }
   return false;
}


void ControlTemplateInstance::Init()
{
   FEATSTD_DEBUG_ASSERT(_owner != 0); // fr83hi: LINT // Warning 613: prio2: Possible use of null pointer
   if (0 != _owner)
   {
      for (SizeType i = 0; i < _widgets.Size(); ++i)
      {
         _widgets[i]->SetParentView(_owner->GetParentView());
         _widgets[i]->Init(_owner->GetAssetProvider());
      }
   }
}


void ControlTemplateInstance::Clear()
{
   _widgets.Clear();
}


ControlTemplateInstance::CompositeAnimationTraverser::CompositeAnimationTraverser(ControlTemplateInstance* templateInstance, const ControlTemplateMap& controlTemplateMap, AnimationControllers& updatedScrollAnimations, AnimationControllers& updatedExpandAnimations, AnimationControllers& updatedSpeedAnimations) :
   _templateInstance(templateInstance),
   _controlTemplateMap(controlTemplateMap),
   _updatedScrollAnimations(updatedScrollAnimations),
   _updatedExpandAnimations(updatedExpandAnimations),
   _updatedSpeedAnimations(updatedSpeedAnimations)
{
}


Candera::TreeTraverser2D::TraverserAction ControlTemplateInstance::CompositeAnimationTraverser::ProcessNode(Candera::Node2D& node)
{
   Candera::CompositeGroup2D* compositeGroup(Candera::Dynamic_Cast<Candera::CompositeGroup2D*>(&node));
   if (0 != compositeGroup)
   {
      CompositeGroup2D::AnimationIterator it(compositeGroup->GetAnimationIterator());
      while (it.IsValid())
      {
         Candera::Animation::AnimationPlayer::SharedPointer animPlayer(Candera::Dynamic_Cast<Candera::Animation::AnimationPlayer::SharedPointer>(*it));

         _templateInstance->UpdateClonedAnimation(animPlayer, _controlTemplateMap, _updatedScrollAnimations, _updatedExpandAnimations, _updatedSpeedAnimations);

         it++;
      }
   }

   return ProceedTraversing;
}


void ControlTemplateInstance::UpdateClonedAnimations(const AnimationsType& animations, const ControlTemplateMap& controlTemplateMap)
{
   AnimationControllers updatedScrollAnimations;
   AnimationControllers updatedExpandAnimations;
   AnimationControllers updatedSpeedAnimations;

   _expandAnimationsManager.RemoveAllAnimations();

   for (UInt i(0); i < animations.GetCount(); ++i)
   {
      Animation::AnimationPlayer::SharedPointer player(animations.Get(i));
      UpdateClonedAnimation(player, controlTemplateMap, updatedScrollAnimations, updatedExpandAnimations, updatedSpeedAnimations);
   }

   if (0 != _templateNode)
   {
      CompositeAnimationTraverser traverser(this, controlTemplateMap, updatedScrollAnimations, updatedExpandAnimations, updatedSpeedAnimations);
      traverser.Traverse(*_templateNode);
   }

   _scrollAnimations = updatedScrollAnimations;
   _expandAnimations = updatedExpandAnimations;
   _speedAnimations = updatedSpeedAnimations;
}


class NoModifier : public ClonedAnimationModifier
{
   public:
      virtual void Modify(const Candera::Animation::AnimationController::SharedPointer& /*animation*/) override
      {
      }
};


class TimelineModifier : public ClonedAnimationModifier
{
   public:
      TimelineModifier(Int ratio) :
         _ratio(ratio)
      {}

      virtual void Modify(const Candera::Animation::AnimationController::SharedPointer& animation) override
      {
         if (!animation.PointsToNull())
         {
            for (SizeType pi(0); pi < animation->GetNumberOfProperties(); ++pi)
            {
               const SharedPointer<Animation::AnimationBlendedProperty>& prop(animation->GetProperty(pi));

               for (SizeType kfsi(0); kfsi < prop->GetNumberOfKeyframeSequences(); ++kfsi)
               {
                  const SharedPointer<Animation::AnimationKeyframeSequence>& keyframeSequence(prop->GetKeyframeSequence(kfsi));
                  Animation::SequenceTimeType* kfst(const_cast<Animation::SequenceTimeType*>(keyframeSequence->GetKeyframeSequenceTimes()));

                  for (Int ki(0); ki < keyframeSequence->GetKeyframeCount(); ++ki)
                  {
                     kfst[ki] *= _ratio;
                  }
               }
            }
         }
      }

   private:
      Int _ratio;
};


void ControlTemplateInstance::UpdateClonedAnimation(Animation::AnimationPlayer::SharedPointer& player, const ControlTemplateMap& controlTemplateMap, AnimationControllers& updatedScrollAnimations, AnimationControllers& updatedExpandAnimations, AnimationControllers& updatedSpeedAnimations)
{
   if (!player.PointsToNull())
   {
      ListItemAnimationType animationType(static_cast<ListItemAnimationType>(ListItemAnimationDynamicPropertyHost::GetListItemAnimationType(player)));

      static NoModifier s_noModifier;
      static TimelineModifier s_timelineModifier(_scrollAnimationSensitivity);

      switch (animationType)
      {
         case LIAScroll:
         {
            ClonedAnimationModifier* modifier(&s_noModifier);
            if (_scrollAnimationInput == Candera::ScrollAnimationInput::IndexBased)
            {
               modifier = &s_timelineModifier;
            }
            UpdateClonedAnimationByType(player, controlTemplateMap, _scrollAnimations, updatedScrollAnimations, modifier);
         }
         break;

         case LIAExtend:
         {
            const Candera::Animation::AnimationController::SharedPointer& animation(UpdateClonedAnimationByType(player, controlTemplateMap, _expandAnimations, updatedExpandAnimations, &s_noModifier));
            _expandAnimationsManager.AddAnimation(animation);
            break;
         }

         case LIASpeed:
         {
            ClonedAnimationModifier* modifier(&s_noModifier);
            UpdateClonedAnimationByType(player, controlTemplateMap, _speedAnimations, updatedSpeedAnimations, modifier);
            break;
         }

         default:
            // animation type not recognized
            break;
      }
   }
}


void ControlTemplateInstance::SetDisposer(ControlTemplateInstanceDisposer* disposer)
{
   _disposer = disposer;
}


bool ControlTemplateInstance::ClonePropertiesInternal()
{
   return _controlTemplateMap.ClonePropertiesInternal();
}


const Candera::Animation::AnimationController::SharedPointer ControlTemplateInstance::UpdateClonedAnimationByType(Animation::AnimationPlayer::SharedPointer& player, const ControlTemplateMap& controlTemplateMap, AnimationControllers& srcAnimations, AnimationControllers& dstAnimations, ClonedAnimationModifier* modifier)
{
   Candera::Animation::AnimationController::SharedPointer clonedAnimation(0);

   AnimationControllers::iterator it(srcAnimations.find(player.GetPointerToSharedInstance()));
   if (it == srcAnimations.end()) // animation not found
   {
      clonedAnimation = AnimationCloner::CloneAnimationController(player.GetPointerToSharedInstance(), ControlTemplateInstancePtr(this), controlTemplateMap);
      if (modifier != 0)
      {
         modifier->Modify(clonedAnimation);
      }
   }
   else // animation found
   {
      clonedAnimation = it->second;
   }

   if (!clonedAnimation.PointsToNull())
   {
      dstAnimations.insert(AnimationControllers::value_type(player.GetPointerToSharedInstance(), clonedAnimation));
   }

   return clonedAnimation;
}


void ControlTemplateInstance::OnPlaybackFinished(bool forward)
{
   _expandPlaybackFinished = true;
   Candera::Node2D* itemRootNode(GetItemRootNode());

   if (0 != itemRootNode)
   {
      const tSharedPtrIDataItem& dataContext(ControlTemplate::GetDataContext(*itemRootNode));
      if (!dataContext.PointsToNull())
      {
         ListItemExpandEndUpdMsg* endMsg(COURIER_MESSAGE_NEW(ListItemExpandEndUpdMsg)(_ownerId, _index, forward));
         if (0 != endMsg)
         {
            endMsg->Post();
         }
      }
   }
}


void ControlTemplateInstance::SetWidgetName(Candera::WidgetBase& widget, const Candera::String& name)
{
   _names[name.GetCString()] = name;
   widget.SetName(name.GetCString());
}


void ControlTemplateInstance::SetWidgetStringId(Candera::WidgetBase& widget, const Candera::StringIdentifier* stringId)
{
   if (0 != stringId)
   {
      //Candera::StringIdentifier* stringIdClone = FEATSTD_NEW(Candera::StringIdentifier);

      //if (0 == stringIdClone)
      //{
      //   ETG_TRACE_ERR(("Cloned string identifier could not be allocated."));
      //   return;
      //}

      //static_cast<void>(_stringIds.Add(stringIdClone));

      // clone ID
      const Char* id = stringId->GetId();
      FeatStd::String& cachedId = _names[id];
      cachedId = id;

      //stringIdClone->SetId(cachedId.GetCString(), NULL);

      // clone owner chain
      //Candera::StringIdentifier* childClone = stringIdClone;
      //const Candera::StringIdentifier* owner = stringId->GetOwner();
      //while (0 != owner)
      //{
      //   Candera::StringIdentifier* ownerClone = FEATSTD_NEW(Candera::StringIdentifier);

      //   if (0 == ownerClone)
      //   {
      //      ETG_TRACE_ERR(("Cloned string identifier owner could not be allocated."));
      //      return;
      //   }

      //   if (widget.GetStringId()->GetOwner() == 0)
      //   {
      //      // fr83hi 2016/11/23: with following fix we got a problem to identify the identifier of e.g. the button sender
      //      //  Problem with Names in ListTemplate instances : "listTplate_Btn5RowSimple/ListItem_8002_0_0#ListText5Items/Text"

      //      //// widget.SetOwnerId(ownerClone);
      //   }

      //   static_cast<void>(_stringIds.Add(ownerClone));
      //   // clone owner ID
      //   const Char* id = owner->GetId();
      //   FeatStd::String& cachedId = _names[id];
      //   cachedId = id;
      //   ownerClone->SetId(cachedId.GetCString(), NULL);

      //   // connect child and owner clones
      //   childClone->SetOwner(ownerClone);
      //   // go one step up the owner chain
      //   childClone = ownerClone;
      //   owner = owner->GetOwner();
      //}

      //widget.SetStringId(stringIdClone->GetId(), NULL);
      widget.SetStringId(cachedId.GetCString(), NULL);
   }
}


void ControlTemplateInstance::AddWidget(Candera::WidgetBase* widget)
{
   if (0 != widget)
   {
      static_cast<void>(_widgets.Add(widget));
   }
}


void ControlTemplateInstance::AddWidgetToUpdateAfterScrollAnimations(Candera::WidgetBase* widget)
{
   if (0 != widget)
   {
      static_cast<void>(_widgetsToUpdateAfterScrollAnimation.Add(widget));
   }
}
