/* ***************************************************************************************
* FILE:          ControlTemplateCloningStrategy.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ControlTemplateCloningStrategy is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "AppDataCleanTraverser.h"
#include "ControlTemplateCloneTraverser2D.h"
#include "ControlTemplateCloningStrategy.h"
#include "AnimationCloner.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_LIST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ControlTemplateCloningStrategy.cpp.trc.h"
#endif
#include <CanderaPlatform/Device/Common/Effects/SolidColorBrushBlend.h>
#include "Widgets/2D/Common/ListItemAnimationDynamicPropertyHost.h"
#include <Widgets/2D/ControlTemplate/CloneEvents.h>
#include <Candera/System/Rtti/Rtti.h>
#include <Widgets/2D/List/ListWidget2D.h>

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/LayoutTypes.h>
#else
#include <Candera/Engine2D/Layout/LayoutTypes.h>
#endif


using namespace Candera;


FEATSTD_RTTI_BASECLASS_DEFINITION(ControlTemplateCloningStrategy);
Candera::Rectangle& ControlTemplateCloningStrategy::LayoutAreaDefault()
{
   FEATSTD_UNSYNCED_STATIC_OBJECT(Candera::Rectangle, s_layoutAreaDefault);
   return s_layoutAreaDefault;
}


ControlTemplateCloningStrategy::ControlTemplateCloningStrategy(CloneCachingStrategy& cachingStrategy) :
   _cachingStrategy(cachingStrategy)
{
}


ControlTemplateInstancePtr ControlTemplateCloningStrategy::CloneNode(const ITemplateRetriever& templateRetriever, const tSharedPtrListDataItem& source, BaseWidget2D* owner, FeatStd::UInt32 ownerId, FeatStd::UInt32 index)
{
   Candera::Node2D* templateNode = templateRetriever.GetTemplate(source);
   if (templateNode != 0)
   {
      const ListWidget2DBase::TemplateScrollAnimationType& templateAnimations = templateRetriever.GetTemplateScrollAnimations();

      ControlTemplateInstancePtr clonedInstance(_cachingStrategy.Retrieve(*templateNode, source, owner, ownerId, index, templateAnimations));

      if (clonedInstance.PointsToNull() || templateRetriever.IsInvalidTemplate(templateNode))
      {
         clonedInstance = Clone(*templateNode, source, owner, ownerId, index, templateAnimations);

         _cachingStrategy.ProcessAfterCloning(clonedInstance, index);
      }

      if (!clonedInstance.PointsToNull())
      {
         clonedInstance->SetIndex(index);
      }
      return clonedInstance;
   }

   return ControlTemplateInstancePtr(0);
}


void ControlTemplateCloningStrategy::Flush(FeatStd::UInt32 firstVisibleIndex, FeatStd::UInt32 lastVisibleIndex)
{
   _cachingStrategy.ProcessBeforeFlush(firstVisibleIndex, lastVisibleIndex);
}


void ControlTemplateCloningStrategy::Finalize()
{
   _cachingStrategy.Finalize();
}


void ControlTemplateCloningStrategy::Dispose(ControlTemplateInstance* instance)
{
   bool dispose(_cachingStrategy.ProcessBeforeDisposed(instance));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ControlTemplateCloningStrategy::Dispose %p, _cachingStrategy= %p, dispose=%d", this, &_cachingStrategy, dispose));

   if (dispose)
   {
      if (0 != instance)
      {
         EventHub::Notify(DisposeEvent(instance));

         Node2D* rootNode(instance->GetItemRootNode());
         if (0 != rootNode)
         {
            AppDataCleanTraverser appDataCleanTraverser;
            appDataCleanTraverser.Traverse(*rootNode);
         }

         FEATSTD_DELETE(instance);
      }
   }
}


#ifdef BACKGROUND_DEBUG

const Candera::Color colorTable[] =
{
   Candera::Color(1.0, 0.0, 1.0, 0.75),
   Candera::Color(0.4, 0.4, 0.0, 0.75),
   Candera::Color(0.0, 0.6, 0.0, 0.75),
   Candera::Color(0.0, 0.0, 0.5, 0.75),
   Candera::Color(1.0, 1.0, 0.0, 0.75),
   Candera::Color(0.4, 0.0, 1.0, 0.75),
   Candera::Color(0.0, 0.0, 0.6, 0.75),
   Candera::Color(0.6, 0.0, 0.4, 0.75),
   Candera::Color(0.4, 0.5, 0.4, 0.75),
   Candera::Color(0.0, 0.6, 0.4, 0.75),
};


const FeatStd::Int32 colorsSize = sizeof(colorTable) / sizeof(Candera::Color);

static FeatStd::Int32 colorIndex = 0;

#endif


ControlTemplateInstancePtr ControlTemplateCloningStrategy::Clone(Candera::Node2D& templateNode, const tSharedPtrIDataItem& source, BaseWidget2D* owner, FeatStd::UInt32 ownerId, FeatStd::UInt32 index, const ListWidget2DBase::TemplateScrollAnimationType& templateAnimations)
{
   ControlTemplateInstancePtr controlTemplateInstance(CANDERA_NEW(ControlTemplateInstance));
   if (!controlTemplateInstance.PointsToNull())
   {
      Candera::Group2D* item = Candera::Group2D::Create();
      if (0 != item)
      {
         Candera::Layouter::SetHorizontalAlignment(*item, Candera::HStretch);	// sat2hi: left to stretch changed 2016/06/06
         Candera::Layouter::SetVerticalAlignment(*item, Candera::VStretch);	// sat2hi: top to stretch changed 2016/06/06

#ifdef BACKGROUND_DEBUG
         Candera::RenderNode* backgound(Candera::RenderNode::Create());
         Candera::SolidColorBrushBlend::SharedPointer solidColor(Candera::SolidColorBrushBlend::Create());
         if ((0 != backgound) && (!solidColor.PointsToNull()))
         {
            backgound->AddEffect(solidColor.GetPointerToSharedInstance());
            Candera::Layouter::SetHorizontalAlignment(*backgound, Candera::HStretch);
            Candera::Layouter::SetVerticalAlignment(*backgound, Candera::VStretch);
            Candera::Layouter::SetStretchBehavior(*backgound, LayoutAlignment::StretchBehavior::Fill);

            Candera::Color color(colorTable[colorIndex]);
            solidColor->GetSolidColorBrush().Color().Set(color);
            solidColor->GetSolidColorBrush().Size().Set(Candera::Vector2(1.0, 1.0));

            item->AddChild(backgound);

            colorIndex = (colorIndex + 1) % colorsSize;
         }
#endif

         controlTemplateInstance->SetOwner(owner);
         controlTemplateInstance->SetOwnerId(ownerId);
         controlTemplateInstance->SetIndex(index);
         controlTemplateInstance->SetItemRootNode(item);
         controlTemplateInstance->SetTemplateNode(&templateNode);

         DefaultControlTemplateMap& controlTemplateMap(controlTemplateInstance->GetControlTemplateMap());
         ControlTemplateCloneTraverser2D cloneTraverser(*controlTemplateInstance, source, controlTemplateMap);
         cloneTraverser.Traverse(templateNode);

         Node2D* rootClone = cloneTraverser.GetRootClone();
         if (0 != rootClone)
         {
            // Set rendering enabled of cloned root node to true, as template node might be hidden
            rootClone->SetRenderingEnabled(true);
         }

         // check cached layout
         ListWidget2D* listWidget = Candera::Dynamic_Cast<ListWidget2D*>(owner);
         if (0 != listWidget && listWidget->GetCachedLayout())
         {
            Node2D* cachedLayoutNode = controlTemplateInstance->GetCachedLayoutNode();
            if (cachedLayoutNode != 0)
            {
               if (!Layouter::IsCached(*cachedLayoutNode))
               {
                  Layouter::SetCached(*cachedLayoutNode, true);
               }
               else
               {
                  // If cached flag already set on the node with cached layout, invalidate the layout of the node as the layout of template node was already validated
                  Layouter::InvalidateLayout(cachedLayoutNode);
               }
            }
         }

         ControlTemplateInstanceOwner* controlTemplateInstanceOwner = dynamic_cast<ControlTemplateInstanceOwner*>(owner);
         if (0 != controlTemplateInstanceOwner)
         {
            controlTemplateInstanceOwner->OnCloneUpdateBegin(controlTemplateInstance);
         }
         controlTemplateInstance->UpdateClonedAnimations(templateAnimations, controlTemplateMap);
         controlTemplateInstance->Update();
         if (0 != controlTemplateInstanceOwner)
         {
            controlTemplateInstanceOwner->OnCloneUpdateEnd(controlTemplateInstance);
         }
      }
      EventHub::Notify(CloneNodeEvent(controlTemplateInstance, templateNode, source, owner));
   }
   return controlTemplateInstance;
}
