/* ***************************************************************************************
* FILE:          ControlTemplate.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ControlTemplate is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "ViewWidgetEnumerator.h"

#include "ControlTemplate.h"

void ControlTemplate::SetControlTemplateInstance(Candera::Node2D& node, ControlTemplateInstance* value)
{
   ControlTemplateData* controlTemplateData = GetControlTemplateData(node);
   if (0 != controlTemplateData)
   {
      controlTemplateData->SetControlTemplateInstance(value);
   }
}


bool ControlTemplate::IsControlTemplateInstanceSet(const Candera::Node2D& node)
{
   ControlTemplateData* controlTemplateData = TryGetControlTemplateData(node);
   return (0 == controlTemplateData) ? false : (0 != controlTemplateData->GetControlTemplateInstance());
}


ControlTemplateInstance* ControlTemplate::GetControlTemplateInstance(const Candera::Node2D& node)
{
   ControlTemplateData* controlTemplateData = TryGetControlTemplateData(node);
   return (0 == controlTemplateData) ? 0 : controlTemplateData->GetControlTemplateInstance();
}


void ControlTemplate::AddAssociatedWidget(Candera::Node2D& node, Candera::WidgetBase* value)
{
   ControlTemplateData* controlTemplateData = GetControlTemplateData(node);
   if (0 != controlTemplateData)
   {
      controlTemplateData->AddAssociatedWidget(value);
   }
}


void ControlTemplate::SetDataContext(Candera::Node2D& node, const tSharedPtrIDataItem& value)
{
   ControlTemplateData* controlTemplateData = GetControlTemplateData(node);
   if (0 != controlTemplateData)
   {
      controlTemplateData->SetDataContext(value);
   }
}


bool ControlTemplate::IsDataContextSet(const Candera::Node2D& node, bool followParent)
{
   const ControlTemplateData* controlTemplateData = TryGetControlTemplateDataWithDataContext(node, followParent);
   return (0 == controlTemplateData) ? false : !controlTemplateData->GetDataContext().PointsToNull();
}


tSharedPtrIDataItem ControlTemplate::GetDataContext(const Candera::Node2D& node, bool followParent)
{
   const ControlTemplateData* controlTemplateData = TryGetControlTemplateDataWithDataContext(node, followParent);
   return (0 == controlTemplateData) ? tSharedPtrIDataItem() : controlTemplateData->GetDataContext();
}


void ControlTemplate::SetTemplateNode(Candera::Node2D& node, bool value)
{
   ControlTemplateData* controlTemplateData = GetControlTemplateData(node);
   if (0 != controlTemplateData)
   {
      controlTemplateData->SetTemplateNode(value);
   }
}


bool ControlTemplate::IsTemplateNode(const Candera::Node2D& node, bool followParent)
{
   const Candera::Node2D* current = &node;
   while (0 != current)
   {
      ControlTemplateData* controlTemplateData = TryGetControlTemplateData(*current);
      if ((0 != controlTemplateData) && (controlTemplateData->IsTemplateNode()))
      {
         return true;
      }
      current = (followParent) ? current->GetParent() : 0;
   }
   return false;
}


WidgetBaseEnumerator ControlTemplate::EnumerateAssociatedWidgets(Courier::ViewScene& view, const Candera::Node2D& node)
{
   ControlTemplateData* controlTemplateData = TryGetControlTemplateData(node);
   if (0 != controlTemplateData)
   {
      return controlTemplateData->EnumerateAssociatedWidgets();
   }
   WidgetBaseEnumerator enumerator(CANDERA_NEW(ViewWidgetEnumerator)(view.GetFrameworkWidgetPtrVector(), node));
   return enumerator;
}


void ControlTemplate::ResetControlTemplateData(Candera::Node2D& node, bool deleteTemplateData)
{
   ControlTemplateData* controlTemplateData = node.GetValue(CdaDynamicPropertyInstance(ControlTemplateData));
   if (0 != controlTemplateData)
   {
      if (deleteTemplateData)
      {
         FEATSTD_DELETE(controlTemplateData);
      }
      node.SetValue(CdaDynamicPropertyInstance(ControlTemplateData), 0);
   }
}


const Candera::DynamicProperties::DynamicPropertyHost* ControlTemplate::ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host)
{
   CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
   return static_cast<const Candera::Node2D*>(host)->GetParent();
}


ControlTemplateData* ControlTemplate::TryGetControlTemplateDataWithDataContext(const Candera::Node2D& node, bool followParent)
{
   const Candera::Node2D* current = &node;
   while (0 != current)
   {
      ControlTemplateData* controlTemplateData = TryGetControlTemplateData(*current);
      if ((0 != controlTemplateData) && (!controlTemplateData->GetDataContext().PointsToNull()))
      {
         return controlTemplateData;
      }
      current = (followParent) ? current->GetParent() : 0;
   }
   return 0;
}


ControlTemplateData* ControlTemplate::TryGetControlTemplateData(const Candera::Node2D& node)
{
   ControlTemplateData* controlTemplateData = node.GetValue(CdaDynamicPropertyInstance(ControlTemplateData));
   return controlTemplateData;
}


ControlTemplateData* ControlTemplate::GetControlTemplateData(Candera::Node2D& node)
{
   ControlTemplateData* controlTemplateData = TryGetControlTemplateData(node);
   if (0 == controlTemplateData)
   {
      controlTemplateData = CANDERA_NEW(ControlTemplateData)();
      FEATSTD_DEBUG_ASSERT(0 != controlTemplateData);
      node.SetValue(CdaDynamicPropertyInstance(ControlTemplateData), controlTemplateData);
   }
   return controlTemplateData;
}
