/* ***************************************************************************************
* FILE:          ComboBoxWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ComboBoxWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ComboBoxWidget2D.h"

//source widgets
#include <Widgets/2D/Button/ButtonWidget2D.h>
#include <Widgets/2D/WidgetFinder2D.h>
#include <Widgets/2D/List/ListWidget2D.h>
#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_COMBOBOX
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ComboBoxWidget2D.cpp.trc.h"
#endif

#define LISTID_MAX_ITR 2

namespace hmibase {
namespace widget {
namespace combobox {

CGI_WIDGET_RTTI_DEFINITION(ComboBoxWidget2D);

/****************************************************************************/

ComboBoxWidget2D::ComboBoxWidget2D() : _isOpen(false), _invalid(true), _comboBoxListId(0), _listIdIterator(0)
{
   _comboboxLayouter.SetComboBoxWidget(this);
}


/****************************************************************************/

ComboBoxWidget2D::~ComboBoxWidget2D()
{
   const ContentNodeType& node = GetContentNode();
   if (NULL != node)
   {
      node->SetLayouter(NULL);
   }
}


/***************************************************************************/

void ComboBoxWidget2D::Update()
{
   Base::Update();
   if (_invalid)
   {
      UpdateContentVisibility();
      _invalid = false;
   }
   if ((_listIdIterator < LISTID_MAX_ITR) && (_comboBoxListId == 0))
   {
      SearchChildList();
   }
}


/***************************************************************************/

void ComboBoxWidget2D::OnChanged(Courier::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case ContentNodePropertyId:
      {
         const ContentNodeType& contentNode = GetContentNode();
         if ((NULL != contentNode) && (contentNode->GetLayouter() != &_comboboxLayouter))
         {
            contentNode->SetLayouter(&_comboboxLayouter);
         }
      }
      break;
      default:
         break;
   }
}


/*****************************************************************************/

void ComboBoxWidget2D::UpdateContentVisibility()
{
   const ContentNodeType& contentNode = GetContentNode();

   if ((NULL != contentNode) && (contentNode->IsRenderingEnabled() != _isOpen))
   {
      contentNode->SetRenderingEnabled(_isOpen);

      Invalidate();
      const ContentNodeType& contentNode = GetContentNode();
      if (NULL != contentNode)
      {
         Candera::Layouter::InvalidateLayout(contentNode);
      }

      PostComboBoxUpdMsg();
   }
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UpdateContentVisibility id=%d open=%u %s",
                       GetComboBoxId(),
                       _isOpen,
                       HMIBASE_TO_STRING_VW(this)));
}


/****************************************************************************/

void ComboBoxWidget2D::PostComboBoxUpdMsg()
{
   if (NULL != GetParentView())
   {
      ComboBoxUpdMsg* msg = COURIER_MESSAGE_NEW(ComboBoxUpdMsg)(this->GetParentView()->GetId(), Courier::Identifier(this->GetLegacyName()), this->GetUserData(), GetComboBoxId(), _isOpen);
      if (NULL != msg)
      {
         msg->Post();
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostComboBoxUpdMsg Posted id=%d open=%u %s",
                             GetComboBoxId(),
                             _isOpen,
                             HMIBASE_TO_STRING_VW(this)));
      }
   }
}


/****************************************************************************/

bool ComboBoxWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ComboBoxWidget2D* original = CLONEABLE_WIDGET_CAST<const ComboBoxWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }
      SetButtonNode(controlTemplateMap.ResolveNodeClone(original->GetButtonNode()));
      SetContentNode(controlTemplateMap.ResolveNodeClone(original->GetContentNode()));
      SetViewportNode(controlTemplateMap.ResolveNodeClone(original->GetViewportNode()));
      SetComboBoxId(original->GetComboBoxId());
      SetContentSize(original->GetContentSize());
      cloned = true;
   }
   return cloned;
}


/****************************************************************************/

bool ComboBoxWidget2D::OnTouch(const Candera::Camera2D& camera, const Candera::Vector2& touchedPoint)
{
   bool nodeTouched = Base::OnTouch(camera, touchedPoint);
   if (GetContentNode() != NULL && !nodeTouched && _isOpen)
   {
      /* Below line of chnage is required when custom bouding rectangle is set for combobox node which excludes contentnode node.
      eg: Bouding width of combobox button is changed and not the content.*/
      nodeTouched = IsPickIntersectingNode(camera, GetContentNode(), touchedPoint);
      if (!nodeTouched)
      {
         _isOpen = false;
         _invalid = true;
         triggerUpdate();
      }
   }

   return nodeTouched;
}


/****************************************************************************/
bool ComboBoxWidget2D::OnMessage(const Courier::Message& msg)
{
   bool msgConsumed = Base::OnMessage(msg);

   switch (msg.GetId())
   {
      case ComboBoxReqMsg::ID:
      {
         const ComboBoxReqMsg* comboBoxReqMsg = Courier::message_cast<const ComboBoxReqMsg*>(&msg);
         if (NULL != comboBoxReqMsg)
         {
            msgConsumed = OnComboBoxReqMsg(*comboBoxReqMsg);
         }
      }
      break;
      case ListStatusUpdMsg::ID:
      {
         const ListStatusUpdMsg* listStatusUpdMsg = Courier::message_cast<const ListStatusUpdMsg*>(&msg);
         if (NULL != listStatusUpdMsg && (listStatusUpdMsg->GetListId() != _comboBoxListId)) //If combobox's List is scrolled, combobox will not be closed
         {
            msgConsumed = OnListStatusUpMsg(*listStatusUpdMsg);
         }
      }
      break;
      case ButtonReactionMsg::ID:
      {
         const ButtonReactionMsg* buttonReactionMsg = Courier::message_cast<const ButtonReactionMsg*>(&msg);
         if (NULL != buttonReactionMsg)
         {
            msgConsumed = OnButtonReactionMsg(*buttonReactionMsg);
         }
      }
      break;
      default:
         break;
   }
   return msgConsumed;
}


/****************************************************************************/

void ComboBoxWidget2D::SearchChildList()
{
   Candera::Scene2DContext* sceneContext = WidgetFinder::GetSceneContext(this);

   ListWidget2D* listwidget = NULL;
   if ((NULL != sceneContext) && (NULL != GetNode()))
   {
      listwidget = WidgetFinder::FindDescendantWidget<ListWidget2D>(sceneContext, GetNode());
      _comboBoxListId = ((listwidget == NULL) ? 0 : listwidget->GetListId());
      if (_comboBoxListId != 0)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ComboBoxWidget2D::SearchChildList SubList Found = %d , %s", _comboBoxListId, HMIBASE_TO_STRING_VW(this)));
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ComboBoxWidget2D::SearchChildList No SubList Found as ComboBox child %s", HMIBASE_TO_STRING_VW(this)));
      }
   }
   ++_listIdIterator;
}


/****************************************************************************/

bool ComboBoxWidget2D::OnComboBoxReqMsg(const ComboBoxReqMsg& msg)
{
   bool open = _isOpen;
   bool result = false;

   if ((msg.GetComboBoxId() != 0u) && (msg.GetComboBoxId() == GetComboBoxId()))
   {
      result = true;
      switch (msg.GetAction())
      {
         case enComboBoxAction::Open:
            open = true;
            break;
         case enComboBoxAction::Close:
            open = false;
            break;
         case enComboBoxAction::Toggle:
            open = !_isOpen;
            break;
         default:
            break;
      }
   }
   //close this dropdown without consuming the message
   //this allows us to close all dropdowns with the same message
   else if ((msg.GetComboBoxId() == 0u) && (msg.GetAction() == enComboBoxAction::Close))
   {
      open = false;
   }

   if (_isOpen != open)
   {
      _isOpen = open;
      _invalid = true;
      triggerUpdate();
   }

   return result;
}


/****************************************************************************/

bool ComboBoxWidget2D::OnListStatusUpMsg(const ListStatusUpdMsg& msg)
{
   switch (msg.GetStatus())
   {
      case ListScrolling:
      case ListSwiping:
      case ListEncoder:
      case ListScrollBar:
      {
         if (_isOpen)
         {
            _isOpen = false;
            _invalid = true;
            triggerUpdate();
         }
      }
      break;
      default:
         break;
   }

   //don't consume this broadcast message so that others can also react on it
   return false;
}


bool ComboBoxWidget2D::OnButtonReactionMsg(const ButtonReactionMsg& msg)
{
   if ((NULL != GetButtonNode()) && (msg.GetEnReaction() == enRelease) && (msg.GetView() == GetParentView()->GetId()))
   {
      ButtonWidget2D* btnWidget = WidgetFinder::FindDescendantWidget<ButtonWidget2D>(WidgetFinder::GetSceneContext(this), GetButtonNode());
      if (NULL != btnWidget)
      {
         if (msg.GetSender() == Courier::Identifier(btnWidget->GetLegacyName()))
         {
            _isOpen = !_isOpen;
            _invalid = true;
            triggerUpdate();
            return true;
         }
      }
   }
   return false;
}


/****************************************************************************/

Candera::Vector2 ComboBoxWidget2D::ComboBoxLayouter::OnMeasure(const Candera::AbstractNodePointer& node, const Candera::Vector2& clientArea)
{
   Candera::Float clientWidth = 0.0F;
   Candera::Float clientHeight = 0.0F;
   Candera::Node2D* node2D = NULL;
   Candera::Node2D* child = NULL;
   if (node.IsValid())
   {
      node2D = node.ToNode2D();
   }
   if ((NULL != node2D) && (node2D->IsEffectiveRenderingEnabled()))
   {
      Candera::Vector2 comboBoxSize = _comboboxWidget->GetContentSize();
      clientWidth = (comboBoxSize.GetX() >= 0) ? comboBoxSize.GetX() : clientArea.GetX();
      clientHeight = (comboBoxSize.GetY() >= 0) ? comboBoxSize.GetY() : clientArea.GetY();
      child = node2D->GetFirstChild();
   }
   SetMeasureHeight(clientHeight);
   Candera::Vector2 childArea(clientWidth, clientHeight);
   while (NULL != child)
   {
      Layouter* childLayouter = child->GetLayouter();
      if (NULL != childLayouter)
      {
         childLayouter->Measure(*child, childArea);
      }
      child = child->GetNextSibling();
   }
   return Candera::Vector2(0.0f, 0.0f);
}


/****************************************************************************/

void ComboBoxWidget2D::ComboBoxLayouter::OnArrange(const Candera::AbstractNodePointer& node, const Candera::Rectangle& clientArea)
{
   Candera::Node2D& node2D = *(node.ToNode2D());
   Candera::Float clientWidth = 0.0F;
   Candera::Float clientHeight = 0.0F;
   if (node2D.IsEffectiveRenderingEnabled())
   {
      Candera::Vector2 comboBoxSize = _comboboxWidget->GetContentSize();
      clientWidth = (comboBoxSize.GetX() >= 0) ? comboBoxSize.GetX() : clientArea.GetWidth();
      clientHeight = (comboBoxSize.GetY() >= 0) ? comboBoxSize.GetY() : GetMeasureHeight();
   }
   Candera::Node2D* child = node2D.GetFirstChild();
   if (NULL != child)
   {
      Layouter* childLayouter = child->GetLayouter();
      if (NULL != childLayouter)
      {
         Candera::Rectangle childArea(0.0F, 0.0F, clientWidth, clientHeight);
         childLayouter->Arrange(*child, childArea);
         if ((NULL != _comboboxWidget) && (NULL != _comboboxWidget->GetViewportNode()) && (NULL != _comboboxWidget->GetButtonNode()))
         {
            Candera::Vector2 viewportSize;
            Candera::Layouter* layouter = _comboboxWidget->GetViewportNode()->GetLayouter();
            if (NULL != layouter)
            {
               viewportSize = layouter->GetClientSize(*_comboboxWidget->GetViewportNode());
            }

            Candera::Float listPos = _comboboxWidget->GetViewportNode()->GetWorldPosition().GetY();
            Candera::Float listHeight = viewportSize.GetY() + listPos;

            Candera::Float contentPos = child->GetWorldPosition().GetY();
            Candera::Float contentHeight = clientHeight + contentPos;

            if (listHeight < contentHeight)
            {
               Candera::Rectangle rect;
               _comboboxWidget->GetButtonNode()->GetEffectiveBoundingRectangle(rect);
               Candera::Float finalPos = ((child->GetPosition().GetY()) - (clientHeight + rect.GetHeight()));
               child->SetPosition(Candera::Vector2(0.0, finalPos));
            }
         }
      }
   }
}


}// namespace combobox
}// namespace widget
}// namespace hmibase
