/* ***************************************************************************************
* FILE:          ColorSwitchWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ColorSwitchWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ColorSwitchWidget2D.h"
#include <View/CGI/CgiExtensions/ArrayUtils.h>
#include <Widgets/2D/WidgetFinder2D.h>
#include <Widgets/2D/Appearance/Appearance2D.h>
#include <Widgets/utils/ColorPropertyAccessor.h>

//source widgets
#include <Widgets/2D/Button/ButtonWidget2D.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>
#include <Widgets/2D/DirectTextureConsumer/DirectTextureConsumer2D.h>
#include <Widgets/2D/Gesture/GestureWidget2D.h>
#include <Widgets/2D/Gizmo/GizmoWidget2D.h>
#include <Widgets/2D/List/ListWidget2D.h>
#include <Widgets/2D/ScrollBar/ScrollBarWidget2D.h>
#include <Widgets/2D/Slider/SliderWidget2D.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_APPEARANCE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include <trcGenProj/Header/ColorSwitchWidget2D.cpp.trc.h>
#endif


namespace hmibase {
namespace widget {
namespace color {


CGI_WIDGET_RTTI_DEFINITION(ColorSwitchWidget2D);
using namespace hmibase::widget::appearance;

static const FeatStd::UInt8 MAX_SEARCH_COUNT = 2;

/*****************************************************************************/
ColorSwitchWidget2D::ColorSwitchWidget2D() : Base(), _colorPropertyAccessor(NULL), _sourceSearchCount(0), _targetSearchCount(0)
{
}


/*****************************************************************************/
ColorSwitchWidget2D::~ColorSwitchWidget2D()
{
   ClearColorPropertyAccessor();
   _colorPropertyAccessor = NULL;
}


/*****************************************************************************/
void ColorSwitchWidget2D::ClearColorPropertyAccessor()
{
   if (_colorPropertyAccessor != NULL)
   {
      //clear appearance before deleting the property accessor
      ClearAppearance();

      _sourceSearchCount = 0;
      _targetSearchCount = 0;

      FEATSTD_DELETE(_colorPropertyAccessor);
      _colorPropertyAccessor = NULL;
   }
}


/*****************************************************************************/
bool ColorSwitchWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned = false;
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ColorSwitchWidget2D* original = CLONEABLE_WIDGET_CAST<const ColorSwitchWidget2D*>(originalWidget);
      if (original != NULL)
      {
         SetTarget(original->GetTarget());
         SetIndexSource(original->GetIndexSource());
         SetStateFlags(original->GetStateFlags());
         if (!hmibase::util::Arrays::equals(GetColors(), original->GetColors()))
         {
            SetColors(original->GetColors());
         }

         if (ControlTemplateBinding::IsSelectedBindable(*this))
         {
            Candera::Int32 value = ControlTemplateBinding::GetSelectedValue(*this);
            FEATSTD_DEBUG_ASSERT(value >= 0);
            SetIndex(static_cast<IndexType>(value));
         }
         else
         {
            SetIndex(original->GetIndex());
         }

         cloned = true;
      }
   }
   return cloned;
}


/*****************************************************************************/
void ColorSwitchWidget2D::OnBeforeNodeChanged()
{
   Base::OnBeforeNodeChanged();

   ClearColorPropertyAccessor();
}


/*****************************************************************************/
void ColorSwitchWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case TargetPropertyId:
      case IndexSourcePropertyId:
      case StateFlagsPropertyId:
      case ColorsPropertyId:
         ClearColorPropertyAccessor();
         break;

      default:
         break;
   }
}


/*****************************************************************************/
#define __COLOR_SWITCH_CHECK_FLAG(pFlag) \
   if ((GetStateFlags() & enStateFlags::pFlag) != 0) \
   { \
      if (state.is##pFlag()) \
      { \
         index += (1u << pos); \
      } \
      ++pos; \
   }


/*****************************************************************************/
const Candera::Color& ColorSwitchWidget2D::GetStateColor(const AppearanceState& state) const
{
   size_t index = 0;
   size_t pos = 0;

   __COLOR_SWITCH_CHECK_FLAG(Enabled)
   __COLOR_SWITCH_CHECK_FLAG(Pressed)
   __COLOR_SWITCH_CHECK_FLAG(Active)
   __COLOR_SWITCH_CHECK_FLAG(Focused)

   if (index >= GetColors().GetCount())
   {
      static Candera::Color _invalidColor(1.0F, 1.0F, 1.0F, 1.0F);
      return _invalidColor;
   }

   return GetColors().Get(index);
}


/*****************************************************************************/
#define __COLOR_SWITCH_FIND_STATE_SOURCE(pEnum, pWidget) \
         case enIndexSource::pEnum: \
            ++_sourceSearchCount; \
            stateSourceWidget = WidgetFinder::FindAncestorWidget< pWidget >(this); \
            break;


/*****************************************************************************/
#define __COLOR_SWITCH_GET_STATE_COLOR(pState) GetStateColor(AppearanceState(AppearanceState::en##pState))


/*****************************************************************************/
void ColorSwitchWidget2D::InitializeColorAppearance()
{
   if ((GetNode() != NULL) && (_colorPropertyAccessor != NULL) && (_sourceSearchCount < MAX_SEARCH_COUNT))
   {
      BaseWidget2D* stateSourceWidget = NULL;
      switch (GetIndexSource())
      {
            __COLOR_SWITCH_FIND_STATE_SOURCE(ButtonWidgetState, ButtonWidget2D)
            __COLOR_SWITCH_FIND_STATE_SOURCE(DirectTextureConsumerWidgetState, DirectTextureConsumer2D)
            __COLOR_SWITCH_FIND_STATE_SOURCE(GestureWidgetState, hmibase::widget::gesture::GestureWidget2D)
            __COLOR_SWITCH_FIND_STATE_SOURCE(GizmoWidgetState, hmibase::widget::gizmo::GizmoWidget2D)
            __COLOR_SWITCH_FIND_STATE_SOURCE(ListWidgetState, ListWidget2D)
            __COLOR_SWITCH_FIND_STATE_SOURCE(ScrollBarWidgetState, ScrollBarWidget2D)
            __COLOR_SWITCH_FIND_STATE_SOURCE(SliderWidgetState, SliderWidget2D)

         case enIndexSource::IndexProperty:
         default:
            break;
      }

      if (stateSourceWidget != NULL)
      {
         AppearanceColorData::SharedPointer colorData = AppearanceColorData::create(
                  __COLOR_SWITCH_GET_STATE_COLOR(Normal),
                  __COLOR_SWITCH_GET_STATE_COLOR(Pressed),
                  __COLOR_SWITCH_GET_STATE_COLOR(Disabled),
                  __COLOR_SWITCH_GET_STATE_COLOR(PressedDisabled),
                  __COLOR_SWITCH_GET_STATE_COLOR(ActiveNormal),
                  __COLOR_SWITCH_GET_STATE_COLOR(ActivePressed),
                  __COLOR_SWITCH_GET_STATE_COLOR(ActiveDisabled),
                  __COLOR_SWITCH_GET_STATE_COLOR(ActivePressedDisabled),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedNormal),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedPressed),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedDisabled),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedPressedDisabled),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedActiveNormal),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedActivePressed),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedActiveDisabled),
                  __COLOR_SWITCH_GET_STATE_COLOR(FocusedActivePressedDisabled));

         Appearance* app = FEATSTD_NEW(ColorAccessorAppearance)(*stateSourceWidget, *_colorPropertyAccessor, colorData, NULL);

         SetAppearance(app);
      }
      else if (_sourceSearchCount == MAX_SEARCH_COUNT)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "InitializeColorAppearance() State source widget of type [%u] not found as ancestor for %s!",
                            GetIndexSource(), HMIBASE_TO_STRING_VW(this)));
      }
      else
      {
         //nothing to do
      }
   }
}


/*****************************************************************************/
void ColorSwitchWidget2D::Update()
{
   if ((GetNode() != NULL) && (_colorPropertyAccessor == NULL) && (_targetSearchCount < MAX_SEARCH_COUNT))
   {
      ++_targetSearchCount;
      _colorPropertyAccessor = ColorPropertyHelper::Create(*this, *GetNode(), GetTarget());

      if ((_colorPropertyAccessor == NULL) && (_targetSearchCount == MAX_SEARCH_COUNT))
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Update() Color property of type [%u] not found for %s!",
                            GetTarget(), HMIBASE_TO_STRING_VW(this)));
      }
   }

   //initialize appearance before calling Base::Update
   if ((GetIndexSource() != enIndexSource::IndexProperty) && (GetAppearance() == NULL))
   {
      InitializeColorAppearance();
   }

   //base update also updates the appearance
   Base::Update();

   //update color based on index
   if ((GetIndexSource() == enIndexSource::IndexProperty) && (_colorPropertyAccessor != NULL))
   {
      Candera::Color color;
      if (GetIndex() < GetColors().GetCount())
      {
         color = GetColors().Get(GetIndex());
      }
      if (_colorPropertyAccessor->Get() != color)
      {
         _colorPropertyAccessor->Set(color);
         Invalidate();
      }
   }
}


}
}


}
