/* ***************************************************************************************
* FILE:          ClockWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Label Widget as part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
///
/// ClockWiget is used to display an analogue clock with a clock face as background image
/// and two or three clock hand images for hours, minute and second.The clock is skin-able,
/// meaning the complete set of images(clock face and hands) can be changed during runtime
///
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "ClockWidget2D.h"
#include "CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_CLOCK
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ClockWidget2D.cpp.trc.h"
#endif

CGI_WIDGET_RTTI_DEFINITION(ClockWidget2D);


/****************************************************************************
*     Constructor
****************************************************************************/
ClockWidget2D::ClockWidget2D() : _nodeIsInvalid(true)
{
   memset(_rnode, 0, sizeof(_rnode));
}


/****************************************************************************
*     Destructor
****************************************************************************/
ClockWidget2D::~ClockWidget2D()
{
}


/****************************************************************************
*    Initializes the widget so that all referred resource can be
*    resolved
****************************************************************************/
void ClockWidget2D::InitWidget()
{
   Base::InitWidget();
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ClockWidget2D [%s]: Initialized", GetLegacyName()));
}


/****************************************************************************
*    overrides WidgetBase::Update, the painter
****************************************************************************/
void ClockWidget2D::Update()
{
   Base::Update();

   bool invalidate = false;

   if (_nodeIsInvalid)
   {
      unsigned int ii = 0;
      memset(_rnode, 0, sizeof(_rnode));
      setRenderNodes(GetNode(), ii);
      setImagePath();
      _nodeIsInvalid = false;
      _clockFace.clear();
      _hourHand.clear();
      _minHand.clear();
      _secHand.clear();
      invalidate = true;
   }

   if (_clockFace.isInvalid())
   {
      _clockFace.update(GetAssetProvider(), 24, _rnode[0], _imagePath, GetClockFaceImageName(), true);
      invalidate = true;
   }

   if (_hourHand.isInvalid())
   {
      _hourHand.update(GetAssetProvider(), 24, _rnode[1], _imagePath, GetHourImageName());
      invalidate = true;
   }

   if (_minHand.isInvalid())
   {
      _minHand.update(GetAssetProvider(), 60, _rnode[2], _imagePath, GetMinuteImageName());
      invalidate = true;
   }

   if (_secHand.isInvalid())
   {
      _secHand.update(GetAssetProvider(), 60, _rnode[3], _imagePath, GetSecondImageName());
      invalidate = true;
   }

   if (invalidate)
   {
      Invalidate();
   }
}


/****************************************************************************
*    Overrides ClockWidget2DBase::OnChanged
****************************************************************************/
void ClockWidget2D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case SkinIndexPropertyId:
      case ImagePathPropertyId:
         _nodeIsInvalid = true;
         break;

      case ClockFaceImageNamePropertyId:
         _clockFace._needsBmpUpload = true;
         break;

      case HourImageNamePropertyId:
         _hourHand._needsBmpUpload = true;
         break;

      case MinuteImageNamePropertyId:
         _minHand._needsBmpUpload = true;
         break;

      case SecondImageNamePropertyId:
         _secHand._needsBmpUpload = true;
         break;

      case HourPropertyId:
      case MinutePropertyId:
      {
         Candera::UInt8 v = GetMinute() % 60;
         if (_minHand._value != v)
         {
            _minHand._index = v;
            _minHand._value = v;
            _minHand._angle = (Candera::Float)(6 * GetMinute());
            _minHand._timeIsInvalid = true;
         }

         _hourHand._value = GetHour() % 12;
         FeatStd::UInt8 idx = static_cast<FeatStd::UInt8>((((5 * _hourHand._value) + _minHand._value / 12) % 60));

         _hourHand._angle = (Candera::Float)(6 * idx);
         //Total clock angle = 360 degree
         //360 / 60 = 6 degree change for every division.
         //idx is the current division of the hour hand.

         if (idx != _hourHand._index)
         {
            _hourHand._index = idx;
            _hourHand._timeIsInvalid = true;
         }
         break;
      }

      case SecondPropertyId:
      {
         Candera::UInt8 v = GetSecond() % 60;
         if (_secHand._value != v)
         {
            _secHand._value = v;
            _secHand._index = v;
            _secHand._angle = (Candera::Float)(6 * GetSecond());
            _secHand._timeIsInvalid = true;
         }
         break;
      }
      default:
         break;
   }
}


/****************************************************************************
*    Collect the first 4 Rendernodes, take care that the designer could
*     design the SceneTree with diff Group helpers for layouts
****************************************************************************/
void ClockWidget2D::setRenderNodes(Candera::Node2D* node, unsigned int& i)
{
   while (node != 0)
   {
      Candera::Node2D* node2 = node->GetFirstChild();
      if (node2)
      {
         setRenderNodes(node2, i);
      }
      Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(node);
      if (renderNode && i < TABSIZE(_rnode))
      {
         _rnode[i++] = renderNode;
      }
      node = node->GetNextSibling();
   }
}


/****************************************************************************
*    Set the candera path of the skin[0] or skin[1] ...
****************************************************************************/
void ClockWidget2D::setImagePath()
{
   _imagePath = GetImagePath().GetCString();

   bool useSkins = strchr(_imagePath.c_str(), '%') != NULL;

   if (useSkins)
   {
      char tempPath[200];
      SNPRINTF(tempPath, sizeof(tempPath), _imagePath.c_str(), GetSkinIndex());
      _imagePath = tempPath;
   }
}


/****************************************************************************
*    Uploads the image of one hand and perhaps rotate the bar
****************************************************************************/
void ClockWidget2D::ClockItemData::update(Candera::AssetProvider* assetProvider, unsigned /*base*/, Candera::RenderNode* node,
      const std::string& imgPath, const Candera::String& imgName, bool isFace)
{
   if (node != 0 && !imgName.IsEmpty() && !imgPath.empty())
   {
      bool useRotate = strchr(imgName.GetCString(), '%') == NULL;

      if (useRotate || isFace)
      {
         // version, where one single bitmap for each position is used
         char tempPath[200];
         SNPRINTF(tempPath, sizeof(tempPath), "%s#%s", imgPath.c_str(), imgName.GetCString());
         _canderaBmpName = tempPath;
      }
      else
      {
         // version, where for each position a single bitmap is used
         char tempName[100];
         SNPRINTF(tempName, sizeof(tempName), imgName.GetCString(), _index);
         char tempPath[200];
         SNPRINTF(tempPath, sizeof(tempPath), "%s#%s", imgPath.c_str(), tempName);
         if (_canderaBmpName != tempPath)
         {
            _canderaBmpName = tempPath;
            _needsBmpUpload = true;
         }
      }
      if (_needsBmpUpload)
      {
         if (setImage(node, assetProvider) == true)
         {
            _needsBmpUpload = false;
         }
         _timeIsInvalid = true;
      }

      if (_timeIsInvalid)
      {
         if (useRotate && !isFace)
         {
            // all single hands should be painted in direction from top left (center node) to
            // bottom right ( 45° ), means 135° from 12:00 o'clock
            // commented the below lines solve hour hand issue, to avoid the LINT Warning
            //const float corrFaktor = 135.0f;
            //float valRotation = (((360.0f) / base) * _value) + 360.0f - corrFaktor;
            node->SetRotation(_angle);
         }
         _timeIsInvalid = false;
      }
   }
   else
   {
      _timeIsInvalid = false;
      _needsBmpUpload = false;
   }
}


/****************************************************************************
*    changes the attached to a BitmapBrush by unloading the previously
*     attached image and uploading the new one.
****************************************************************************/
bool ClockWidget2D::ClockItemData::setImage(const Candera::RenderNode* pRenderNode, Candera::AssetProvider* assetProvider)
{
   if (pRenderNode != 0 && assetProvider != 0)
   {
      Candera::Effect2D* effect = pRenderNode->GetEffect(0);
      if (effect != 0)
      {
         Candera::BitmapBrush* bitmapBrush = Candera::Dynamic_Cast<Candera::BitmapBrush*>(effect->GetBrushEffect2D());
         if (bitmapBrush != 0)
         {
            Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D> newBitmap = assetProvider->GetBitmapImage2DById(
                     Candera::Internal::AssetProviderFunctions::GetIdByName(assetProvider, Candera::BitmapLib, _canderaBmpName.c_str()));

            if (!newBitmap.PointsToNull())
            {
               // Candera::Image2D* oldImage = bitmapBrush->Image().Get();
               bitmapBrush->Unload();
               bitmapBrush->Image().Set(newBitmap);
               bitmapBrush->Upload();
               return true;
            }
            ETG_TRACE_ERR(("ClockWidget:: GetBitmapImage2DById(imageId) failed Name='%s'", _canderaBmpName.c_str()));
         }
      }
   }
   return false;
}


bool ClockWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ClockWidget2D* original = CLONEABLE_WIDGET_CAST<const ClockWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }
      SetSkinIndex(original->GetSkinIndex());
      SetHour(original->GetHour());
      SetMinute(original->GetMinute());
      SetSecond(original->GetSecond());

      SetImagePath(original->GetImagePath());
      SetClockFaceImageName(original->GetClockFaceImageName());
      SetHourImageName(original->GetHourImageName());

      SetMinuteImageName(original->GetMinuteImageName());
      SetSecondImageName(original->GetSecondImageName());
      cloned = true;
   }
   return cloned;
}
