/* ***************************************************************************************
* FILE:          ButtonGroupController2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonGroupController2D.h is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(ButtonGroupController2D_h)
#define ButtonGroupController2D_h

#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Widgets/2D/Adorner/Adorner.h>
#include <View/CGI/Widget/WidgetController.h>

class ButtonGroupWidget2D;
class ButtonWidget2D;


/*****************************************************************************/
/* Request which can be sent by another controller to change the active button.
* The Action parameter determines how the new active button is determined:
* - enSelectIndex - new active button is indicated by the index parameter;
* - enSelectIdentifier - new active button is indicated by the identifier parameter;
* - enSelectNextOfIdentifier - new active button is the button which follows the one indicated by the identifier parameter;
* - enSelectPreviousOfIdentifier - new active button is the button which preceeds the one indicated by the identifier parameter;
*/
/*****************************************************************************/
class ButtonGroupSelectReqEvent : public hmibase::widget::WidgetControllerEvent
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetControllerEvent);

   public:
      CANDERA_RTTI_DECLARATION(ButtonGroupSelectReqEvent);

      ButtonGroupSelectReqEvent(enButtonGroupAction action, Courier::Identifier identifier = Courier::Identifier(), int index = 0, bool animate = false)
         : _action(action), _identifier(identifier), _index(index), _animate(animate)
      {
      }

      enButtonGroupAction GetAction() const
      {
         return _action;
      }

      Courier::Identifier GetButtonIdentifier() const
      {
         return _identifier;
      }

      int GetButtonIndex() const
      {
         return _index;
      }

      bool GetAnimate() const
      {
         return _animate;
      }

   private:
      enButtonGroupAction _action;
      Courier::Identifier _identifier;
      int _index;
      bool _animate;
};


/*****************************************************************************/
/* Request sent by the button group controller to the new active button in order to trigger the transition animation.
*/
/*****************************************************************************/
class ButtonGroupAnimationReqEvent : public hmibase::widget::WidgetControllerEvent
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetControllerEvent);

   public:
      CANDERA_RTTI_DECLARATION(ButtonGroupAnimationReqEvent);

      ButtonGroupAnimationReqEvent(ButtonGroupWidget2D* buttonGroup, ButtonWidget2D* startWidget, ButtonWidget2D* endWidget, unsigned int animationDuration = 0)
         : _buttonGroup(buttonGroup), _startWidget(startWidget), _endWidget(endWidget), _animationDuration(animationDuration)
      {
      }

      ~ButtonGroupAnimationReqEvent()
      {
         _buttonGroup = NULL;
         _startWidget = NULL;
         _endWidget = NULL;
      }

      ButtonGroupWidget2D* getButtonGroup() const
      {
         return _buttonGroup;
      }

      ButtonWidget2D* getStartWidget() const
      {
         return _startWidget;
      }

      ButtonWidget2D* getEndWidget() const
      {
         return _endWidget;
      }

      unsigned int getAnimationDuration() const
      {
         return _animationDuration;
      }

   private:
      ButtonGroupWidget2D* _buttonGroup;
      ButtonWidget2D* _startWidget;
      ButtonWidget2D* _endWidget;
      unsigned int _animationDuration;
};


/*****************************************************************************/
class ButtonGroupController2DData : public hmibase::widget::WidgetControllerData
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetControllerData);

   public:
      CANDERA_RTTI_DECLARATION(ButtonGroupController2DData);

      ButtonGroupController2DData() {}
};


/*****************************************************************************/
/* Abstract class used for button group controllers.
*/
/*****************************************************************************/
class ButtonGroupController2D : public hmibase::widget::WidgetController
{
      FEATSTD_TYPEDEF_BASE(hmibase::widget::WidgetController);

   public:
      CANDERA_RTTI_DECLARATION(ButtonGroupController2D);

      ButtonGroupController2D() {}

      virtual bool OnMessage(DelegateWidget& widget, const Courier::Message& msg) override;
      virtual bool OnEvent(DelegateWidget& widget, const hmibase::widget::WidgetControllerEvent& e) override;

      virtual bool findButtons(ButtonGroupWidget2D& buttonGroup, std::vector<ButtonWidget2D*>& buttons) = 0;

   protected:
      virtual bool OnButtonGroupReqMsg(DelegateWidget& widget, const ButtonGroupReqMsg& msg) = 0;
      virtual bool OnButtonGroupSelectReqEvent(ButtonGroupWidget2D& widget, const ButtonGroupSelectReqEvent& e) = 0;
      virtual void sendAnimationReqEvent(ButtonGroupWidget2D& buttonGroup, ButtonWidget2D* oldButton, ButtonWidget2D* newButton) = 0;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ButtonGroupController2D);
};


/*****************************************************************************/
class DefaultButtonGroupData
{
   public:
      FEATSTD_TYPEDEF_SHARED_POINTER(DefaultButtonGroupData);
      DefaultButtonGroupData() : ActiveIndex(-1) {}

      hmibase::widget::adorner::Adorner::SharedPointer AdornerHandle;

      int ActiveIndex;
      std::string ActiveName;

   private:
      FEATSTD_SHARED_POINTER_DECLARATION();
      FEATSTD_MAKE_CLASS_UNCOPYABLE(DefaultButtonGroupData);
};


/*****************************************************************************/
class DefaultButtonGroupController2DData : public ButtonGroupController2DData
{
      FEATSTD_TYPEDEF_BASE(ButtonGroupController2DData);

   public:
      CANDERA_RTTI_DECLARATION(DefaultButtonGroupController2DData);

      DefaultButtonGroupController2DData()
      {
         Group = DefaultButtonGroupData::SharedPointer(CANDERA_NEW(DefaultButtonGroupData));
      }

      DefaultButtonGroupData::SharedPointer Group;
};


/*****************************************************************************/
class DefaultButtonGroupController2D : public ButtonGroupController2D
{
      FEATSTD_TYPEDEF_BASE(ButtonGroupController2D);

   public:
      CANDERA_RTTI_DECLARATION(DefaultButtonGroupController2D);

      DefaultButtonGroupController2D();

      /* Singleton instance. */
      static DefaultButtonGroupController2D& getInstance()
      {
         static DefaultButtonGroupController2D instance;
         return instance;
      }

      /* If set to true then all the buttons will be used even the hidden ones otherwise only the visible buttons will be used. */
      void SetUseAllButtons(bool value);
      bool GetUseAllButtons() const
      {
         return _useAllButtons;
      }

      /* If set to a value greater then 0 then it will be used as render order rank for the node associated with the active button; the nodes of all inactive buttons will get 0 as render order rank. */
      void SetActiveButtonRenderOrderRank(int value);
      int GetActiveButtonRenderOrderRank() const
      {
         return _activeButtonRenderOrderRank;
      }

      //overrides from WidgetController
      virtual hmibase::widget::WidgetControllerData* CreateData(DelegateWidget&)  override
      {
         return CANDERA_NEW(DefaultButtonGroupController2DData);
      }
      virtual void Update(DelegateWidget& widget) override;

      virtual bool findButtons(ButtonGroupWidget2D& buttonGroup, std::vector<ButtonWidget2D*>& buttons);

   protected:
      virtual bool OnButtonGroupReqMsg(DelegateWidget& widget, const ButtonGroupReqMsg& msg) override;
      virtual bool OnButtonGroupSelectReqEvent(ButtonGroupWidget2D& buttonGroup, const ButtonGroupSelectReqEvent& e) override;

      virtual void updateButtons(ButtonGroupWidget2D& buttonGroup, std::vector<ButtonWidget2D*>& buttons, ButtonWidget2D* newButton, bool animate);
      virtual void updateButton(ButtonGroupWidget2D& buttonGroup, ButtonWidget2D& button, bool active);
      virtual void sendAnimationReqEvent(ButtonGroupWidget2D& buttonGroup, ButtonWidget2D* oldButton, ButtonWidget2D* newButton) override;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(DefaultButtonGroupController2D);
      bool _useAllButtons;
      int _activeButtonRenderOrderRank;
};


#endif//ButtonGroupController2D
