/* ***************************************************************************************
* FILE:          ButtonTouchHandler.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonTouchHandler is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(ButtonTouchHandler_h)
#define ButtonTouchHandler_h

#include <hmibase/util/ItemRegistry.h>

namespace Candera {
class Camera2D;
class Vector2;
}


class ButtonWidget2D;
typedef ButtonWidget2D BaseButtonWidget2D;

/*
 * By default the sensitive area of a button is the bounding rectangle of the associated node.
 * A touch handler can provide a custom sensitive area for buttons (for example circle, triangle, etc).
 * Touch handlers are registered at application start-up. The ids used during registration can be used to configure the button widgets in scene composer.

 * How to use the default touch handlers (more can be created by taking the defaults as model):

 * - add some touch handler id constants in the hmi contract
 <constants>
 <const name="TOUCH_HANDLER_ID_ELLIPSE" type="int32" uid="" value="1"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_UP" type="int32" uid="" value="11"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_UPPER_RIGHT" type="int32" uid="" value="12"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_RIGHT" type="int32" uid="" value="13"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_LOWER_RIGHT" type="int32" uid="" value="14"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_DOWN" type="int32" uid="" value="15"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_LOWER_LEFT" type="int32" uid="" value="16"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_LEFT" type="int32" uid="" value="17"/>
 <const name="TOUCH_HANDLER_ID_TRIANGLE_UPPER_LEFT" type="int32" uid="" value="18"/>
 </constants>

 * - add a button touch handler databinding source the hmi contract
 <bindingSources>
 <bindingSource name="ButtonTouchHandlers" readonly="true" access="asynchronous" uid="">
 <item name="Ellipse" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_ELLIPSE"/>
 <item name="TriangleUp" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_UP"/>
 <item name="TriangleUpperRight" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_UPPER_RIGHT"/>
 <item name="TriangleRight" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_RIGHT"/>
 <item name="TriangleLowerRight" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_LOWER_RIGHT"/>
 <item name="TriangleDown" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_DOWN"/>
 <item name="TriangleLowerLeft" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_LOWER_LEFT"/>
 <item name="TriangleLeft" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_LEFT"/>
 <item name="TriangleUpperLeft" type="Candera::Int32" readonly="true" uid="" defaultValue="TOUCH_HANDLER_ID_TRIANGLE_UPPER_LEFT"/>
 </bindingSource>
 </bindingSources>

 * - instantiate the databinding source in the DataModel and send update on Startup message
 DataBindingItem<ButtonTouchHandlersDataBindingSource>  _ButtonTouchHandlersData;
 _ButtonTouchHandlersData.SendUpdate(true);

 * - register the touch handler at application startup
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_ELLIPSE, &EllipseButtonTouchHandler::getInstance());
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_UP, &TriangleButtonTouchHandler::getInstance(hmibase::Up));
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_UPPER_RIGHT, &TriangleButtonTouchHandler::getInstance(hmibase::UpperRight));
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_RIGHT, &TriangleButtonTouchHandler::getInstance(hmibase::Right));
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_LOWER_RIGHT, &TriangleButtonTouchHandler::getInstance(hmibase::LowerRight));
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_DOWN, &TriangleButtonTouchHandler::getInstance(hmibase::Down));
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_LOWER_LEFT, &TriangleButtonTouchHandler::getInstance(hmibase::LowerLeft));
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_LEFT, &TriangleButtonTouchHandler::getInstance(hmibase::Left));
 ButtonTouchHandlerRegistry::registerItem(TOUCH_HANDLER_ID_TRIANGLE_UPPER_LEFT, &TriangleButtonTouchHandler::getInstance(hmibase::UpperLeft));

 * - in scene composer bind the TouchHandler property of the button widget to an item in the button touch handlers databinding source
 TouchHandler@/ButtonTouchHandlers/TriangleUp

 */
class ButtonTouchHandler
{
   public:
      virtual ~ButtonTouchHandler() {}

      /* Checks if the point is contained in the button widget considering the specified camera. */
      virtual bool contains(ButtonWidget2D& widget, const Candera::Camera2D& camera, const Candera::Vector2& point) = 0;

      typedef unsigned int IdType;
      _FEATSTD_DEPRECATED("Use ButtonTouchHandlerRegistry::registerItem() instead.", static bool registerHandler(IdType id, ButtonTouchHandler* handler));
      _FEATSTD_DEPRECATED("Use ButtonTouchHandlerRegistry::unregisterItem() instead.", static bool unregisterHandler(IdType id));
      _FEATSTD_DEPRECATED("Use ButtonTouchHandlerRegistry::getItem() instead.", static ButtonTouchHandler* getHandler(IdType id));
};


typedef hmibase::util::ItemPtrRegistry<unsigned int, ButtonTouchHandler> ButtonTouchHandlerRegistry;


/* Convenient base class for touch handlers which performs some calculations similar to the existing code in the button widget. */
class BaseButtonTouchHandler : public ButtonTouchHandler
{
   public:
      BaseButtonTouchHandler() {}
      virtual ~BaseButtonTouchHandler() {}

      virtual bool contains(ButtonWidget2D& widget, const Candera::Camera2D& camera, const Candera::Vector2& touchedPoint);

   private:
      BaseButtonTouchHandler(const BaseButtonTouchHandler&);
      BaseButtonTouchHandler& operator=(const BaseButtonTouchHandler&);

      /* Checks if the point is contained in the specified rectangle. This rectangle can be considered as the bounds of an ellipse or triangle. */
      virtual bool containsImpl(const Candera::Rectangle& bounds, const Candera::Vector2& point) = 0;
};


/* Default implementation of an ellipse touch handler. It is considered that the ellipse is enclosed in the bounding rectangle of the associated node. */
class EllipseButtonTouchHandler : public BaseButtonTouchHandler
{
   public:
      static EllipseButtonTouchHandler& getInstance();

      /* Checks if point is included in the ellipse which has the rectangle as outer bounds. */
      static bool ellipseContains(const Candera::Rectangle& ellipseBounds, const Candera::Vector2& point);

   private:
      EllipseButtonTouchHandler() {}

      virtual bool containsImpl(const Candera::Rectangle& ellipseBounds, const Candera::Vector2& point);
};


/* Default implementation of a triangle touch handler. It is considered that the triangle is enclosed in the bounding rectangle of the associated node. */
class TriangleButtonTouchHandler : public BaseButtonTouchHandler
{
   public:
      TriangleButtonTouchHandler(hmibase::FocusDirectionEnum orientation) : _orientation(orientation) {}

      static TriangleButtonTouchHandler& getInstance(hmibase::FocusDirectionEnum orientation);

      /* Checks if point p is included in the triangle defined by p0,p1,p2. */
      static bool triangleContains(const Candera::Vector2& p0, const Candera::Vector2& p1, const Candera::Vector2& p2, const Candera::Vector2& p);

   private:
      virtual bool containsImpl(const Candera::Rectangle& bounds, const Candera::Vector2& point);

      hmibase::FocusDirectionEnum _orientation;
};


#endif // ButtonTouchHandler_h
