/* ***************************************************************************************
* FILE:          ButtonTouchHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonTouchHandler is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ButtonWidget2D.h"
#include "ButtonTouchHandler.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BUTTON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonTouchHandler.cpp.trc.h"
#endif


/****************************************************************************************/
bool ButtonTouchHandler::registerHandler(IdType id, ButtonTouchHandler* handler)
{
   return ButtonTouchHandlerRegistry::registerItem(id, handler);
}


/****************************************************************************************/
bool ButtonTouchHandler::unregisterHandler(IdType id)
{
   return ButtonTouchHandlerRegistry::unregisterItem(id);
}


/****************************************************************************************/
ButtonTouchHandler* ButtonTouchHandler::getHandler(IdType id)
{
   return ButtonTouchHandlerRegistry::getItem(id);
}


/****************************************************************************************/
bool BaseButtonTouchHandler::contains(ButtonWidget2D& widget, const Candera::Camera2D& camera, const Candera::Vector2& touchedPoint)
{
   bool nodeTouched = false;
   Candera::Node2D* node = widget.GetNode();
   if (node != NULL)
   {
      Candera::Vector2 pointInWorldSpace = Candera::Math2D::TransformViewportToScene(camera, Candera::Math2D::TransformRenderTargetToViewport(camera, touchedPoint));

      Candera::Matrix3x2 nodeMatrix(node->GetWorldTransform());
      nodeMatrix.Inverse();

      Candera::Vector2 pointInNodeSpace = nodeMatrix.Multiply(pointInWorldSpace);

      const Candera::Rectangle& touchableArea = widget.GetTouchableArea();
      if ((touchableArea.GetHeight() > 0.0f) && (touchableArea.GetHeight() > 0.0f))
      {
         nodeTouched = containsImpl(touchableArea, pointInNodeSpace);
      }
      else
      {
         Candera::Rectangle boundingRect;
         node->GetEffectiveBoundingRectangle(boundingRect);
         nodeTouched = containsImpl(boundingRect, pointInNodeSpace);
      }
   }
   return nodeTouched;
}


/****************************************************************************************/
#define TriangleButtonTouchHandlerInstance(direction)\
case direction:\
{\
   static TriangleButtonTouchHandler _instance(direction);\
   return _instance;\
}


/****************************************************************************************/
TriangleButtonTouchHandler& TriangleButtonTouchHandler::getInstance(hmibase::FocusDirectionEnum orientation)
{
   switch (orientation)
   {
      default:
         TriangleButtonTouchHandlerInstance(hmibase::Up)
         TriangleButtonTouchHandlerInstance(hmibase::UpperRight)
         TriangleButtonTouchHandlerInstance(hmibase::Right)
         TriangleButtonTouchHandlerInstance(hmibase::LowerRight)
         TriangleButtonTouchHandlerInstance(hmibase::Down)
         TriangleButtonTouchHandlerInstance(hmibase::LowerLeft)
         TriangleButtonTouchHandlerInstance(hmibase::Left)
         TriangleButtonTouchHandlerInstance(hmibase::UpperLeft)
   }
}


/****************************************************************************************/
bool TriangleButtonTouchHandler::containsImpl(const Candera::Rectangle& bounds, const Candera::Vector2& point)
{
   Candera::Vector2 p0;
   Candera::Vector2 p1;
   Candera::Vector2 p2;

   float x = bounds.GetLeft();
   float y = bounds.GetTop();
   float w = bounds.GetWidth();
   float h = bounds.GetHeight();

   switch (_orientation)
   {
      case hmibase::Up:
         p0.SetX(x);
         p0.SetY(y + h);
         p1.SetX(x + w / 2);
         p1.SetY(y);
         p2.SetX(x + w);
         p2.SetY(y + h);
         break;

      case hmibase::Down:
         p0.SetX(x);
         p0.SetY(y);

         p1.SetX(x + w / 2);
         p1.SetY(y + h);

         p2.SetX(x + w);
         p2.SetY(y);
         break;

      case hmibase::Left:
         p0.SetX(x);
         p0.SetY(y + h / 2);

         p1.SetX(x + w);
         p1.SetY(y);

         p2.SetX(x + w);
         p2.SetY(y + h);
         break;

      case hmibase::Right:
         p0.SetX(x);
         p0.SetY(y);

         p1.SetX(x + w);
         p1.SetY(y + h / 2);

         p2.SetX(x);
         p2.SetY(y + h);
         break;

      case hmibase::UpperLeft:
         p0.SetX(x);
         p0.SetY(y);
         p1.SetX(x + w);
         p1.SetY(y);
         p2.SetX(x);
         p2.SetY(y + h);
         break;

      case hmibase::UpperRight:
         p0.SetX(x);
         p0.SetY(y);
         p1.SetX(x + w);
         p1.SetY(y);
         p2.SetX(x + w);
         p2.SetY(y + h);
         break;

      case hmibase::LowerLeft:
         p0.SetX(x);
         p0.SetY(y);
         p1.SetX(x + w);
         p1.SetY(y + h);
         p2.SetX(x);
         p2.SetY(y + h);
         break;

      case hmibase::LowerRight:
         p0.SetX(x + w);
         p0.SetY(y);
         p1.SetX(x + w);
         p1.SetY(y + h);
         p2.SetX(x);
         p2.SetY(y + h);
         break;

      default:
         break;
   }

   return triangleContains(p0, p1, p2, point);
}


/****************************************************************************************/
bool TriangleButtonTouchHandler::triangleContains(const Candera::Vector2& p0, const Candera::Vector2& p1, const Candera::Vector2& p2, const Candera::Vector2& p)
{
   float p0X = p0.GetX();
   float p0Y = p0.GetY();
   float p1X = p1.GetX();
   float p1Y = p1.GetY();
   float p2X = p2.GetX();
   float p2Y = p2.GetY();
   float pX = p.GetX();
   float pY = p.GetY();

   bool contains = false;
   float s = p0Y * p2X - p0X * p2Y + (p2Y - p0Y) * pX + (p0X - p2X) * pY;                                               //lint !e834
   float t = p0X * p1Y - p0Y * p1X + (p0Y - p1Y) * pX + (p1X - p0X) * pY;                                               //lint !e834

   if ((s < 0.0f) == (t < 0.0f))
   {
      float A = -p1Y * p2X + p0Y * (p2X - p1X) + p0X * (p1Y - p2Y) + p1X * p2Y;
      if (A < 0.0f)
      {
         s = -s;
         t = -t;
         A = -A;
      }
      contains = (s > 0.0f) && (t > 0.0f) && ((s + t) <= A);
   }
   return contains;
}


/****************************************************************************************/
EllipseButtonTouchHandler& EllipseButtonTouchHandler::getInstance()
{
   static EllipseButtonTouchHandler _instance;
   return _instance;
}


/****************************************************************************************/
bool EllipseButtonTouchHandler::ellipseContains(const Candera::Rectangle& ellipseBounds, const Candera::Vector2& point)
{
   bool contains = false;
   float radiusX = ellipseBounds.GetWidth() / 2.0f;
   float radiusY = ellipseBounds.GetHeight() / 2.0f;

   if ((radiusX > 0.0f) && (radiusY > 0.0f))
   {
      float centerX = ellipseBounds.GetLeft() + radiusX;
      float centerY = ellipseBounds.GetTop() + radiusY;

      float normalizedX = point.GetX() - centerX;
      float normalizedY = point.GetY() - centerY;

      // X^2/a^2 + Y^2/b^2 <= 1
      contains = ((normalizedX * normalizedX) / (radiusX * radiusX)
                  + (normalizedY * normalizedY) / (radiusY * radiusY) <= 1);
   }
   return contains;
}


/****************************************************************************************/
bool EllipseButtonTouchHandler::containsImpl(const Candera::Rectangle& ellipseBounds, const Candera::Vector2& point)
{
   return ellipseContains(ellipseBounds, point);
}
