/* ***************************************************************************************
* FILE:          ButtonSlideController2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonSlideController2D.h is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(ButtonSlideController2D_h)
#define ButtonSlideController2D_h

#include <Candera/EngineBase/Animation/AnimationPlayerListener.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Widgets/2D/Button/ButtonWidget2D.h>
#include <Widgets/2D/ButtonGroup/ButtonGroupController2D.h>

#include <bitset>


/*****************************************************************************/
class ButtonSlideController2DData : public ButtonController2DData
{
      FEATSTD_TYPEDEF_BASE(ButtonController2DData);

   public:
      CANDERA_RTTI_DECLARATION(ButtonSlideController2DData);

      ButtonSlideController2DData() : IsGroupSearched(false) {}

      bool IsGroupSearched;
      DefaultButtonGroupData::SharedPointer GroupData;
      Courier::Identifier GroupId;

      Courier::Identifier DestinationId;

      /* Due to a bug in the gesture framework when multiple gestures are enabled for a widget, the Start of a new gesture is received before End/Abort of the old gesture.
      * For this reason we need to keep track what gestures are active in order to consider the gizmo as pressed. */
      std::bitset<static_cast<size_t>(hmibase::input::gesture::GestureEvent::GT_None)> ActiveGestures;
};


/*****************************************************************************/
/* Provides support to slide option buttons inside a button group using drag gestures.
* Attach this controller to buttons linked to nodes being descendant of a group which is attached to a button group widget.
* During the drag operation this controller will post ButtonSlideMsg message to inform the application about the user action.
* The application must control the button group using data binding for its ActiveIndex property or using ButtonGroupReqMsg.
* See the the ButtonGroup section in the WidgetUserGuide for more information.
*/
/*****************************************************************************/
class ButtonSlideController2D : public ButtonController2D, public hmibase::widget::adorner::AdornerAnimation::AnimationListener
{
      FEATSTD_TYPEDEF_BASE(ButtonController2D);

   public:
      CANDERA_RTTI_DECLARATION(ButtonSlideController2D);

      ButtonSlideController2D(hmibase::widget::adorner::AdornerManager& adornerManager, const hmibase::widget::adorner::AdornerMarkerFilter& adornerMarkerFilter = hmibase::widget::adorner::AdornerMarkerFilter());
      virtual ~ButtonSlideController2D();

      //overrides from WidgetController
      virtual hmibase::widget::WidgetControllerData* CreateData(DelegateWidget&) override
      {
         return CANDERA_NEW(ButtonSlideController2DData);
      }
      virtual bool OnEvent(DelegateWidget& widget, const hmibase::widget::WidgetControllerEvent& e) override;
      virtual bool OnMessage(DelegateWidget& widget, const Courier::Message& msg) override;
      virtual void CheckGestureConfig(const DelegateWidget& widget, hmibase::widget::WidgetGestureConfig& config) override;
      virtual bool OnGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData) override;
      virtual bool OnDragGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData) override;

      //overrides from ButtonController2D
      virtual AppearanceState getAppearanceState(ButtonWidget2D& button) override;

      //overrides from AdornerAnimation::AnimationListener
      virtual void onAnimationEnded(hmibase::widget::adorner::AdornerAnimation* animation) override;

      void setAdornerHideTimerTimeout(unsigned int timeout);

   protected:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ButtonSlideController2D);

      struct AdornerDragStartInfo
      {
         Candera::Vector2 touchCoordinates;
         Candera::Vector2 adornerCoordinates;
      };

      virtual bool onTimerExpiredMessage(DelegateWidget& widget, const TimerExpiredMsg& msg);

      virtual bool onButtonGroupAnimationReqEvent(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const ButtonGroupAnimationReqEvent& e);

      virtual void onGestureBeforeStart(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureAfterStart(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureBeforeEnd(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureAfterEnd(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureBeforeAbort(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual void onGestureAfterAbort(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);

      virtual bool onDragGestureStart(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onDragGestureMove(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onDragGestureEnd(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);
      virtual bool onDragGestureAbort(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, const hmibase::input::gesture::GestureEvent& gestureData);

      virtual bool moveAdornerOnDragMove(ButtonGroupWidget2D& group, ButtonWidget2D& button, ButtonSlideController2DData& buttonData,
                                         hmibase::widget::adorner::Adorner::SharedPointer adorner, const hmibase::input::gesture::GestureEvent& gestureData);

      virtual ButtonGroupWidget2D* findGroup(ButtonWidget2D& button, ButtonSlideController2DData& buttonData);
      virtual void updateButtonData(ButtonWidget2D& button, ButtonSlideController2DData& buttonData);

      void startAnimation(ButtonWidget2D& startWidget, ButtonWidget2D& endWidget, unsigned int animationDuration);

      virtual hmibase::widget::adorner::Adorner::SharedPointer createAdorner(ButtonWidget2D& button, ButtonSlideController2DData& buttonData);
      virtual void showAdorner(ButtonWidget2D& button, ButtonSlideController2DData& buttonData);
      virtual void hideAdorner(ButtonWidget2D& button, ButtonSlideController2DData& buttonData);

      virtual void startAdornerHideTimer();
      virtual void stopAdornerHideTimer();
      virtual void onAdornerHideTimerExpired(DelegateWidget& widget);
      virtual void scheduleAdornerHide(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, bool startTimer = true);
      virtual void hideOldAdorner();

      virtual bool postMessage(ButtonWidget2D& button, ButtonSlideController2DData& buttonData, hmibase::widget::gesture::enGestureEvent::Enum evt);

      hmibase::widget::adorner::AdornerManager& _adornerManager;
      hmibase::widget::adorner::AdornerMarkerFilter _adornerMarkerFilter;
      hmibase::widget::adorner::AdornerAnimation* _adornerAnimation;

      DefaultButtonGroupData::SharedPointer _adornerGroupToHide;
      Util::Timer* _adornerHideTimer;
      unsigned int _adornerHideTimerTimeout;
};


#endif
