/* ***************************************************************************************
* FILE:          ButtonExtensionWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ButtonExtensionWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ButtonWidget2D.h"
#include "ButtonAppearance2D.h"
#include "ButtonExtensionWidget2D.h"
#include "Widgets/2D/Text/TextWidget2D.h"
#include "Widgets/2D/WidgetFinder2D.h"

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_APPEARANCE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ButtonExtensionWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ButtonExtensionWidget2D);
CGI_WIDGET_RTTI_DEFINITION(ButtonTextExtensionWidget2D);

ButtonExtension::ButtonExtension() :
   _invalid(true),
   _ancestorSearchDone(0),
   _buttonWidget(NULL),
   _buttonState()
{
}


ButtonExtension::~ButtonExtension()
{
   _buttonWidget = NULL;
}


void ButtonExtension::SetInvalid(bool invalid)
{
   _invalid = invalid;
}


bool ButtonExtension::IsInvalid() const
{
   return _invalid;
}


ButtonWidget2D* ButtonExtension::GetButtonWidget() const
{
   return _buttonWidget;
}


const hmibase::widget::appearance::AppearanceState& ButtonExtension::GetButtonState() const
{
   return _buttonState;
}


void ButtonExtension::OnNodeChanged(Candera::Node2D* /*node*/)
{
   _buttonWidget = NULL;
   _invalid = true;
   _ancestorSearchDone = 0;
}


void ButtonExtension::UpdateButtonWidget(BaseWidget2D& widget)
{
   if ((_buttonWidget == NULL) && (widget.GetNode() != NULL))
   {
      _buttonWidget = WidgetFinder::FindAncestorWidget<ButtonWidget2D>(&widget);
      if (_buttonWidget != NULL)
      {
         _invalid = true;
      }
      else if (++_ancestorSearchDone == 2)
      {
         ETG_TRACE_ERR(("UpdateButtonWidget Widget %s does not have a ButtonWidget2D as ancestor!", HMIBASE_TO_STRING_VW(&widget)));
      }
   }
}


void ButtonExtension::UpdateButtonState()
{
   //check if the cached state is different from the actual state of the button
   if (_buttonWidget != NULL)
   {
      hmibase::widget::appearance::AppearanceState state = _buttonWidget->GetAppearanceState();
      if (state != _buttonState)
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UpdateButtonState oldState=0x%x newState=0x%x button=%50s %s",
                             _buttonState.asEnum(), state.asEnum(),
                             HMIBASE_TO_STRING_W(_buttonWidget),
                             HMIBASE_TO_STRING_VW(dynamic_cast<hmibase::widget::Widget*>(this))));

         _buttonState = state;
         _invalid = true;
      }
   }
}


void ButtonExtension::Update(BaseWidget2D& widget)
{
   UpdateButtonWidget(widget);
   UpdateButtonState();
}


ButtonExtensionWidget2D::ButtonExtensionWidget2D() : Base(), ButtonExtension()
{
}


ButtonExtensionWidget2D::~ButtonExtensionWidget2D()
{
}


void ButtonExtensionWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();
   ButtonExtension::OnNodeChanged(GetNode());
}


void ButtonExtensionWidget2D::Update()
{
   Base::Update();
   ButtonExtension::Update(*this);
}


bool ButtonExtensionWidget2D::PrepareForCloning()
{
   //if there are databinding configured for this widget we assume that they will change the bitmap or the color properties for various button states so we can not have an appearance template
   if ((*GetBindingsIterator()) != NULL)
   {
      //widget must be cloned
      return true;
   }

   ////try to register the appearance template for this extension widget
   ////it that is successful we don't this this widget to be cloned anymore
   //if ((GetButtonWidget() != NULL) && hmibase::widget::appearance::ButtonAppearanceHelper::registerExtensionAppearanceTemplate(*GetButtonWidget(), *this))
   //{
   //   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "PrepareForCloning appearance template registered for buttonExtension=%p button=[%p.%50s.%s]", this,
   //                       GetButtonWidget(), GetButtonWidget()->GetParentView() != NULL ? GetButtonWidget()->GetParentView()->GetId().CStr() : "<null>", GetButtonWidget()->GetLegacyName()));

   //   //widget should not be cloned because an appearance temple was registered for it and will do the same changes on the node as this widget
   //   return false;
   //}

   //widget must be cloned
   return true;
}


ButtonTextExtensionWidget2D::ButtonTextExtensionWidget2D() : Base(),
   _textWidget(NULL),
   _ancestorTextWidgetSearchDone(0)
{
}


ButtonTextExtensionWidget2D::~ButtonTextExtensionWidget2D()
{
   _textWidget = NULL;
}


TextWidget2D* ButtonTextExtensionWidget2D::GetTextWidget() const
{
   return _textWidget;
}


void ButtonTextExtensionWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();

   _textWidget = NULL;
   _ancestorTextWidgetSearchDone = 0;
}


void ButtonTextExtensionWidget2D::UpdateTextWidget()
{
   if ((_textWidget == NULL) && (GetNode() != NULL))
   {
      _textWidget = WidgetFinder::FindAncestorWidget<TextWidget2D>(this);
      if (_textWidget != NULL)
      {
         SetInvalid(true);
      }
      else if (++_ancestorTextWidgetSearchDone == 2)
      {
         ETG_TRACE_ERR(("UpdateTextWidget Widget %s does not have a TextWidget2D as ancestor!", HMIBASE_TO_STRING_VW(this)));
      }
   }
}


void ButtonTextExtensionWidget2D::Update()
{
   Base::Update();

   UpdateTextWidget();
}
