/* ***************************************************************************************
* FILE:          BlurWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BlurWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"

#include "BlurWidget2D.h"
#include "Widgets/utils/Blur/BlurUtilsCommon.h"
#include "Widgets/utils/Blur/BlurCameraSetter.h"
#include <View/CGI/CgiExtensions/SCHostUtils.h>

#ifdef COURIER_ENHANCED_ENABLED
#include "BlurViewScene2D.h"
#include <Courier/Visualization/ViewHandler.h>
#endif  // ifdef COURIER_ENHANCED_ENABLED
#include <View/CGI/CgiExtensions/Renderer.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BLUR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BlurWidget2D.cpp.trc.h"
#endif  // ifdef VARIANT_S_FTR_ENABLE_TRC_GEN

#include <Widgets/2D/ControlTemplate/ControlTemplateCloneableWidget.h>
#include <CanderaPlatform/Device/Genivi/GeniviFrameBufferObject.h>

using namespace Candera;

namespace hmibase {
namespace widget {
namespace blur {


CGI_WIDGET_RTTI_DEFINITION(BlurWidget2D)

const Candera::Int32 BlurWidget2D::m_minRadius = 1;

OriginalCamera2DVectorType BlurWidget2D::s_originalCameraVector = OriginalCamera2DVectorType();

BlurWidget2D::BlurWidget2D() :
   m_originalNodeRenderingEnabled(false),
   m_bitmapMaskRectangle(0.0F, 0.0F, 0.0F, 0.0F),
   m_bitmapMaskTexture(Candera::BitmapTextureImage::Create()),
   m_outputRenderModeBlending(Candera::RenderMode::Create()),
   m_outputRenderModeOverwrite(Candera::RenderMode::Create()),
   m_oldNodeMask(0),
   m_validateBlurParameters(true),
   m_validateShaderType(true),
   m_skipSetup(true),
   m_isRunningInSceneComposer(false),
   m_isOriginalCameraModifiedBySceneComposer(false),
   m_blurEnabledInternal(false),
   m_blurCameraMap(),
   m_maskTextureUnit(1),
   m_sourceTextureUnit(2)
#ifdef COURIER_ENHANCED_ENABLED
   , m_viewSceneActive(false),
   m_viewSceneRenderingEnabled(false),
   m_updateAfterInactive(false)
#endif
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Ctor Blur Widget: [%s]", GetLegacyName()));
   m_darkenUniform[0] = 1.0F;
   m_darkenUniform[1] = 1.0F;
   m_darkenUniform[2] = 1.0F;
   m_darkenUniform[3] = 1.0F;

   m_noDarkenUniform[0] = 1.0F;
   m_noDarkenUniform[1] = 1.0F;
   m_noDarkenUniform[2] = 1.0F;
   m_noDarkenUniform[3] = 1.0F;

   if (m_outputRenderModeBlending.PointsToNull() == false)
   {
      // Inherit everything
      m_outputRenderModeBlending->SetInheritanceBitsToOne(0xFFFFFFFF);

      // Except this
      m_outputRenderModeBlending->SetBlendingEnabled(true);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::BlendingEnabledBit);

      m_outputRenderModeBlending->SetDepthTestEnabled(false);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::DepthTestEnabledBit);

      m_outputRenderModeBlending->SetDepthWriteEnabled(false);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::DepthWriteEnabledBit);

      m_outputRenderModeBlending->SetBlendModeSeparate(Candera::RenderMode::SourceAlpha, Candera::RenderMode::InverseSourceAlpha, Candera::RenderMode::Add,
            Candera::RenderMode::One, Candera::RenderMode::InverseSourceAlpha, Candera::RenderMode::Add);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::BlendModeBit);
   }

   if (m_outputRenderModeOverwrite.PointsToNull() == false)
   {
      // Inherit everything
      m_outputRenderModeOverwrite->SetInheritanceBitsToOne(0xFFFFFFFF);

      // Except this
      m_outputRenderModeOverwrite->SetBlendingEnabled(true);
      m_outputRenderModeOverwrite->SetInheritanceBitsToZero(RenderMode::BlendingEnabledBit);

      m_outputRenderModeOverwrite->SetDepthTestEnabled(false);
      m_outputRenderModeOverwrite->SetInheritanceBitsToZero(RenderMode::DepthTestEnabledBit);

      m_outputRenderModeOverwrite->SetDepthWriteEnabled(false);
      m_outputRenderModeOverwrite->SetInheritanceBitsToZero(RenderMode::DepthWriteEnabledBit);
   }
}


BlurWidget2D::~BlurWidget2D()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Dtor Blur Widget: [%s]", GetLegacyName()));
   Finalize();
}


void BlurWidget2D::Finalize()
{
   if (m_blurEnabledInternal == true)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Finalize Blur Widget: [%s]", GetLegacyName()));
      ActivateContext();
      CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap>(0));
      SetupCourier(false);
      SetupBlurScene(false);
      SetupRenderTarget(false);
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Skipping Finalize (disabled internal) Blur Widget: [%s]", GetLegacyName()));
   }
}


bool BlurWidget2D::IsRunningInSceneComposer() const
{
   return hmibase::view::util::SCHostUtils::IsSCHostLoaded();
}


bool BlurWidget2D::CheckForOriginalCameraModifications()
{
   bool isModified = false;

   if (m_isRunningInSceneComposer)
   {
      BlurCamera2DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
      while (itBlurCameraMap != m_blurCameraMap.end())
      {
         FeatStd::UInt32 cameraCount = Candera::Renderer2D::GetCameraCount();
         for (FeatStd::UInt32 cameraIndex = 0; cameraIndex < cameraCount; cameraIndex++)
         {
            OriginalCamera2D* camera = Candera::Renderer2D::GetCamera(cameraIndex);
            OriginalCamera2D* cameraWidget = itBlurCameraMap->first;
            if ((camera == cameraWidget) &&
                  ((camera->GetRenderTarget() != itBlurCameraMap->second.passOneGdu->ToRenderTarget2D()) ||
                   (camera->GetRenderTarget() == itBlurCameraMap->second.passTwoGdu->ToRenderTarget2D())))
            {
               isModified = true;
            }
         }
         ++itBlurCameraMap;
      }

      if (isModified)
      {
         m_isOriginalCameraModifiedBySceneComposer = true;
         CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap>(0));
         SetupCourier(false);
         SetupBlurScene(false);
         SetupRenderTarget(false);
         InvalidateBlurWidget();
         m_isOriginalCameraModifiedBySceneComposer = false;
      }
   }
   return isModified;
}


Candera::RenderTarget3D* BlurWidget2D::RetrieveOwnerRenderTarget(Candera::GraphicDeviceUnit* gdu) const
{
   Candera::RenderTarget3D* result = 0;
   if (gdu != 0)
   {
      Candera::Int displayId = gdu->GetDisplay();
      Candera::Int count = 0;

      // GDU is display -> Is owner
      if ((gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::DisplayTarget2D, displayId, count))) ||
            (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::DisplayTarget3D, displayId, count))) ||
            (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::Mixed2D3DDisplayTarget, displayId, count))))
      {
         result = gdu->ToRenderTarget3D();
      } // GDU itself is offscreen render target -> Has owner -> But should not be set, as position of offscreen pixel has no relevance to touch input
      else if ((gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget2D, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget3D, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::Mixed2D3DOffscreenTarget, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget2Dto3D, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget3Dto2D, displayId, count))))
      {
         // Do NOT set owner for offscreen RT, as it interferes with touch input position
         //result = 0;

         //fet2hi:without an owner rendering into on offscreen render target will get corrupted
         Candera::GeniviFrameBufferObject* fbo = dynamic_cast<Candera::GeniviFrameBufferObject*>(gdu);
         result = fbo != NULL ? fbo->GetProperties().GetOwner() : NULL;
      } // GDU is of unknown type
      else
      {
         result = 0;
      }
   }

   return result;
}


bool BlurWidget2D::IsCameraPartOfBlur(Candera::Camera2D* camera)
{
   if ((camera != 0) && (m_blurCameraMap.count(camera) > 0))
   {
      return true;
   }

   return false;
}


bool BlurWidget2D::IsRenderTargetPartOfBlur(Candera::RenderTarget* renderTarget)
{
   BlurCamera2DMapType::iterator renderSetupIterator = m_blurCameraMap.begin();
   BlurCamera2DMapType::iterator renderSetupIteratorEnd = m_blurCameraMap.end();

   while (renderSetupIterator != renderSetupIteratorEnd)
   {
      if (renderTarget == renderSetupIterator->second.originalRenderTarget)
      {
         return true;
      }
      ++renderSetupIterator;
   }

   return false;
}


void BlurWidget2D::Update()
{
   if (GetBlurEnabled() == true)
   {
      // During Scene Composer update, the updated scene is first created and then the old scene deleted.
      // To first clean up the blurred scene before setting up the new blur, the setup is delayed by one frame
      // in Scene Composer.
      m_isRunningInSceneComposer = IsRunningInSceneComposer();
   }
#ifdef COURIER_ENHANCED_ENABLED
   bool blurEnabled = ((GetBlurEnabled()) && (m_viewSceneActive) && (m_viewSceneRenderingEnabled));

   if (m_isRunningInSceneComposer == true)
   {
      blurEnabled = (GetBlurEnabled());
   }
#else
   bool blurEnabled = GetBlurEnabled();
#endif
   if (blurEnabled && m_blurEnabledInternal)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Update BlurWidget: Blur running: [%s]", GetLegacyName()));
      ActivateContext();
      if (m_isRunningInSceneComposer && CheckForOriginalCameraModifications())
      {
         ETG_TRACE_ERR(("BlurWidget2D: Original camera has been modified in background: [%s]", GetLegacyName()));
      }
      else
      {
         UpdateBlurBitmapTransform();
         ValidateBlurParameters();
      }
   }
   else if (blurEnabled && (!m_blurEnabledInternal))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Update BlurWidget: Blur being enabled: [%s]", GetLegacyName()));
      ActivateContext();
      if (m_skipSetup && m_isRunningInSceneComposer)
      {
         m_skipSetup = false;
         UpdateBlurBitmapTransform();
         InvalidateBlurWidget();
      }
      else
      {
         SetupRenderTarget(true);
         CreateTextureFromBitmap(GetBitmapMask());
         UpdateBlurBitmapTransform();
         SetupBlurScene(true);

#ifdef COURIER_ENHANCED_ENABLED
         SetupCourier(true);
#endif  //#ifdef COURIER_ENHANCED_ENABLED
         InvalidateBlurWidget();
      }
   }
   else if ((!blurEnabled) && m_blurEnabledInternal)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Update BlurWidget: Blur being disabled: [%s]", GetLegacyName()));
      ActivateContext();
      CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap>(0));
#ifdef COURIER_ENHANCED_ENABLED
      SetupCourier(false);
#endif  //#ifdef COURIER_ENHANCED_ENABLED
      SetupBlurScene(false);
      SetupRenderTarget(false);
      InvalidateBlurWidget();
   }
   else
   {
      /* do nothing */
   }
}


void BlurWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);//calls triggerUpdate

   switch (propertyId)
   {
      case BlurEnabledPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: BlurEnabled Changed: [%s]", GetLegacyName()));
         break;
      case DarkenPropertyId:
         OnDarkenPropertyChanged();
         break;

      case BlurRadiusPropertyId:
      case BlurRadiusToSigmaPropertyId:
         m_validateBlurParameters = true;
         break;

      case BlurMaskVariantPropertyId:
         OnBlurMaskVariantPropertyChanged();
         break;

      case RenderNodeMaskPropertyId:
         OnRenderNodeMaskPropertyChanged();
         break;

      case BlurShaderVariantPropertyId:
         OnBlurShaderVariantPropertyChanged();
         break;
      case BitmapMaskPropertyId:
         OnBitmapMaskPropertyChanged();
         break;
      case BlendOutputPropertyId:
         OnBlendOutputPropertyChanged();
         break;
      case MultipassPropertyId:
         OnMultipassPropertyChanged();
         break;
   }
}


void BlurWidget2D::SetupCameraNode(OriginalNode2D* node, bool blurEnabled)
{
   if (m_isRunningInSceneComposer && (IsSceneComposerCamera(node)))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Scene Composer Blur Preview not active!: [%s]", GetLegacyName()));
   }
   else
   {
      SetupRenderTargetForCamera(node, blurEnabled);
   }
}


void BlurWidget2D::SetupRenderTarget(bool blurEnabled)
{
   /* Only enter function, if we actually need to change something */
   if (blurEnabled != m_blurEnabledInternal)
   {
      OriginalNode2D* node = GetNode();
      if (NULL != node)
      {
         if (node->GetDynamicTypeId() == OriginalCamera2D::GetTypeId())
         {
            SetupCameraNode(node, blurEnabled);
         }
         else if (node->GetDynamicTypeId() == OriginalScene2D::GetTypeId())
         {
            OriginalNode2D* child = node->GetFirstChild();
            while (child != 0)
            {
               if (child->GetDynamicTypeId() == OriginalCamera2D::GetTypeId())
               {
                  SetupCameraNode(child, blurEnabled);
               }
               child = child->GetNextSibling();
            }
         }
         else
         {
            ETG_TRACE_ERR(("BlurWidget2D: Found no scene or camera that could be blurred!: [%s]", GetLegacyName()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget2D: There is no Node assigned to the Blur Widget: [%s]", GetLegacyName()));
      }
   }
}


void BlurWidget2D::SetupRenderTargetForCamera(OriginalNode2D* child, bool blurEnabled)
{
   OriginalCamera2D* camera = Dynamic_Cast<OriginalCamera2D*>(child);

   if ((camera != 0) && blurEnabled) /* load blur render targets */
   {
      if (IsCameraAlreadyBlurred(camera))
      {
         ETG_TRACE_ERR(("BlurWidget2D: Try to blur camera multiple times: [%s]", GetLegacyName()));
         return;
      }

      OriginalRenderTarget2D* originalRT = camera->GetRenderTarget();

      if (originalRT != 0)
      {
         if (m_blurCameraMap.count(camera) == 0) /* Camera not already known */
         {
            /* RT not known -> create it */
            Candera::GraphicDeviceUnit* passOneGdu = CreateOffscreenRendertarget(camera);
            Candera::GraphicDeviceUnit* passTwoGdu = NULL;

            if (GetMultipass())
            {
               passTwoGdu = CreateOffscreenRendertarget(camera);
            }

            if (passOneGdu != 0)
            {
               m_blurEnabledInternal = true;

               RenderTargetSetup2D renderTargetSetup;
               renderTargetSetup.passOneGdu = passOneGdu;
               renderTargetSetup.passTwoGdu = passTwoGdu;
               renderTargetSetup.passOneShader = NULL;
               renderTargetSetup.passTwoShader = NULL;
               renderTargetSetup.passOneScene = NULL;
               renderTargetSetup.passTwoScene = NULL;
               renderTargetSetup.originalRenderTarget = originalRT;

               SetOriginalCamera(camera, passOneGdu->ToRenderTarget2D(), renderTargetSetup);

               std::pair<OriginalCamera2D*, RenderTargetSetup2D> pairCameraRenderTargetSetup(camera, renderTargetSetup);
               std::pair<BlurCamera2DMapType::iterator, bool> result = m_blurCameraMap.insert(pairCameraRenderTargetSetup);
               s_originalCameraVector.push_back(camera);

               if (!result.second)
               {
                  ETG_TRACE_ERR(("BlurWidget2D: Insert of the original render target into blurring GDU map failed: [%s]", GetLegacyName()));
               }
            }
         }
      }
   }
   else if ((camera != 0) && (!blurEnabled))
   {
      BlurCamera2DMapType::iterator iterator = m_blurCameraMap.begin();
      while (iterator != m_blurCameraMap.end())
      {
         ResetOriginalCamera(iterator->first);

         GraphicDeviceUnit* firstPass = iterator->second.passOneGdu;
         GraphicDeviceUnit* secondPass = iterator->second.passTwoGdu;

         if (firstPass != 0)
         {
            Candera::Internal::GraphicDeviceUnitOwnerAccess::SetGraphicDeviceUnitOwner(firstPass, 0);
            firstPass->Unload();
            DevicePackageInterface::DestroyGraphicDeviceUnit(firstPass);
         }

         if (secondPass != 0)
         {
            Candera::Internal::GraphicDeviceUnitOwnerAccess::SetGraphicDeviceUnitOwner(secondPass, 0);
            secondPass->Unload();
            DevicePackageInterface::DestroyGraphicDeviceUnit(secondPass);
         }

         //iterator = m_blurCameraMap.erase(iterator);//not compilable on the target
         m_blurCameraMap.erase(iterator);
         iterator = m_blurCameraMap.begin();

         //Remove camera from global blur list
         OriginalCamera2DVectorType::iterator iteratorCam = s_originalCameraVector.begin();
         while (iteratorCam != s_originalCameraVector.end())
         {
            if (*iteratorCam == camera)
            {
               iteratorCam = s_originalCameraVector.erase(iteratorCam);
            }
            else
            {
               ++iteratorCam;
            }
         }
      }

      if (m_blurCameraMap.empty())
      {
         m_blurCameraMap.clear();
         m_blurEnabledInternal = false;
      }
   }
   else
   {
      //do nothing
   }
}


bool BlurWidget2D::IsCameraAlreadyBlurred(OriginalCamera2D const* camera) const
{
   OriginalCamera2DVectorType::iterator iterator = s_originalCameraVector.begin();
   while (iterator != s_originalCameraVector.end())
   {
      if (*iterator == camera)
      {
         return true;
      }
      ++iterator;
   }
   return false;
}


Candera::GraphicDeviceUnit* BlurWidget2D::CreateOffscreenRendertarget(OriginalCamera2D const* camera)
{
   /* Create a similar render target to the one given */
   GraphicDeviceUnit* blurringGdu = 0;
   const OriginalRenderTarget2D* renderTarget = camera->GetRenderTarget();

   if (renderTarget != 0)
   {
      GraphicDeviceUnit* originalGDU = renderTarget->GetGraphicDeviceUnit();

      Int count = 0;

      if (originalGDU != 0)
      {
         const GraphicDeviceUnitTypeHandle* handle = DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::Mixed2D3DOffscreenTarget, originalGDU->GetDisplay(), count);

         if ((handle != 0) && (count > 0))
         {
            blurringGdu = DevicePackageInterface::CreateGraphicDeviceUnit(handle[0]);

            if (blurringGdu != NULL)
            {
               if (IsRunningInSceneComposer() == false)
               {
                  Candera::RenderTarget3D* ownerRenderTarget = RetrieveOwnerRenderTarget(originalGDU);
                  if (ownerRenderTarget != 0)
                  {
                     Candera::Internal::GraphicDeviceUnitOwnerAccess::SetGraphicDeviceUnitOwner(blurringGdu, ownerRenderTarget);
                  }
               }

               Candera::Int newWidth = 0;
               Candera::Int newHeight = 0;
               GetBlurSceneSize(renderTarget, camera, newWidth, newHeight);

               FeatStd::Int subsamplingFactor = GetSubsamplingFactor();

               if (subsamplingFactor != 0)
               {
                  newWidth /= subsamplingFactor;
                  newHeight /= subsamplingFactor;
               }

               if (newWidth > 0)
               {
                  char metaInfoWidth[] = "Width\0";
                  SetMetaInfoInt(metaInfoWidth, newWidth, blurringGdu);
               }
               else
               {
                  ETG_TRACE_ERR(("BlurWidget2D: The width of the render target is not valid: [%s]", GetLegacyName()));
               }

               if (newHeight > 0)
               {
                  char metaInfoHeight[] = "Height\0";
                  SetMetaInfoInt(metaInfoHeight, newHeight, blurringGdu);
               }
               else
               {
                  ETG_TRACE_ERR(("BlurWidget2D: The height of the render target is not valid: [%s]", GetLegacyName()));
               }

               blurringGdu->SetDisplay(originalGDU->GetDisplay());

               if (!blurringGdu->Upload())
               {
                  ETG_TRACE_ERR(("BlurWidget2D: Upload of the blurring GDU failed: [%s]", GetLegacyName()));
               }
               else
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Blurring GDU successfully uploaded: [%s]", GetLegacyName()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("BlurWidget2D: Could create offscreen rendertarget: [%s]", GetLegacyName()));
            }
         }
         else
         {
            ETG_TRACE_ERR(("BlurWidget2D: Could not obtain handle to create offscreen rendertarget or"));
            ETG_TRACE_ERR(("The number of available graphic device units with this category on the specified display <= 0: [%s]", GetLegacyName()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget2D: Rendertarget of Original Camera cannot retrieve GDU: [%s]", GetLegacyName()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("BlurWidget2D: Rendertarget of Original Camera cannot be retrieved: [%s]", GetLegacyName()));
   }
   return blurringGdu;
}


void BlurWidget2D::UpdateBlurBitmapTransform()
{
   if (m_blurEnabledInternal)
   {
      Candera::Node2D* nodeMask = GetRenderNodeMask();
      FeatStd::Int subsamplingFactor = GetSubsamplingFactor();

      if ((nodeMask != 0) && (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask))
      {
         Candera::Rectangle rect;

         Candera::RenderNode* renderNode = 0;
         bool hasEffect = false;

         // If effect is applied, use bounding box settings of node
         if (nodeMask->GetDynamicTypeId() == Candera::RenderNode::GetTypeId())
         {
            renderNode = Dynamic_Cast<Candera::RenderNode*>(nodeMask);

            if ((renderNode) != 0 && (renderNode->GetEffect(0) != 0))
            {
               renderNode->GetComputedBoundingRectangle(rect);
               Vector2 scale = renderNode->GetScale();
               rect.SetHeight(rect.GetHeight() * scale.GetY());
               rect.SetWidth(rect.GetWidth() * scale.GetX());
               hasEffect = true;
            }
         }

         if (hasEffect == false) // If no effect is applied, use settings of the bitmap bounding box
         {
            FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap> bitmap = m_bitmapMaskTexture->GetBitmap();
            if (bitmap.PointsToNull() == false)
            {
               Vector2 scale = nodeMask->GetScale();
               rect.SetHeight(bitmap->GetHeight() * scale.GetY());
               rect.SetWidth(bitmap->GetWidth() * scale.GetX());
            }
         }

         Vector2 position = nodeMask->GetWorldPosition();
         rect.SetLeft(position.GetX());
         rect.SetTop(position.GetY());

         if (subsamplingFactor > 1)
         {
            rect.SetLeft(rect.GetLeft() / subsamplingFactor);
            rect.SetTop(rect.GetTop() / subsamplingFactor);
            rect.SetWidth(rect.GetWidth() / subsamplingFactor);
            rect.SetHeight(rect.GetHeight() / subsamplingFactor);
         }

         if (rect != m_bitmapMaskRectangle)
         {
            m_bitmapMaskRectangle = rect;
            m_validateBlurParameters = true;
         }
      }
      else
      {
         // no rendernodemask applied, or disabled
         Candera::Rectangle rect(0.0F, 0.0F, 0.0F, 0.0F);

         if (subsamplingFactor > 1)
         {
            rect.SetLeft(rect.GetLeft() / subsamplingFactor);
            rect.SetTop(rect.GetTop() / subsamplingFactor);
            rect.SetWidth(rect.GetWidth() / subsamplingFactor);
            rect.SetHeight(rect.GetHeight() / subsamplingFactor);
         }

         if (rect != m_bitmapMaskRectangle)
         {
            m_bitmapMaskRectangle = rect;
            m_validateBlurParameters = true;
         }
      }
   }
}


void BlurWidget2D::CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap> bitmap)
{
   if ((m_bitmapMaskTexture.PointsToNull() == false) &&
         (m_bitmapMaskTexture->IsUploaded() == true))
   {
      m_bitmapMaskTexture->Unload();
   }

   if (bitmap.PointsToNull() == false)
   {
      bool result = m_bitmapMaskTexture->SetBitmap(bitmap);
      if (result == true)
      {
         result = m_bitmapMaskTexture->Upload();

         if (result != true)
         {
            ETG_TRACE_ERR(("BlurWidget2D: Uploading blur mask texture failed: [%s]", GetLegacyName()));
         }
      }
   }
}


Candera::Scene* BlurWidget2D::SetupBlurPass(Candera::RenderTarget3D* dstRenderTarget, const Candera::RenderTarget3D* srcRenderTarget, Candera::ImageSource3D* mskRenderTarget,
      Candera::Int width, Candera::Int height, Candera::ImageSource3D* imageSource, Candera::String cameraName, Candera::Int sequenceNumber,
      IBlurShader* passShaderGenerator, bool darken)
{
   Scene* blurScene = Scene::Create();
   blurScene->SetName(cameraName.GetCString());

   Camera* blurCamera = Camera::Create();
   MemoryManagement::SharedPointer<OrthographicProjection> projection = OrthographicProjection::Create();

   projection->SetNearZ(0.001F);
   projection->SetFarZ(10.0F);

   Candera::Float projectionWidth = static_cast<Candera::Float>(width);
   projection->SetWidth(projectionWidth);
   Candera::Float projectionHeight = static_cast<Candera::Float>(height);
   projection->SetHeight(projectionHeight);

   Candera::ClearMode clearMode;
   clearMode.SetClearColor(Candera::Color(0.0F, 0.0F, 0.0F, 0.0F));
   clearMode.SetColorClearEnabled(true);
   clearMode.SetDepthClearEnabled(false);
   clearMode.SetClearDepth(1.0F);

   Candera::Rectangle rect(0.0F, 0.0F, 1.0F, 1.0F);
   blurCamera->SetProjection(projection);
   blurCamera->SetName(cameraName.GetCString());
   blurCamera->SetSequenceNumber(sequenceNumber);
   blurCamera->SetRenderTarget(dstRenderTarget);
   blurCamera->SetViewport(rect);
   blurCamera->SetClearMode(clearMode);
   blurCamera->SetSwapEnabled(true);
   blurCamera->SetRenderingEnabled(true);

   blurCamera->SetAppearance(Candera::Appearance::Create());
   FeatStd::MemoryManagement::SharedPointer<Candera::Appearance> cameraAppearance = blurCamera->GetAppearance();
   cameraAppearance->SetRenderMode(m_outputRenderModeOverwrite);

   Candera::Float billboardWidth = static_cast<Candera::Float>(width);
   Candera::Float billboardHeight = static_cast<Candera::Float>(height);
   Billboard* blurBillboard = Billboard::Create(billboardWidth, billboardHeight);
   Vector3 position(0.0F, 0.0F, -1.0F);
   blurBillboard->SetPosition(position);

   FeatStd::MemoryManagement::SharedPointer<MultiPassAppearance> appearance = MultiPassAppearance::Create();
   FeatStd::MemoryManagement::SharedPointer<Texture> proxyTex = Texture::Create();
   FeatStd::MemoryManagement::SharedPointer<ProxyTextureImage> proxyTexImg = ProxyTextureImage::Create(imageSource);

   proxyTex->SetWrapModeU(Texture::ClampToEdge);
   proxyTex->SetWrapModeV(Texture::ClampToEdge);
   proxyTex->SetTextureImage(proxyTexImg);

   if (!appearance->SetTexture(proxyTex, 0))
   {
      ETG_TRACE_ERR(("BlurWidget2D: Setting the blur source texture failed: [%s]", GetLegacyName()));
   }
   bool applyMask = false;

   if ((NULL != srcRenderTarget) && (NULL != mskRenderTarget))
   {
      Candera::ImageSource3D* mskImage = mskRenderTarget;
      FeatStd::MemoryManagement::SharedPointer<Texture> mskTex = Texture::Create();
      FeatStd::MemoryManagement::SharedPointer<ProxyTextureImage> mskTexImg = ProxyTextureImage::Create(mskImage);

      mskTex->SetWrapModeU(Texture::ClampToEdge);
      mskTex->SetWrapModeV(Texture::ClampToEdge);
      mskTex->SetTextureImage(mskTexImg);

      if (!appearance->SetTexture(mskTex, m_maskTextureUnit))
      {
         ETG_TRACE_ERR(("BlurWidget2D: Setting the mask texture unit failed: [%s]", GetLegacyName()));
      }
      Candera::ImageSource3D* srcImage = srcRenderTarget->GetGraphicDeviceUnit()->ToImageSource3D();
      FeatStd::MemoryManagement::SharedPointer<Texture> srcTex = Texture::Create();
      FeatStd::MemoryManagement::SharedPointer<ProxyTextureImage> srcTexImg = ProxyTextureImage::Create(srcImage);

      srcTex->SetWrapModeU(Texture::ClampToEdge);
      srcTex->SetWrapModeV(Texture::ClampToEdge);
      srcTex->SetTextureImage(srcTexImg);

      if (!appearance->SetTexture(srcTex, m_sourceTextureUnit))
      {
         ETG_TRACE_ERR(("BlurWidget2D: Setting the source texture unit failed: [%s]", GetLegacyName()));
      }
      applyMask = true;
   }

   if (!SetupBlurShader(passShaderGenerator, width, height, darken, applyMask))
   {
      ETG_TRACE_ERR(("BlurWidget2D: Setup of blur shader for render pass failed: [%s]", GetLegacyName()));
   }
   appearance->SetShader(passShaderGenerator->GetShader());
   appearance->SetShaderParamSetter(passShaderGenerator->GetShaderParams());

   blurBillboard->SetAppearance(appearance);
   bool result = blurScene->AddChild(blurCamera);

   if (result == false)
   {
      return NULL;
   }

   result = blurScene->AddChild(blurBillboard);

   if (result == false)
   {
      return NULL;
   }

   result = blurScene->UploadAll();

   if (result == false)
   {
      return NULL;
   }

   return blurScene;
}


void BlurWidget2D::GetBlurSceneSize(OriginalRenderTarget2D const* renderTarget, OriginalCamera2D const* camera, Candera::Int& width, Candera::Int& height)
{
   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("BlurWidget2D: Missing render target: [%s]", GetLegacyName()));
   }

   if (NULL == camera)
   {
      ETG_TRACE_ERR(("BlurWidget2D: Missing camera: [%s]", GetLegacyName()));
   }

   WindowRect viewport;
   GetViewport(camera, renderTarget, viewport);
   width = static_cast<Candera::Int>(viewport.GetRectInPixel().GetWidth());
   height = static_cast<Candera::Int>(viewport.GetRectInPixel().GetHeight());
}


void BlurWidget2D::SetupBlurScene(bool blurEnabled)
{
   if (blurEnabled && (!m_blurCameraMap.empty()))
   {
      BlurCamera2DMapType::iterator iterator = m_blurCameraMap.begin();
      BlurCamera2DMapType::iterator iteratorEnd = m_blurCameraMap.end();

      while (iterator != iteratorEnd)
      {
         if ((iterator->second.passOneScene == NULL) && (iterator->second.passTwoScene == NULL))
         {
            RenderTarget3D*         renderTargetFirstPass = 0;
            RenderTarget3D*         renderTargetSecondPass = 0;
            ImageSource3D*          imageSourceFirstPass = 0;
            ImageSource3D*          imageSourceSecondPass = 0;
            RenderTarget3D*         originalRenderTarget = iterator->second.originalRenderTarget->GetGraphicDeviceUnit()->ToRenderTarget3D();
            GraphicDeviceUnit*      gduFirstPass = iterator->second.passOneGdu;
            GraphicDeviceUnit*      gduSecondPass = iterator->second.passTwoGdu;

            if (gduFirstPass != 0)
            {
               renderTargetFirstPass = gduFirstPass->ToRenderTarget3D();
               imageSourceFirstPass = gduFirstPass->ToImageSource3D();

               if (GetMultipass() && (gduSecondPass != 0))
               {
                  renderTargetSecondPass = gduSecondPass->ToRenderTarget3D();
                  imageSourceSecondPass = gduSecondPass->ToImageSource3D();
               }
            }

            if ((0 != originalRenderTarget) && (0 != renderTargetFirstPass) && (0 != imageSourceFirstPass))
            {
               Candera::Int16 cameraSequenceNumber = static_cast<Candera::Int16>(iterator->first->GetSequenceNumber());

               // The original camera sequence number gets lowered by 2 (there is no difference regarding this between single and multipass
               // Thus, multipass number (worst case) is chosen
               iterator->first->SetSequenceNumber(cameraSequenceNumber - 2);

               iterator->second.passOneCameraName = GetBlurSceneName(iterator->first->GetName(), 1);

               CreateShaderType(&iterator->second.passOneShader, &iterator->second.passTwoShader, GetBlurShaderVariant(), GetMultipass());

               Candera::Int width = 0;
               Candera::Int height = 0;
               GetBlurSceneSize(iterator->second.originalRenderTarget, (*iterator).first, width, height);
               Candera::ImageSource3D* imagesourceMask = 0;

               if (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask)
               {
                  if (m_bitmapMaskTexture.PointsToNull() == false)
                  {
                     imagesourceMask = m_bitmapMaskTexture->ToImageSource3D();
                  }

                  if (imagesourceMask == 0)
                  {
                     ETG_TRACE_ERR(("BlurWidget2D: Bitmap Blur Mask could not be retrieved as ImageSource3D: [%s]", GetLegacyName()));
                  }
               }
               else if (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)
               {
                  if (GetRenderTargetMask() != 0)
                  {
                     if (GetRenderTargetMask()->GetGraphicDeviceUnit() != 0)
                     {
                        imagesourceMask = GetRenderTargetMask()->GetGraphicDeviceUnit()->ToImageSource3D();
                     }
                  }

                  if (imagesourceMask == 0)
                  {
                     ETG_TRACE_ERR(("BlurWidget2D: Rendertarget Blur Mask could not be retrieved as ImageSource3D: [%s]", GetLegacyName()));
                  }
               }
               else
               {
                  // Do nothing (no mask applied)
               }

               if (GetMultipass())
               {
                  // The first pass of a multipass process needs to render after the original camera, but before the second pass camera
                  iterator->second.passOneScene = SetupBlurPass(renderTargetSecondPass, NULL, NULL, width, height, imageSourceFirstPass, iterator->second.passOneCameraName, cameraSequenceNumber - 1, iterator->second.passOneShader, false);
               }
               else
               {
                  // In single pass, the blur pass receives the original camera sequence number, so the resulting draw order is the same as with the original camera
                  iterator->second.passOneScene = SetupBlurPass(originalRenderTarget, renderTargetFirstPass, imagesourceMask, width, height, imageSourceFirstPass, iterator->second.passOneCameraName, cameraSequenceNumber, iterator->second.passOneShader, true);
                  Candera::Camera* passOneCamera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(iterator->second.passOneScene, Candera::Camera::GetTypeId()));
                  if (NULL != passOneCamera)
                  {
                     SetFinalPassCamera(iterator->first, passOneCamera);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("BlurWidget2D: Final camera for single pass is missing (Null Pointer): [%s]", GetLegacyName()));
                  }
               }

               /* Scene setup second pass */
               if (GetMultipass() && (0 != renderTargetSecondPass) && (0 != imageSourceSecondPass))
               {
                  // In multipass rendering, the second pass is to be rendered at the same sequence number as the original camera (so the render order stays the same)
                  iterator->second.passTwoCameraName = GetBlurSceneName(iterator->first->GetName(), 2);

                  iterator->second.passTwoScene = SetupBlurPass(originalRenderTarget, renderTargetFirstPass, imagesourceMask, width, height, imageSourceSecondPass, iterator->second.passTwoCameraName, cameraSequenceNumber, iterator->second.passTwoShader, true);
                  Candera::Camera* passTwoCamera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(iterator->second.passTwoScene, Candera::Camera::GetTypeId()));
                  if (NULL != passTwoCamera)
                  {
                     SetFinalPassCamera(iterator->first, passTwoCamera);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("BlurWidget2D: Final camera for two pass is missing (Null Pointer): [%s]", GetLegacyName()));
                  }
               }
            }
         }
         ++iterator;
      }

      // Shader setup now correct
      m_validateBlurParameters = false;
      m_validateShaderType = false;
   }

   if (blurEnabled == false)
   {
      BlurCamera2DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
      BlurCamera2DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

      while (itBlurCameraMap != itBlurCameraMapEnd)
      {
         Scene* scene = itBlurCameraMap->second.passOneScene;
         if (NULL != scene)
         {
            if (!scene->UnloadAll())
            {
               ETG_TRACE_ERR(("BlurWidget2D: Unload All failed: [%s]", GetLegacyName()));
            }
            scene->Dispose();
         }
         itBlurCameraMap->second.passOneScene = NULL;

         scene = itBlurCameraMap->second.passTwoScene;
         if (NULL != scene)
         {
            if (!scene->UnloadAll())
            {
               ETG_TRACE_ERR(("BlurWidget2D: Unload All failed: [%s]", GetLegacyName()));
            }
            scene->Dispose();
         }
         itBlurCameraMap->second.passTwoScene = NULL;

         delete itBlurCameraMap->second.passOneShader;
         itBlurCameraMap->second.passOneShader = NULL;

         delete itBlurCameraMap->second.passTwoShader;
         itBlurCameraMap->second.passTwoShader = NULL;

         ++itBlurCameraMap;
      }
   }
}


bool BlurWidget2D::ValidateShaderType()
{
   bool result = true;

   BlurCamera2DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
   BlurCamera2DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

   while (itBlurCameraMap != itBlurCameraMapEnd)
   {
      CreateShaderType(&itBlurCameraMap->second.passOneShader, &itBlurCameraMap->second.passTwoShader, GetBlurShaderVariant(), GetMultipass());

      if ((NULL == itBlurCameraMap->second.passOneShader) || (GetMultipass() && (NULL == itBlurCameraMap->second.passTwoShader)))
      {
         ETG_TRACE_ERR(("BlurWidget2D: The validation of a shader type failed: [%s]", GetLegacyName()));
         result = false;
      }
      ++itBlurCameraMap;
   }
   return result;
}


bool BlurWidget2D::ValidateBlurShaders()
{
   bool result = true;

   BlurCamera2DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
   BlurCamera2DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

   while (itBlurCameraMap != itBlurCameraMapEnd)
   {
      /* Search for Pass One Billboard to apply the shader */
      Billboard* passOneBillboard = Dynamic_Cast<Billboard*>(GetFirstChildNodeOfType(itBlurCameraMap->second.passOneScene, Billboard::GetTypeId()));

      if (NULL == passOneBillboard)
      {
         ETG_TRACE_ERR(("BlurWidget2D: The billboard for the first pass blur scene is missing: [%s]", GetLegacyName()));
         return false;
      }
      Candera::Int width = static_cast<Candera::Int>(passOneBillboard->GetWidth());
      Candera::Int height = static_cast<Candera::Int>(passOneBillboard->GetHeight());

      bool applyMask = ((((GetRenderNodeMask() != 0) && (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)) ||
                         ((m_bitmapMaskTexture.PointsToNull() == false) && (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask))) &&
                        (GetMultipass() == false));

      if (!SetupBlurShader(itBlurCameraMap->second.passOneShader, width, height, !GetMultipass(), applyMask))
      {
         ETG_TRACE_ERR(("BlurWidget2D: Setup of blur shader for first pass failed: [%s]", GetLegacyName()));
         result = false;
      }
      SharedPointerShaderType passOneShader = itBlurCameraMap->second.passOneShader->GetShader();

      passOneBillboard->GetAppearance()->SetShader(passOneShader);
      passOneBillboard->GetAppearance()->SetShaderParamSetter(itBlurCameraMap->second.passOneShader->GetShaderParams());

      //Activate the camera to make sure the upload is done into the right rendering context.
      itBlurCameraMap->second.passOneGdu->ToRenderTarget3D()->Activate();
      if ((!passOneShader->IsUploaded()) && (!passOneShader->Upload()))
      {
         ETG_TRACE_ERR(("BlurWidget2D: Upload of the first pass blur shader failed: [%s]", GetLegacyName()));
         result = false;
      }

      /* Pass Two */
      if (GetMultipass())
      {
         /* Search for Pass Two Billboard to apply the shader */
         Billboard* passTwoBillboard = Dynamic_Cast<Billboard*>(GetFirstChildNodeOfType(itBlurCameraMap->second.passTwoScene, Billboard::GetTypeId()));

         if (NULL == passTwoBillboard)
         {
            ETG_TRACE_ERR(("BlurWidget2D: The billboard of the second pass blur scene is missing: [%s]", GetLegacyName()));
            return false;
         }
         Candera::Int width = static_cast<Candera::Int>(passTwoBillboard->GetWidth());
         Candera::Int height = static_cast<Candera::Int>(passTwoBillboard->GetHeight());

         bool applyMask = (((GetRenderNodeMask() != 0) && (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)) ||
                           ((m_bitmapMaskTexture.PointsToNull() == false) && (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask)));

         if (!SetupBlurShader(itBlurCameraMap->second.passTwoShader, width, height, true, applyMask))
         {
            ETG_TRACE_ERR(("BlurWidget2D: Setup of blur shader for second pass failed: [%s]", GetLegacyName()));
            result = false;
         }
         SharedPointerShaderType passTwoShader = itBlurCameraMap->second.passTwoShader->GetShader();

         passTwoBillboard->GetAppearance()->SetShader(passTwoShader);
         passTwoBillboard->GetAppearance()->SetShaderParamSetter(itBlurCameraMap->second.passTwoShader->GetShaderParams());
         //Activate the camera to make sure the upload is done into the right rendering context.
         itBlurCameraMap->second.passTwoGdu->ToRenderTarget3D()->Activate();

         if ((!passTwoShader->IsUploaded()) && (!passTwoShader->Upload()))
         {
            ETG_TRACE_ERR(("BlurWidget2D: Upload of the second pass blur shader failed: [%s]", GetLegacyName()));
            result = false;
         }
      }
      ++itBlurCameraMap;
   }
   return result;
}


void BlurWidget2D::ValidateBlurParameters()
{
   bool invalidateWidget = false;

   if (m_validateShaderType)
   {
      if (ValidateShaderType())
      {
         m_validateShaderType = false;
      }
      invalidateWidget = true;
   }

   if (m_validateBlurParameters)
   {
      if (ValidateBlurShaders())
      {
         m_validateBlurParameters = false;
      }
      invalidateWidget = true;
   }

   //To force redraw
   if (invalidateWidget)
   {
      InvalidateBlurWidget();
   }
}


bool BlurWidget2D::SetupBlurShader(IBlurShader* shaderGenerator, Candera::Int newWidth, Candera::Int newHeight, bool darken, bool mask)
{
   /* Configure the shader */
   Candera::Int32 blurRadius = GetBlurRadius() / GetSubsamplingFactor();
   if (blurRadius < m_minRadius)
   {
      blurRadius = m_minRadius;
   }

   Candera::Float sigma = static_cast<Candera::Float>(blurRadius) * GetBlurRadiusToSigma();

   shaderGenerator->SetRadius(blurRadius);
   shaderGenerator->SetSigma(sigma);
   shaderGenerator->SetTextureWidth(newWidth);
   shaderGenerator->SetTextureHeight(newHeight);

   bool maskEnabled = GetBlurMaskVariant() != enBlurMaskVariant::NoBlurMask;

   shaderGenerator->SetBlurMaskRectangle(m_bitmapMaskRectangle, GetKeepFrameBufferContent(), maskEnabled);

   if (darken)
   {
      shaderGenerator->SetDarken(m_darkenUniform);
   }
   else
   {
      shaderGenerator->SetDarken(m_noDarkenUniform);
   }

   if (mask)
   {
      shaderGenerator->SetMaskTextureUnit(m_maskTextureUnit, GetCombineMask());
      shaderGenerator->SetSourceTextureUnit(m_sourceTextureUnit);
   }
   else
   {
      shaderGenerator->SetMaskTextureUnit(-1, false);
      shaderGenerator->SetSourceTextureUnit(-1);
   }
   return true;
}


void BlurWidget2D::SetOriginalCamera(OriginalCamera2D* camera, OriginalRenderTarget2D* renderTarget, RenderTargetSetup2D& renderTargetSetup)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("BlurWidget2D: Camera is missing (Null Pointer): [%s]", GetLegacyName()));
   }

   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("BlurWidget2D: Render Target is missing (Null Pointer): [%s]", GetLegacyName()));
   }

   GetViewport(camera, renderTargetSetup.backupViewportRect);
   GetScissor(camera, renderTargetSetup.backupScissorRect);

   Candera::Int renderTargetWidth = 0;
   Candera::Int renderTargetHeight = 0;
   GetBlurSceneSize(renderTargetSetup.originalRenderTarget, camera, renderTargetWidth, renderTargetHeight);

   FeatStd::Int subsamplingFactor = GetSubsamplingFactor();
   renderTargetWidth /= subsamplingFactor;
   renderTargetHeight /= subsamplingFactor;

   WindowRect scissorRect;
   GetScissor(camera, scissorRect);
   Candera::Rectangle pixelScissorRect = scissorRect.GetRectInPixel();

   WindowRect viewportRect;
   GetViewport(camera, viewportRect);
   Candera::Rectangle pixelViewportRect = viewportRect.GetRectInPixel();

   // Shift scissor and viewport to 0/0 pixelViewportRect
   FeatStd::Float left = pixelScissorRect.GetLeft() - pixelViewportRect.GetLeft();
   if (left < 0.0F)
   {
      left = 0.0F;
   }
   pixelScissorRect.SetLeft(left);

   FeatStd::Float top = pixelScissorRect.GetTop() - pixelViewportRect.GetTop();
   if (top < 0.0F)
   {
      top = 0.0F;
   }
   pixelScissorRect.SetTop(top);

   pixelScissorRect.SetWidth(pixelScissorRect.GetWidth() / subsamplingFactor);
   pixelScissorRect.SetHeight(pixelScissorRect.GetHeight() / subsamplingFactor);

   pixelViewportRect.SetLeft(0.0F);
   pixelViewportRect.SetTop(0.0F);
   pixelViewportRect.SetWidth(pixelViewportRect.GetWidth() / subsamplingFactor);
   pixelViewportRect.SetHeight(pixelViewportRect.GetHeight() / subsamplingFactor);

   scissorRect.SetRectInPixel(pixelScissorRect, static_cast<Candera::Float>(renderTargetWidth), static_cast<Candera::Float>(renderTargetHeight));
   viewportRect.SetRectInPixel(pixelViewportRect, static_cast<Candera::Float>(renderTargetWidth), static_cast<Candera::Float>(renderTargetHeight));

   SetViewport(camera, viewportRect);
   SetScissor(camera, scissorRect);

   if (NULL != camera)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: SetOriginalCamera: origRT=%p, blurRT=%p, camera=%s",
                          camera->GetRenderTarget(), renderTarget, camera->GetName() != NULL ? camera->GetName() : "null"));

      // Finally change the render target of the original camera to the new render target
      camera->SetRenderTarget(renderTarget);
      renderTargetSetup.backupClearColorEnabled = camera->IsClearColorEnabled();

      // Since we set the original camera to a lower sequence number, we want to keep a backup of the original number
      renderTargetSetup.backupSequenceNumber = static_cast<Candera::Int16>(camera->GetSequenceNumber());

      //We need to scale the 2D Camera so that the content fits into the smaller render target
      Candera::Vector2 cameraScaling = Candera::Vector2(static_cast<Candera::Float>(subsamplingFactor), static_cast<Candera::Float>(subsamplingFactor));
      camera->Scale(cameraScaling);
   }
}


void BlurWidget2D::ResetOriginalCamera(OriginalCamera2D* camera)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Reset Original Camera: [%s]", GetLegacyName()));
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("BlurWidget2D: Camera is missing (Null Pointer): [%s]", GetLegacyName()));
   }
   else
   {
      RenderTargetSetup2D renderTargetSetup = m_blurCameraMap[camera];

      SetViewport(camera, renderTargetSetup.backupViewportRect);
      SetScissor(camera, renderTargetSetup.backupScissorRect);

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Reset Original Camera: modifiedBySC=%u, origRT=%p, blurRT=%p, camera=%s",
                          m_isOriginalCameraModifiedBySceneComposer, renderTargetSetup.originalRenderTarget,
                          camera->GetRenderTarget(), camera->GetName() != NULL ? camera->GetName() : "null"));

      if (!m_isOriginalCameraModifiedBySceneComposer)
      {
         Courier::Gdu(0).SetActivate(true);//fet2hi:force increment the invalidation id

         camera->SetRenderTarget(renderTargetSetup.originalRenderTarget);
         camera->SetClearColorEnabled(renderTargetSetup.backupClearColorEnabled);
         camera->SetSequenceNumber(renderTargetSetup.backupSequenceNumber);
      }

      //We need to revert the scaling of the 2D Camera
      FeatStd::Int subsamplingFactor = GetSubsamplingFactor();
      Candera::Vector2 cameraScaling = Candera::Vector2(1.0F / static_cast<Candera::Float>(subsamplingFactor), 1.0F / static_cast<Candera::Float>(subsamplingFactor));
      camera->Scale(cameraScaling);
   }
}


void BlurWidget2D::SetFinalPassCamera(OriginalCamera2D* originalCamera, Candera::Camera* finalCamera)
{
   if (NULL == originalCamera)
   {
      ETG_TRACE_ERR(("BlurWidget2D: The camera is missing (Null Pointer): [%s]", GetLegacyName()));
      return;
   }

   if (NULL == finalCamera)
   {
      ETG_TRACE_ERR(("BlurWidget2D: The camera is missing (Null Pointer): [%s]", GetLegacyName()));
   }

   RenderTargetSetup2D renderTargetSetup = m_blurCameraMap[originalCamera];

   SetViewport(finalCamera, renderTargetSetup.backupViewportRect);
   SetScissor(finalCamera, renderTargetSetup.backupScissorRect);

   if (!(m_outputRenderModeBlending.PointsToNull() || (NULL == finalCamera)))
   {
      if (finalCamera->GetAppearance().PointsToNull() == true)
      {
         finalCamera->SetAppearance(Candera::Appearance::Create());
      }

      if (GetBlendOutput() == true)
      {
         FeatStd::MemoryManagement::SharedPointer<Candera::Appearance> appearance = finalCamera->GetAppearance();
         appearance->SetRenderMode(m_outputRenderModeBlending);
      }
      else
      {
         FeatStd::MemoryManagement::SharedPointer<Candera::Appearance> appearance = finalCamera->GetAppearance();
         appearance->SetRenderMode(m_outputRenderModeOverwrite);
      }
   }

   CopyCameraSettings(originalCamera, finalCamera, false, false);

   //enable color clear so that the first pass buffer is cleared before each rendering
   originalCamera->SetClearColorEnabled(true);
}


#ifdef COURIER_ENHANCED_ENABLED
void BlurWidget2D::AddBlurPassGdus()
{
   Courier::View* viewScene = GetParentView();

   if (NULL != viewScene)
   {
      Courier::ViewHandler* blurViewHandler = dynamic_cast<Courier::ViewHandler*>(viewScene->GetViewHandler());

      if (NULL == blurViewHandler)
      {
         ETG_TRACE_ERR(("BlurWidget2D: The View Handler for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      //BlurRenderer* blurRenderer = dynamic_cast<BlurRenderer*>( blurViewHandler->GetRenderer());
      hmibase::view::Renderer* blurRenderer = dynamic_cast<hmibase::view::Renderer*>(blurViewHandler->GetRenderer());

      if (NULL == blurRenderer)
      {
         ETG_TRACE_ERR(("BlurWidget2D: The Renderer for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      BlurCamera2DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
      BlurCamera2DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

      while (itBlurCameraMap != itBlurCameraMapEnd)
      {
         GraphicDeviceUnit* firstPass = itBlurCameraMap->second.passOneGdu;
         GraphicDeviceUnit* secondPass = itBlurCameraMap->second.passTwoGdu;

         if (NULL != firstPass)
         {
            if (!blurRenderer->AddGdu(firstPass))
            {
               ETG_TRACE_ERR(("BlurWidget2D: Adding pass-one GDU failed: [%s]", GetLegacyName()));
            }

            if (!blurRenderer->LoadGdu(firstPass))
            {
               ETG_TRACE_ERR(("BlurWidget2D: Loading pass-one GDU failed: [%s]", GetLegacyName()));
            }
         }

         if (GetMultipass() && (NULL != secondPass))
         {
            if (!blurRenderer->AddGdu(secondPass))
            {
               ETG_TRACE_ERR(("BlurWidget2D: Adding pass-two GDU failed: [%s]", GetLegacyName()));
            }

            if (!blurRenderer->LoadGdu(secondPass))
            {
               ETG_TRACE_ERR(("BlurWidget2D: Loading pass-two GDU failed: [%s]", GetLegacyName()));
            }
         }
         ++itBlurCameraMap;
      }
   }
}


void BlurWidget2D::RemoveBlurPassGdus()
{
   Courier::View* viewScene = GetParentView();

   if (NULL != viewScene)
   {
      Courier::ViewHandler* blurViewHandler = dynamic_cast<Courier::ViewHandler*>(viewScene->GetViewHandler());

      if (NULL == blurViewHandler)
      {
         ETG_TRACE_ERR(("BlurWidget2D: The View Handler for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      //BlurRenderer* blurRenderer = dynamic_cast<BlurRenderer*>( blurViewHandler->GetRenderer());
      hmibase::view::Renderer* blurRenderer = dynamic_cast<hmibase::view::Renderer*>(blurViewHandler->GetRenderer());

      if (NULL == blurRenderer)
      {
         ETG_TRACE_ERR(("BlurWidget2D: The Renderer for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      BlurCamera2DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();

      while (itBlurCameraMap != m_blurCameraMap.end())
      {
         GraphicDeviceUnit* firstPass = itBlurCameraMap->second.passOneGdu;
         GraphicDeviceUnit* secondPass = itBlurCameraMap->second.passTwoGdu;

         if (NULL != firstPass)
         {
            if (!blurRenderer->RemoveGdu(firstPass))
            {
               ETG_TRACE_ERR(("BlurWidget2D: Removing pass-one GDU failed: [%s]", GetLegacyName()));
            }
         }

         if (GetMultipass() && (NULL != secondPass))
         {
            if (!blurRenderer->RemoveGdu(secondPass))
            {
               ETG_TRACE_ERR(("BlurWidget2D: Removing pass-two GDU failed: [%s]", GetLegacyName()));
            }
         }
         ++itBlurCameraMap;
      }
   }
}


void BlurWidget2D::SetupCourier(bool inBlurEnabled)
{
   Courier::View* viewScene = GetParentView();

   if (NULL != viewScene)
   {
      BlurViewScene2D* blurViewScene = dynamic_cast<BlurViewScene2D*>(viewScene);

      if (NULL == blurViewScene)
      {
         ETG_TRACE_ERR(("BlurWidget2D: The Blur View Scene is missing: [%s]", GetLegacyName()));
         return;
      }

      if (inBlurEnabled)
      {
         blurViewScene->AddBlurWidget(this);
      }
      else
      {
         blurViewScene->RemoveBlurWidget(this);
      }
   }
}


void BlurWidget2D::OnParentViewActivate(bool activate)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: OnParentViewActivate = %5s: [%s]", activate ? "true" : "false", GetLegacyName()));
   if (m_viewSceneActive != activate)
   {
      m_viewSceneActive = activate;

      if (((m_viewSceneActive == false) || (m_viewSceneRenderingEnabled == false)) && (m_blurEnabledInternal == true))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: OnParentViewActivate -> Disabling Blur Widget [%s]", GetLegacyName()));
         Finalize();
         m_updateAfterInactive = true;
      }
   }
}


void BlurWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: OnParentViewRenderingEnabled = %5s: [%s]", enable ? "true" : "false", GetLegacyName()));
   if (m_viewSceneRenderingEnabled != enable)
   {
      m_viewSceneRenderingEnabled = enable;

      if (((m_viewSceneActive == false) || (m_viewSceneRenderingEnabled == false)) && (m_blurEnabledInternal == true))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: OnParentViewRenderingEnabled -> Disabling Blur Widget [%s]", GetLegacyName()));
         Finalize();
         m_updateAfterInactive = true;
      }
   }
}


Camera2DVectorType BlurWidget2D::GetBlurCameras(OriginalCamera2D* camera, Candera::Int passIndex)
{
   Camera2DVectorType result;

   if ((passIndex <= 0) || (passIndex > 2))
   {
      ETG_TRACE_ERR(("BlurWidget2D: Invalid blur pass index: [%s]", GetLegacyName()));
      return result;
   }

   if ((camera != 0) && (m_blurCameraMap.count(camera) > 0))
   {
      Candera::Scene* passOneScene = m_blurCameraMap[camera].passOneScene;
      Candera::Scene* passTwoScene = m_blurCameraMap[camera].passTwoScene;

      if (1 == passIndex)
      {
         Candera::Camera* camera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(passOneScene, Camera::GetTypeId()));

         if (NULL != camera)
         {
            result.push_back(camera);
         }
      }
      else if ((2 == passIndex) && (NULL != passTwoScene))
      {
         Candera::Camera* camera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(passTwoScene, Camera::GetTypeId()));

         if (NULL != camera)
         {
            result.push_back(camera);
         }
      }
   }
   return result;
}


OriginalCamera2DVectorType BlurWidget2D::GetOriginalCameras()
{
   OriginalCamera2DVectorType result;
   BlurCamera2DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
   BlurCamera2DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

   while (itBlurCameraMap != itBlurCameraMapEnd)
   {
      result.push_back(itBlurCameraMap->first);
      ++itBlurCameraMap;
   }

   return result;
}


hmibase::widget::blur::WindowRect BlurWidget2D::GetOriginalCameraViewport(OriginalCamera2D* camera)
{
   if ((camera != 0) && (m_blurCameraMap.count(camera) > 0))
   {
      return m_blurCameraMap[camera].backupViewportRect;
   }

   return WindowRect();
}


#endif // COURIER_ENHANCED_ENABLED


void BlurWidget2D::SetMetaInfoInt(char* item, Candera::Int newValue, GraphicDeviceUnit* blurringGdu)
{
   const MetaInfo::GraphicDeviceUnitMetaInfo* metaInfoBlurring = DevicePackageDescriptor::GetMetaInformation(blurringGdu->GetUnitType());

   Candera::Char temp[25];
   if (0 < Candera::StringPlatform::StringPrintf(temp, sizeof(temp), "%d", newValue))
   {
      if (NULL != metaInfoBlurring->LookupItem(item))
      {
         if (!metaInfoBlurring->LookupItem(item)->Set(blurringGdu, temp))
         {
            ETG_TRACE_ERR(("BlurWidget2D: Setting a value of the off-screen render target failed: [%s]", GetLegacyName()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget2D: Meta Info item not found: [%s]", GetLegacyName()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("BlurWidget2D: Writing the height of an render target into char array failed: [%s]", GetLegacyName()));
   }
}


void BlurWidget2D::OnDarkenPropertyChanged()
{
   FeatStd::Float darken = GetDarken();
   if (fabs(m_darkenUniform[0] - darken) > FLT_EPSILON)
   {
      m_darkenUniform[0] = darken;
      m_darkenUniform[1] = darken;
      m_darkenUniform[2] = darken;
      m_darkenUniform[3] = 1.0F;
      m_validateBlurParameters = true;
   }
}


void BlurWidget2D::OnBlurMaskVariantPropertyChanged()
{
   // Reset everything when blur mask variant changes
   Finalize();

   if (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)
   {
      SetSubsamplingFactor(1);
   }
   else if (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask)
   {
      SetSubsamplingFactor(1);
   }
   else if (GetBlurMaskVariant() == enBlurMaskVariant::NoBlurMask)
   {
      // do nothing
   }
   else
   {
      // Unknown variant -> do nothing
   }
}


void BlurWidget2D::OnRenderNodeMaskPropertyChanged()
{
   if (m_oldNodeMask != 0)
   {
      m_oldNodeMask->SetRenderingEnabled(m_originalNodeRenderingEnabled);
   }

   Candera::Node2D* renderNodeMask = GetRenderNodeMask();
   if (renderNodeMask != 0)
   {
      UpdateBlurBitmapTransform();

      m_originalNodeRenderingEnabled = renderNodeMask->IsRenderingEnabled();
      renderNodeMask->SetRenderingEnabled(false);
   }
   m_oldNodeMask = renderNodeMask;
}


void BlurWidget2D::OnBlurShaderVariantPropertyChanged()
{
   enShaderVariant::Enum shaderVariant = GetBlurShaderVariant();
   if ((enShaderVariant::DynamicShader == shaderVariant) && (!GetMultipass()))
   {
      ETG_TRACE_ERR(("BlurWidget2D: The Dynamic Shader does not support single pass blur: [%s]", GetLegacyName()));
   }

   if ((enShaderVariant::StaticShader == shaderVariant) && (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask))
   {
      ETG_TRACE_ERR(("BlurWidget2D: Bitmap Blurring is not supported with the static shader variant: [%s]", GetLegacyName()));
   }

   m_validateShaderType = true;
}


void BlurWidget2D::OnBitmapMaskPropertyChanged()
{
   CreateTextureFromBitmap(GetBitmapMask());
   UpdateBlurBitmapTransform();
}


void BlurWidget2D::OnBlendOutputPropertyChanged()
{
   // As a basic setting is changed, the widget has to be reinitialized
   Finalize();
}


void BlurWidget2D::OnMultipassPropertyChanged()
{
   // As a basic setting is changed, the widget has to be reinitialized
   Finalize();
}


void BlurWidget2D::InvalidateBlurWidget()
{
   // On property changes of the widget, the blur has to be redrawn from the beginning, as all parameters except
   // darken need to redo the first blur pass
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Invalidate BlurWidget: [%s]", GetLegacyName()));

   FeatStd::Optional<Candera::Rectangle> dirtyArea;
   bool dirtyAreaInvalidationEnabled = GetUseManualDirtyArea();
   bool blurEnabled = GetBlurEnabled();
   if (blurEnabled == true)
   {
      if (dirtyAreaInvalidationEnabled == true)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: Invalidate BlurWidget: Manual Dirty Area: [%s]", GetLegacyName()));
         dirtyArea = GetDirtyAreaRectangle();
      }
   }

#ifdef COURIER_ENHANCED_ENABLED
   if (GetParentView() != 0)
   {
      GetParentView()->Invalidate(dirtyArea);
   }
#else
   Invalidate();
#endif
}


void BlurWidget2D::ActivateContext()
{
   OriginalCamera2D* camera = Dynamic_Cast<OriginalCamera2D*>(GetNode());
   if (camera != 0)
   {
      if (m_blurEnabledInternal == false)
      {
         if ((camera->GetRenderTarget() != 0) && (camera->GetRenderTarget()->Activate()))
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: ActivateContext: [%s]", GetLegacyName()));
         }
         else
         {
            ETG_TRACE_ERR(("BlurWidget2D: ActivateContext failed: [%s]", GetLegacyName()));
         }
      }
      else
      {
         if (m_blurCameraMap.count(camera) > 0)
         {
            if ((m_blurCameraMap[camera].originalRenderTarget != 0) && (m_blurCameraMap[camera].originalRenderTarget->Activate()))
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget2D: ActivateContext: [%s]", GetLegacyName()));
            }
            else
            {
               ETG_TRACE_ERR(("BlurWidget2D: ActivateContext failed: [%s]", GetLegacyName()));
            }
         }
         else
         {
            ETG_TRACE_ERR(("BlurWidget2D: ActivateContext failed: [%s]", GetLegacyName()));
         }
      }
   }
}


}   /* hmibase */
}   /* widget */
}   /* blur */
