/* ***************************************************************************************
* FILE:          BaseWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BaseWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "BaseWidget2D.h"

#include <Focus/FData.h>
#include <Focus/FManager.h>
#include <View/CGI/CgiExtensions/ViewScene2D.h>
#include <View/CGI/Widget/WidgetController.h>
#include <Widgets/2D/Appearance/Appearance2D.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>
#include <Widgets/2D/Common/ClippingDynamicPropertyHost.h>


#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BASEWIDGET
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BaseWidget2D.cpp.trc.h"
#endif


/*****************************************************************************/
class InvalidationContainerUtils
{
   public:
      static bool IsInvalidationContainer(Candera::Node2D& node)
      {
         return node.GetValue(CdaDynamicPropertyInstance(InvalidationContainer));
      }

      CdaDynamicPropertiesDefinition(Candera::Layouter, Candera::CanderaObject);

      CdaDynamicProperty(InvalidationContainer, bool);
      CdaDynamicPropertyDefaultValue(false);
      CdaDynamicPropertyDescription("If set to true and dirty rectangle repainting is enabled then the invalidation from the widgets associated to descendant nodes will be done for the bounding rectangle of this node.")
      CdaDynamicPropertyCategory("Layout")
      CdaDynamicPropertyEnd();

      CdaDynamicPropertiesEnd();
};


CGI_WIDGET_RTTI_DEFINITION(BaseWidget2D);


/*****************************************************************************/
BaseWidget2D::BaseWidget2D() :
   _assetProvider(NULL),
   _original(NULL),
   _listItemInfo(NULL),
   _appearance(NULL),
   _cachedName(NULL),
   _cachedStringIdentifier(NULL),
   _invalidationContainerLevel(INVALIDATION_CONTAINER_INVALID)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Widget constructor [%p]", this));

   SetVisibleEnabledInternal(true);//legacy
   SetLastEffectiveRenderingEnabledInternal(true);
}


/*****************************************************************************/
BaseWidget2D::~BaseWidget2D()
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Widget destructor [%p.%50s.%s]", this, GetViewName(),
                       !_legacyName.empty() ? _legacyName.c_str() : (GetName() != NULL ? GetName() : "<null>")));

   ClearController(*this);

   _assetProvider = NULL;
   _original = NULL;

   if (_listItemInfo != NULL)
   {
      FEATSTD_DELETE(_listItemInfo);
      _listItemInfo = NULL;
   }

   _cachedName = NULL;
   CleanupCachedStringIdentifier();
   _cachedStringIdentifier = NULL;

   if (_appearance != NULL)
   {
      ClearAppearance();
      _appearance = NULL;
   }
}


/*****************************************************************************/
void BaseWidget2D::Init(Candera::AssetProvider* assetProvider)
{
   Base::Init(assetProvider);
   _assetProvider = assetProvider;
   InitWidget();
}


/*****************************************************************************/
void BaseWidget2D::Finalize()
{
}


/*****************************************************************************/
const BaseWidget2D* BaseWidget2D::ToBaseWidget2D() const
{
   return this;
}


/*****************************************************************************/
BaseWidget2D* BaseWidget2D::ToBaseWidget2D()
{
   return this;
}


/*****************************************************************************/
bool BaseWidget2D::Is2D() const
{
   return true;
}


/*****************************************************************************/
Candera::Node2D* BaseWidget2D::GetNode2D() const
{
   return GetNode();
}


/*****************************************************************************/
bool BaseWidget2D::Is3D() const
{
   return false;
}


/*****************************************************************************/
Candera::Node* BaseWidget2D::GetNode3D() const
{
   return NULL;
}


/*****************************************************************************/
void BaseWidget2D::SetEnableStatus(bool enabled)
{
   SetEnabled(enabled);
}


/*****************************************************************************/
bool BaseWidget2D::GetEnableStatus() const
{
   return IsEnabled();
}


/*****************************************************************************/
void BaseWidget2D::SetVisibleStatus(bool visible)
{
   SetVisible(visible);
}


/*****************************************************************************/
bool BaseWidget2D::GetVisibleStatus() const
{
   return IsVisible();
}


/*****************************************************************************/
Candera::AssetProvider* BaseWidget2D::GetAssetProvider()
{
   return _assetProvider;
}


/*****************************************************************************/
const ControlTemplateCloneableWidget* BaseWidget2D::GetOriginal() const
{
   return _original;
}


/*****************************************************************************/
const Focus::FListItemData* BaseWidget2D::GetListItemInfo() const
{
   return _listItemInfo;
}


/*****************************************************************************/
hmibase::widget::appearance::Appearance* BaseWidget2D::GetAppearance()
{
   return _appearance;
}


/*****************************************************************************/
void BaseWidget2D::SetAppearance(hmibase::widget::appearance::Appearance* app)
{
   if (_appearance != app)
   {
      ClearAppearance();
      _appearance = app;
   }
}


/*****************************************************************************/
void BaseWidget2D::ClearAppearance()
{
   if (_appearance != NULL)
   {
      FEATSTD_DELETE(_appearance);
      _appearance = NULL;
   }
}


/*****************************************************************************/
hmibase::widget::appearance::AppearanceState BaseWidget2D::GetAppearanceState()
{
   return AppearanceState(IsFocused(), IsActive(), IsTouched(), IsEnabled());
}


/*****************************************************************************/
bool BaseWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   _original = originalWidget;

   if (_listItemInfo == NULL)
   {
      _listItemInfo = FEATSTD_NEW(Focus::FListItemData);
   }
   if (_listItemInfo != NULL)
   {
      _listItemInfo->ListId = controlTemplateMap.GetOwnerId();
      _listItemInfo->Index = controlTemplateMap.GetIndex();
   }

   const BaseWidget2D* original = CLONEABLE_WIDGET_CAST<const BaseWidget2D*>(originalWidget);
   if (original == NULL)
   {
      return false;
   }

   const_cast<BaseWidget2D*>(original)->SetItemTemplateInternal(true);
   SetItemCloneInternal(true);

   controlTemplateMap.CloneProperties(*original, *this);

   SetControllerId(original->GetControllerId());
   DELEGATE_WIDGET_ACTION(CloneFrom(*this, *original, controlTemplateMap));

   if (ControlTemplateBinding::IsEnabledBindable(*this))
   {
      SetEnabled(ControlTemplateBinding::GetEnabledValue(*this));
   }
   else
   {
      SetEnabled(original->IsEnabled());
   }

   if (ControlTemplateBinding::IsUserDataBindable(*this))
   {
      SetUserData(ControlTemplateBinding::GetUserDataValue(*this));
   }
   else
   {
      SetUserData(original->GetUserData());
   }

   SetVisibleEnabled(original->IsVisibleEnabled());
   SetVisible(original->IsVisible());

   SetAppearanceId(original->GetAppearanceId());

   return true;
}


/*****************************************************************************/
bool BaseWidget2D::OnMessage(const Message& msg)
{
   if (DELEGATE_WIDGET_BOOL_FUNC(OnMessage(*this, msg)))
   {
      return true;
   }

   if (Base::OnMessage(msg))
   {
      return true;
   }

   //dispatch view events to cloned widgets
   //todo: test this before integrating
   //if (_original != NULL)
   //{
   //   DispatchViewEvent(msg);
   //}

   bool consumed = false;
   switch (msg.GetId())
   {
      case Courier::CultureChangeIndMsg::ID:
         CultureChanged();
         break;

      case WidgetCheckReqMsg::ID:
      {
         const WidgetCheckReqMsg* widgetCheckReqMsg = Courier::message_cast<const WidgetCheckReqMsg*>(&msg);
         if ((widgetCheckReqMsg != NULL) && (widgetCheckReqMsg->GetCallback() != NULL))
         {
            consumed = widgetCheckReqMsg->GetCallback()->CheckWidget(this);
         }
      }
      break;

      default:
         break;
   }

   return consumed;
}


/*****************************************************************************/
void BaseWidget2D::OnFocus()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnFocus %s", HMIBASE_TO_STRING_VW(this)));

   Courier::View* view = GetParentView();
   if (view != NULL)
   {
      //inform focus manager synchronously about the new focused widget
      Focus::FManager::getInstance().onFocusChanged(view->GetId(), GetLegacyName());
   }

   Invalidate();
}


/*****************************************************************************/
void BaseWidget2D::OnLostFocus()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnLostFocus %s", HMIBASE_TO_STRING_VW(this)));

   Invalidate();
}


/*****************************************************************************/
void BaseWidget2D::OnBeforeNodeChanged()
{
   Base::OnBeforeNodeChanged();

   if (_appearance != NULL)
   {
      FEATSTD_DELETE(_appearance);
      _appearance = NULL;
   }
   _invalidationContainerLevel = INVALIDATION_CONTAINER_INVALID;
}


/*****************************************************************************/
void BaseWidget2D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);
   DELEGATE_WIDGET_ACTION(OnChanged(*this, propertyId));

   switch (propertyId)
   {
      case AppearanceIdPropertyId:
      {
         if (_appearance != NULL)
         {
            FEATSTD_DELETE(_appearance);
            _appearance = NULL;
         }
         break;
      }

      case VisiblePropertyId:
      {
         if ((GetNode() != NULL) && IsVisibleEnabled() && (IsVisible() != GetNode()->IsRenderingEnabled()))
         {
            if (Candera::Layouter::IsCollapsible(*GetNode()))
            {
               Candera::Layouter::InvalidateLayout(GetNode());
            }

            if (IsVisible())
            {
               // 0->1 : First enable rendering, than invalidate the screen
               GetNode()->SetRenderingEnabled(true);
               Invalidate();
            }
            else
            {
               // 1->0 : Invalidate the screen/area before disable rendering
               Invalidate();
               GetNode()->SetRenderingEnabled(false);
            }
         }
         break;
      }

      default:
         break;
   }
}


/*****************************************************************************/
void BaseWidget2D::SetFlagValue(FeatStd::UInt32 propertyId, size_t pos, bool value)
{
   Base::SetFlagValue(propertyId, pos, value);

   // handles situations when the node visibility is changed from other places (or set manually in scene composer)
   if ((propertyId == VisiblePropertyId) && (GetNode() != NULL) && IsVisibleEnabled() && (IsVisible() != GetNode()->IsRenderingEnabled()))
   {
      OnChanged(VisiblePropertyId);
   }
}


/*****************************************************************************/
void BaseWidget2D::Update()
{
   Base::Update();

   UpdateController(*this);
   DELEGATE_WIDGET_ACTION(Update(*this));

   if ((_appearance == NULL) && (GetAppearanceId() > 0))
   {
      hmibase::widget::appearance::AppearanceTemplateContainer* appTemplateContainer = hmibase::widget::appearance::AppearanceTemplateRegistry::getItem(GetAppearanceId());
      if (appTemplateContainer != NULL)
      {
         _appearance = appTemplateContainer->createAppearance(*this);
         if (_appearance == NULL)
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Update Failed to create appearance from template with id=%u %s",
                               GetAppearanceId(), HMIBASE_TO_STRING_VW(this)));
         }
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Update No appearance template registered with id=%u %s",
                            GetAppearanceId(), HMIBASE_TO_STRING_VW(this)));
      }
   }

   if (_appearance != NULL)
   {
      hmibase::widget::appearance::Appearance::UpdateResult result = _appearance->update();
      if (result & hmibase::widget::appearance::Appearance::LayoutRequired)
      {
         InvalidateLayout();
      }
      if (result & hmibase::widget::appearance::Appearance::RenderRequired)
      {
         Invalidate();
      }
   }
}


/*****************************************************************************/
bool BaseWidget2D::IsEffectiveVisible() const
{
   return (GetNode() != NULL) && GetNode()->IsEffectiveRenderingEnabled();
}


/*****************************************************************************/
void BaseWidget2D::SetFocus(bool focus)
{
   Courier::View* view = GetParentView();
   if ((view != NULL) && (view->GetViewHandler() != NULL))
   {
      if (focus)
      {
         view->GetViewHandler()->SetFocus(view->GetId(), Courier::CompositePath(), Courier::ItemId(GetLegacyName()));
      }
      else
      {
         view->GetViewHandler()->LoseFocus(this);
      }
   }
}


/*****************************************************************************/
bool BaseWidget2D::IsPickIntersectingNode(const Candera::Camera2D& camera2D, const Candera::Node2D* node, const Candera::Vector2& point) const
{
   bool intersects = false;
   if (0 != node)
   {
      intersects = true;
      const Candera::Node2D* parent = node->GetParent();
      if (0 != parent)
      {
         const Candera::Vector2& pointInWorldSpace(Candera::Math2D::TransformViewportToScene(camera2D, Candera::Math2D::TransformRenderTargetToViewport(camera2D, point)));

         Candera::Rectangle touchableRectangle = ClippingDynamicPropertyHost::GetTouchableRectangle(*node);

         intersects = touchableRectangle.Contains(pointInWorldSpace);
      }

      intersects = intersects && node->IsPickIntersectingBoundingRectangle(camera2D, point);
   }

   return intersects;
}


/*****************************************************************************/
void BaseWidget2D::InitCachedStringIdentifier()
{
   CleanupCachedStringIdentifier();
   FEATSTD_DEBUG_ASSERT(NULL == _cachedStringIdentifier);

   const Candera::StringIdentifier* stringId = GetStringId();
   if (NULL != stringId)
   {
      // clone owner chain
      const Candera::StringIdentifier* owner = GetStringId();
      Candera::StringIdentifier* childClone = NULL;
      while (NULL != owner)
      {
         Candera::StringIdentifier* clone = FEATSTD_NEW(Candera::StringIdentifier);
         if (NULL == clone)
         {
            ETG_TRACE_ERR(("Memory of string identifier clone could not be allocated. Owner chain will be incomplete."));
            return;
         }

         if (NULL == _cachedStringIdentifier)
         {
            // first element
            _cachedStringIdentifier = clone;
         }

         clone->SetId(owner->GetId(), NULL); //only need the pointer

         if (NULL != childClone)
         {
            childClone->SetOwner(clone);
         }
         childClone = clone;

         owner = owner->GetOwner();
      }
   }
}


/*****************************************************************************/
void BaseWidget2D::CleanupCachedStringIdentifier()
{
   //free cached identifier memory
   const Candera::StringIdentifier* container = _cachedStringIdentifier;
   while (NULL != container)
   {
      const Candera::StringIdentifier* owner = container->GetOwner();
      FEATSTD_DELETE(container);
      container = owner;
   }
   _cachedStringIdentifier = NULL;
}


/*****************************************************************************/
const FeatStd::Char* BaseWidget2D::GetLegacyName()
{
   const Candera::StringIdentifier* container = GetStringId();
   const Candera::StringIdentifier* cachedStringIdentifier = _cachedStringIdentifier;
   bool isNameCacheInvalid = (_cachedName != GetName()) || (cachedStringIdentifier == 0);
   while ((!isNameCacheInvalid) && (container != NULL))
   {
      isNameCacheInvalid = (NULL == cachedStringIdentifier) || (container->GetId() != cachedStringIdentifier->GetId());

      container = container->GetOwner();
      if (cachedStringIdentifier != 0)
      {
         cachedStringIdentifier = cachedStringIdentifier->GetOwner();
      }
   }

   if (isNameCacheInvalid)
   {
      _cachedName = GetName();
      InitCachedStringIdentifier();
      InitLegacyName();
   }

   return _legacyName.c_str();
}


/*****************************************************************************/
static void addToLegacyName(std::string& legacyName, const Candera::StringIdentifier& stringId)
{
   //process parent first
   if (stringId.GetOwner() != NULL)
   {
      addToLegacyName(legacyName, *(stringId.GetOwner()));
   }

   const Candera::Char* id = stringId.GetId();
   if ((id != NULL) && (strnlen(id, 2) != 0))
   {
      std::string toParse(id);
      //look for backslash from right
      size_t pos = toParse.rfind('/');
      if (pos != std::string::npos)
      {
         if (!legacyName.empty())
         {
            legacyName.append("/");
         }

         //append the name contained in the specified StringIdentifier
         legacyName.append(toParse.substr(pos + 1));
      }
   }
}


/*****************************************************************************/
void BaseWidget2D::InitLegacyName()
{
   //clear old name
   _legacyName.clear();

   const Candera::StringIdentifier* stringId = GetStringId();

   //if name of this widget is set => use it
   if (GetName() != NULL)
   {
#if ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 3))
      //skip direct StringIdentifier because it is covered by GetName() (since CGIStudio 3.3)
      if (stringId != NULL)
      {
         stringId = stringId->GetOwner();
      }
#endif

      //append the ancestors
      if (stringId != NULL)
      {
         addToLegacyName(_legacyName, *stringId);
      }

      if (!_legacyName.empty())
      {
         _legacyName.append("/");
      }

      //append the set name
      _legacyName.append(GetName());
   }
   //else use only the StringIdentifier
   else if (stringId != NULL)
   {
      addToLegacyName(_legacyName, *stringId);
   }
   else
   {
      //nothing to do
   }
}


/*****************************************************************************/
Candera::Node2D* BaseWidget2D::GetInvalidationContainer()
{
   if ((GetNode() == NULL) || (_invalidationContainerLevel == INVALIDATION_CONTAINER_NOT_USED))
   {
      return NULL;
   }

   Candera::Node2D* invalidationContainer = NULL;

   //invalidation container was searched previously => get it and check if it is valid
   if (_invalidationContainerLevel >= 0)
   {
      invalidationContainer = GetNode();
      for (InvalidationContainerLevelType level = 0; (level < _invalidationContainerLevel) && (invalidationContainer != NULL); ++level)
      {
         invalidationContainer = invalidationContainer->GetParent();
      }

      //cached invalidation container level does not match the scene structure
      if (invalidationContainer == NULL)
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "GetInvalidationContainer WRONG(OutOfBounds) level=%d %s",
                             _invalidationContainerLevel,
                             HMIBASE_TO_STRING_VW(this)));

         _invalidationContainerLevel = INVALIDATION_CONTAINER_INVALID;
         invalidationContainer = NULL;
      }
      //node indicated by the cached invalidation container level does not not have the flag set
      else if (!InvalidationContainerUtils::IsInvalidationContainer(*invalidationContainer))
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "GetInvalidationContainer WRONG(FlagNotSet) level=%d invalidationContainer=%75s %s",
                             _invalidationContainerLevel,
                             HMIBASE_TO_STRING_N2D(invalidationContainer),
                             HMIBASE_TO_STRING_VW(this)));

         _invalidationContainerLevel = INVALIDATION_CONTAINER_INVALID;
         invalidationContainer = NULL;
      }
      else
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "GetInvalidationContainer CACHED level=%d invalidationContainer=%75s %s",
                             _invalidationContainerLevel,
                             HMIBASE_TO_STRING_N2D(invalidationContainer),
                             HMIBASE_TO_STRING_VW(this)));
      }
   }

   //need to search for invalidation container
   if (invalidationContainer == NULL)
   {
      _invalidationContainerLevel = INVALIDATION_CONTAINER_NOT_USED;

      //search the invalidation container starting with the parent (because for the current node is more optimal to use INVALIDATION_CONTAINER_NOT_USED)
      InvalidationContainerLevelType level = 1;
      for (Candera::Node2D* temp = GetNode()->GetParent(); temp != NULL; temp = temp->GetParent(), ++level)
      {
         //found a node which has the invalidation container flag set
         if (InvalidationContainerUtils::IsInvalidationContainer(*temp))
         {
            _invalidationContainerLevel = level;
            invalidationContainer = temp;

            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "GetInvalidationContainer NEW level=%d invalidationContainer=%75s %s",
                                _invalidationContainerLevel,
                                HMIBASE_TO_STRING_N2D(invalidationContainer),
                                HMIBASE_TO_STRING_VW(this)));
         }

         //root node is not a scene
         if ((temp->GetParent() == NULL) && !temp->IsTypeOf(Candera::Scene2D::GetTypeId()))
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "GetInvalidationContainer node does not have a scene as ancestor %s",
                                HMIBASE_TO_STRING_VW(this)));

            _invalidationContainerLevel = INVALIDATION_CONTAINER_INVALID;
            invalidationContainer = NULL;
         }
      }
   }

   return invalidationContainer;
}


/*****************************************************************************/
static bool CheckNodeVisibility(const Candera::Node2D* node, const Candera::Scene2D*& scene)
{
   bool visible = true;
   scene = NULL;

   while (node != NULL)
   {
      if (!node->IsRenderingEnabled())
      {
         visible = false;
      }
      if (node->GetParent() == NULL)
      {
         scene = Candera::Dynamic_Cast<const Candera::Scene2D*>(node);
      }
      node = node->GetParent();
   }
   return visible;
}


/*****************************************************************************/
void BaseWidget2D::Invalidate(bool forceInvalidate, const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   //no invalidation without parent view being set
   if (GetParentView() == NULL)
   {
      return;
   }

   //dirty area is specified explicitly
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
   if (dirtyArea.IsSet())
#else
   if (dirtyArea == true)
#endif
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Invalidate area=%25s %s",
                          HMIBASE_TO_STRING(*dirtyArea),
                          HMIBASE_TO_STRING_VW(this)));
      Base::Invalidate(dirtyArea);
      return;
   }

   //non HMIBase view scene is used or dirty rectangles are disabled => full invalidate
   hmibase::view::ViewScene2D* view = dynamic_cast<hmibase::view::ViewScene2D*>(GetParentView());
   if ((view == NULL) || !view->IsDirtyRectUsed())
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Invalidate %s", HMIBASE_TO_STRING_VW(this)));
      Base::Invalidate();
      return;
   }

   //node is not set => full invalidate
   if (GetNode() == NULL)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Invalidate WARNING node is not set %s", HMIBASE_TO_STRING_VW(this)));
      Base::Invalidate();
      return;
   }

   const Candera::Scene2D* scene = NULL;
   bool effectiveRenderingEnabled = CheckNodeVisibility(GetNode(), scene);

   //if the node not does not have a scene as ancestor it is not possible to determine its world transformation
   if (scene == NULL)
   {
      if (forceInvalidate)
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Invalidate forced %s", HMIBASE_TO_STRING_VW(this)));
         Base::Invalidate();
      }
      return;
   }

   //changes of the camera view matrix are not tracked. if such changes occur then a special handling has to be implemented for this camera.
   if (forceInvalidate || effectiveRenderingEnabled || IsLastEffectiveRenderingEnabled())
   {
      SetLastEffectiveRenderingEnabledInternal(effectiveRenderingEnabled);

      //widget uses an ancestor node to determine the dirty area
      Candera::Node2D* node = GetInvalidationContainer();
      if (node == NULL)
      {
         node = GetNode();
      }

      //get bounding rectangle and world transformation of the node
      Candera::Rectangle boundingRectangle;
      node->GetEffectiveBoundingRectangle(boundingRectangle);
      const Candera::Matrix3x2 worldTransformation(node->GetWorldTransform());

      //no info about previous occupied area => full invalidate
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
      if (!_occupiedArea.IsSet())
#else
      if (_occupiedArea == false)
#endif
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Invalidate initial %s", HMIBASE_TO_STRING_VW(this)));
         Base::Invalidate();
      }
      else
      {
         //different world transformations => invalidate for both of them
         if (_worldTransformation != worldTransformation)
         {
            Invalidate(*view, _worldTransformation, *_occupiedArea);
            Invalidate(*view, worldTransformation, boundingRectangle);
         }
         //same world transformation => invalidate the union of the rectangles
         else
         {
            Candera::Rectangle boundingRectangleUnion(boundingRectangle);
            boundingRectangleUnion.Union(*_occupiedArea);
            Invalidate(*view, worldTransformation, boundingRectangleUnion);
         }
      }

      _occupiedArea = FeatStd::Optional<Candera::Rectangle>(boundingRectangle);
      _worldTransformation = worldTransformation;
   }
}


/*****************************************************************************/
void BaseWidget2D::Invalidate(Courier::ViewScene2D& view, const Candera::Matrix3x2& worldTransformation, const Candera::Rectangle& boundingRectangle)
{
   Courier::ViewScene2D::CameraPtrVector& cameras = view.GetCameraPtrVector();
   for (size_t index = 0; index < cameras.Size(); ++index)
   {
      Candera::Camera2D* camera = cameras[index];
      if ((camera != NULL) && (camera->GetRenderTarget() != NULL))
      {
         Candera::Matrix3x2 transformation(worldTransformation * camera->GetViewMatrix()); // transformation to view space
         transformation.PreTranslate(camera->GetViewport().GetLeft(), camera->GetViewport().GetTop()); // transformation to screen space

         Candera::Rectangle rectangle(boundingRectangle);
         rectangle.Transform(transformation);

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Invalidate area=%25s camera=%50s %s",
                             HMIBASE_TO_STRING(rectangle),
                             HMIBASE_TO_STRING_N2D(camera),
                             HMIBASE_TO_STRING_VW(this)));
         view.Invalidate(camera, FeatStd::Optional<Candera::Rectangle>(rectangle));
      }
   }
}


/*****************************************************************************/
void BaseWidget2D::Invalidate(const FeatStd::Optional<Candera::Rectangle>& dirtyArea)
{
   Invalidate(false, dirtyArea);
}


/*****************************************************************************/
void BaseWidget2D::DispatchViewEvent(const Message& msg)
{
   switch (msg.GetId())
   {
      case Courier::ParentViewActivateEvent::ID:
      {
         bool activated = Courier::message_cast<const Courier::ParentViewActivateEvent*>(&msg)->GetActivated();
         OnParentViewActivate(activated);
         break;
      }
      case Courier::ParentViewRenderingEnabledEvent::ID:
      {
         bool enabled = Courier::message_cast<const Courier::ParentViewRenderingEnabledEvent*>(&msg)->GetEnabled();
         OnParentViewRenderingEnabled(enabled);
         break;
      }
      case Courier::ParentViewLoadEvent::ID:
      {
         bool loaded = Courier::message_cast<const Courier::ParentViewLoadEvent*>(&msg)->GetLoaded();
         OnParentViewLoad(loaded);
         break;
      }
      case Courier::TransitionStartedEvent::ID:
      {
         const Courier::TransitionStartedEvent* ev = Courier::message_cast<const Courier::TransitionStartedEvent*>(&msg);
         Courier::View* firstView = ev->GetFirstView();
         Courier::View* secondView = ev->GetSecondView();
         OnTransitionStarted(firstView, secondView);
         break;
      }
      case Courier::TransitionFinishedEvent::ID:
      {
         const Courier::TransitionFinishedEvent* ev = Courier::message_cast<const Courier::TransitionFinishedEvent*>(&msg);
         Courier::View* firstView = ev->GetFirstView();
         Courier::View* secondView = ev->GetSecondView();
         bool wasReset = ev->GetWasReset();
         OnTransitionFinished(firstView, secondView, wasReset);
         break;
      }
      default:
      {
         break;
      }
   }
}


/*****************************************************************************/
bool BaseWidget2D::composerPropVisibleFilter_Enabled() const
{
   //todo: set this to false and check all the widgets to find out which support enabled
   return true;
}
