/* ***************************************************************************************
 * FILE:          Appearance2D.h
 * SW-COMPONENT:  HMI-BASE
 *  DESCRIPTION:  Appearance2D is part of HMI-Base Widget Library
 *    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */
#if !defined(hmibase_Appearance2D_h)
#define hmibase_Appearance2D_h

#include "AppearanceState.h"
#include <hmibase/util/ItemRegistry.h>
#include <Widgets/utils/EffectUtils.h>


namespace Candera {
class BitmapBrush;
class Image2D;
class Node2D;
class Widget2D;
}


namespace hmibase {
namespace widget {
namespace color {
class ColorPropertyAccessor;
}


namespace appearance {
class AppearanceTemplate;


/**************************************************************************/
class Appearance
{
      friend class AppearanceNodeTraverser;

   public:
      enum UpdateResult
      {
         NoChange = 0,
         RenderRequired = 1 << 0,
         LayoutRequired = 1 << 1,
         LayoutAndRenderRequired = RenderRequired + LayoutRequired,
      };

      Appearance(AppearanceTemplate* tpl);
      virtual ~Appearance();

      UpdateResult update();

   protected:
      virtual UpdateResult updateSelf() = 0;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(Appearance);

      AppearanceTemplate* _template;
      Appearance* _next;
};


/**************************************************************************/
template <typename TOwner, typename TTarget, typename TData>
class AppearanceBase : public Appearance
{
   public:
      typedef TOwner OwnerType;
      typedef TTarget TargetType;
      typedef TData DataType;

      AppearanceBase(OwnerType& owner, TargetType& target, const typename DataType::SharedPointer& data, AppearanceTemplate* tpl)
         : Appearance(tpl), _owner(owner), _target(target), _data(data)
      {
      }

   protected:
      virtual UpdateResult updateSelf();

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(AppearanceBase);

      OwnerType& _owner;
      TargetType& _target;
      typename DataType::SharedPointer _data;
};


/**************************************************************************/
class AppearanceTemplate
{
      friend class Appearance;

   public:
      AppearanceTemplate() {}
      virtual ~AppearanceTemplate() {}

      typedef int PropertyHintType;
      virtual Appearance* create(Candera::Widget2D& widget, Candera::Node2D& node, PropertyHintType propertyHint) = 0;

   protected:
      virtual void onAppearanceDeleted(Appearance* /*app*/) {}

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(AppearanceTemplate);
};


/**************************************************************************/
class AppearanceTemplateContainer
{
   public:
      typedef const char* IdType;
      typedef AppearanceTemplate::PropertyHintType PropertyHintType;

      struct Entry
      {
         Entry(AppearanceTemplate* tpl = NULL, PropertyHintType propertyHint = PropertyHintType(), Entry* next = NULL) : _Template(tpl), _PropertyHint(propertyHint), _Next(next) {}
         ~Entry()
         {
            _Template = NULL;
            _Next = NULL;
         }

         AppearanceTemplate* _Template;
         PropertyHintType _PropertyHint;
         Entry* _Next;
      };

      struct NodeAndPropertyId
      {
         NodeAndPropertyId(const IdType& nodeId, PropertyHintType propertyHint = 0) : _NodeId(nodeId), _PropertyHint(propertyHint) {}

         IdType _NodeId;
         PropertyHintType _PropertyHint;
      };

      AppearanceTemplateContainer(bool idFilterEnabled = false);
      ~AppearanceTemplateContainer();

      Entry* get(const IdType& id) const;

      void add(const NodeAndPropertyId& nodeAndPropertyId, AppearanceTemplate* appTemplate);
      void clear(const IdType& id);

      void enableIdFilter(bool value);
      bool isIdFilterEnabled() const;
      bool matchesIdFilter(const IdType& id) const;

      Appearance* createAppearance(Candera::Widget2D& widget);

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(AppearanceTemplateContainer);

      typedef Courier::Identifier KeyType;
      typedef std::map<KeyType, Entry*> MapType;
      MapType _map;

      bool _idFilterEnabled;
};


/**************************************************************************/
class AppearanceHelper
{
   public:
      template <typename TTarget>
      static TTarget* getTarget(Candera::Node2D& node, int propertyHint);

   private:
      FEATSTD_MAKE_CLASS_STATIC(AppearanceHelper);
};


/**************************************************************************/
template <typename TAppearance, typename TTemplateData>
class AppearanceTemplateBase : public AppearanceTemplate
{
   public:
      typedef TAppearance AppearanceType;
      typedef typename TAppearance::OwnerType OwnerType;
      typedef typename TAppearance::TargetType TargetType;
      typedef TTemplateData TemplateDataType;

      AppearanceTemplateBase(const TemplateDataType& templateData) : _templateData(templateData)
      {
      }

      virtual ~AppearanceTemplateBase() {}

      virtual Appearance* create(Candera::Widget2D& widget, Candera::Node2D& node, int propertyHint)
      {
         Appearance* app = NULL;
         OwnerType* owner = Candera::Dynamic_Cast<OwnerType*>(&widget);
         TargetType* target = AppearanceHelper::getTarget<TargetType>(node, propertyHint);
         if ((owner != NULL) && (target != NULL))
         {
            app = createImpl(*owner, *target);
         }

         return app;
      }

   protected:
      virtual AppearanceType* createImpl(OwnerType& owner, TargetType& target) = 0;

      TemplateDataType _templateData;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(AppearanceTemplateBase);
};


/**************************************************************************/
typedef hmibase::util::ItemPtrRegistry<unsigned int, AppearanceTemplateContainer> AppearanceTemplateRegistry;


/**************************************************************************/
struct ImageWrapper
{
   typedef Candera::Id TId;

   TId _id;
   Candera::MemoryManagement::SharedPointer<Candera::Image2D> _image;
   bool _initialized;

   ImageWrapper(TId id = TId());
   ImageWrapper(Candera::MemoryManagement::SharedPointer<Candera::Image2D> image);

   void init();
};


/**************************************************************************/
typedef AppearanceStateData<Candera::Id> AppearanceAssetIdData;
typedef ShareableAppearanceStateData<Candera::MemoryManagement::SharedPointer<Candera::Image2D> > AppearanceImageData;
typedef ShareableAppearanceStateData<ImageWrapper> AppearanceImageWrapperData;
typedef ShareableAppearanceStateData<Candera::Color> AppearanceColorData;


typedef AppearanceBase<BaseWidget2D, Candera::BitmapBrush, AppearanceImageData> ImageAppearance;
typedef AppearanceBase<BaseWidget2D, Candera::BitmapBrush, AppearanceImageWrapperData> ImageWrapperAppearance;
typedef AppearanceBase<BaseWidget2D, Candera::ColorProperty, AppearanceColorData> ColorAppearance;
typedef AppearanceBase<BaseWidget2D, hmibase::widget::color::ColorPropertyAccessor, AppearanceColorData> ColorAccessorAppearance;


/**************************************************************************/
class ImageIdAppearanceTemplate : public AppearanceTemplateBase<ImageWrapperAppearance, AppearanceAssetIdData>
{
      typedef AppearanceTemplateBase<ImageWrapperAppearance, AppearanceAssetIdData> Base;

   public:
      ImageIdAppearanceTemplate(const AppearanceAssetIdData& templateData) : Base(templateData)
      {
      }

   protected:
      virtual ImageWrapperAppearance* createImpl(OwnerType& owner, TargetType& target);
      virtual void onAppearanceDeleted(Appearance* app);

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ImageIdAppearanceTemplate);

      AppearanceImageWrapperData::SharedPointer _data;
};


/**************************************************************************/
class ImageAppearanceTemplate : public AppearanceTemplateBase<ImageAppearance, AppearanceImageData::SharedPointer>
{
      typedef AppearanceTemplateBase<ImageAppearance, AppearanceImageData::SharedPointer> Base;

   public:
      ImageAppearanceTemplate(const AppearanceImageData::SharedPointer& templateData) : Base(templateData)
      {
      }

   protected:
      virtual ImageAppearance* createImpl(OwnerType& owner, TargetType& target)
      {
         return FEATSTD_NEW(ImageAppearance)(owner, target, _templateData, NULL/*this class is used for appearance templates on the fly with short life*/);
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ImageAppearanceTemplate);
};


/**************************************************************************/
class ColorAppearanceTemplate : public AppearanceTemplateBase<ColorAppearance, AppearanceColorData::SharedPointer>
{
      typedef AppearanceTemplateBase<ColorAppearance, AppearanceColorData::SharedPointer> Base;

   public:
      ColorAppearanceTemplate(const AppearanceColorData::SharedPointer& templateData) : Base(templateData)
      {
      }

   protected:
      virtual ColorAppearance* createImpl(OwnerType& owner, TargetType& target)
      {
         return FEATSTD_NEW(ColorAppearance)(owner, target, _templateData, this);
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(ColorAppearanceTemplate);
};


/**************************************************************************/
//#define HMIBASE_APPEARANCE_ASSET_IMAGE(imgPrefix, imgName) ((imgName[0] == 0) ? Candera::MemoryManagement::SharedPointer<Candera::Image2D>() : Candera::MemoryManagement::SharedPointer<Candera::Image2D>(ImageLoader::getAssetBitmapImage(imgPrefix imgName)))

#define HMIBASE_APPEARANCE_ASSET_ID(id) AssetId_##id
#define HMIBASE_APPEARANCE_ASSET_ID_BITMASK(bitMask, bitIndex, id) (((bitMask & 1u << bitIndex) == 0) ? 0 : HMIBASE_APPEARANCE_ASSET_ID(id))
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, idSuffix) AssetId_##idPrefix##idSuffix
#define HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, bitIndex, idPrefix, idSuffix) (((bitMask & 1u << bitIndex) == 0) ? 0 : HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, idSuffix))

#define HMIBASE_APPEARANCE_BEGIN(id) \
   { \
      const unsigned int appId = id; \
      ::hmibase::widget::appearance::AppearanceTemplateContainer* appContainer = FEATSTD_NEW(::hmibase::widget::appearance::AppearanceTemplateContainer); \
      if (appContainer != NULL) \
      {
#define HMIBASE_APPEARANCE_END() \
         ::hmibase::widget::appearance::AppearanceTemplateRegistry::registerItem(appId, appContainer); \
      } \
   }

#define HMIBASE_APPEARANCE_NODE_NAME_FILTER_ENABLE(value) appContainer->enableIdFilter(value);

// appearance is applied to the widget's node
#define HMIBASE_APPEARANCE_NODE_NAME_WIDGET_NODE ""

// appearance is applied to all descendant nodes
//#define HMIBASE_APPEARANCE_NODE_NAME_ALL_NODES "#"

// appearance is applied to the outline color property of the specified node
#define HMIBASE_APPEARANCE_PROPERTY_OUTLINE_COLOR(nodeName) ::hmibase::widget::appearance::AppearanceTemplateContainer::NodeAndPropertyId(nodeName, ::hmibase::widget::utils::EffectUtils::colorid::Outline)

// appearance is applied to the drop shadow color property of the specified node
#define HMIBASE_APPEARANCE_PROPERTY_DROP_SHADOW_COLOR(nodeName) ::hmibase::widget::appearance::AppearanceTemplateContainer::NodeAndPropertyId(nodeName, ::hmibase::widget::utils::EffectUtils::colorid::DropShadow)

/**************************************************************************/
/* Image Id */
/**************************************************************************/
#define _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, imageData) \
   appContainer->add(nodeName, FEATSTD_NEW(::hmibase::widget::appearance::ImageIdAppearanceTemplate)(imageData));


/**************************************************************************/
#define HMIBASE_APPEARANCE_ASSET_ID_DATA(normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedPressedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId, focusedActivePressedDisabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID(normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(pressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(pressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(activeNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(activePressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(activeDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(activePressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedPressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedPressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedActiveNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedActivePressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedActiveDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID(focusedActivePressedDisabledId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, normalId, pressedId, disabledId, activeNormalId, activePressedId, activeDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_DATA(normalId, pressedId, disabledId, activeNormalId, activePressedId, activeDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId));


/**************************************************************************/
/* Disabled */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_D(idPrefix, normalId, disabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), ::Candera::Id(), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_D(nodeName, idPrefix, normalId, disabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_D(idPrefix, normalId, disabledId));


/**************************************************************************/
/* Disabled, Active */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_DA(idPrefix, normalId, disabledId, activeNormalId, activeDisabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), ::Candera::Id(), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), ::Candera::Id(), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activeNormalId), ::Candera::Id(), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activeDisabledId), ::Candera::Id())

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_DA(nodeName, idPrefix, normalId, disabledId, activeNormalId, activeDisabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_DA(idPrefix, normalId, disabledId, activeNormalId, activeDisabledId));

/**************************************************************************/
/* Pressed */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_P(idPrefix, normalId, pressedId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, pressedId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_P(nodeName, idPrefix, normalId, pressedId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_P(idPrefix, normalId, pressedId));


/**************************************************************************/
/* Pressed, Disabled */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_PD(idPrefix, normalId, pressedId, disabledId, pressedDisabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, pressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, pressedDisabledId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_PD(nodeName, idPrefix, normalId, pressedId, disabledId, pressedDisabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_PD(idPrefix, normalId, pressedId, disabledId, pressedDisabledId));

/**************************************************************************/
/* Pressed, Disabled, Active */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_PDA(idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, pressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, pressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activeNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activePressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activeDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activePressedDisabledId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_PDA(nodeName, idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_PDA(idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId));


/**************************************************************************/
/* Pressed, Disabled, Active, Focused */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_PDAF(idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedPressedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId, focusedActivePressedDisabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, pressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, pressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activeNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activePressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activeDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, activePressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedPressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedPressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedActiveNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedActivePressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedActiveDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedActivePressedDisabledId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_PDAF(nodeName, idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedPressedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId, focusedActivePressedDisabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_PDAF(idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedPressedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId, focusedActivePressedDisabledId));


/**************************************************************************/
/* Focused */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_F(idPrefix, normalId, focusedNormalId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_F(nodeName, idPrefix, normalId, focusedNormalId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_F(idPrefix, normalId, focusedNormalId));


/**************************************************************************/
/* Disabled, Focused */
#define HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_DF(idPrefix, normalId, disabledId, focusedNormalId, focusedDisabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_PREFIX(idPrefix, focusedDisabledId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_PREFIX_DF(nodeName, idPrefix, normalId, disabledId, focusedNormalId, focusedDisabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_PREFIX_DATA_DF(idPrefix, normalId, disabledId, focusedNormalId, focusedDisabledId));


/**************************************************************************/
/* bit mask */
namespace mask {
static const unsigned int _bit_N = 0;
static const unsigned int _bit_P = 1;
static const unsigned int _bit_D = 2;
static const unsigned int _bit_PD = 3;
static const unsigned int _bit_AN = 4;
static const unsigned int _bit_AP = 5;
static const unsigned int _bit_AD = 6;
static const unsigned int _bit_APD = 7;
static const unsigned int _bit_FN = 8;
static const unsigned int _bit_FP = 9;
static const unsigned int _bit_FD = 10;
static const unsigned int _bit_FPD = 11;
static const unsigned int _bit_FAN = 12;
static const unsigned int _bit_FAP = 13;
static const unsigned int _bit_FAD = 14;
static const unsigned int _bit_FAPD = 15;

static const unsigned int N = 1u << _bit_N;
static const unsigned int P = 1u << _bit_P;
static const unsigned int D = 1u << _bit_D;
static const unsigned int PD = 1u << _bit_PD;
static const unsigned int AN = 1u << _bit_AN;
static const unsigned int AP = 1u << _bit_AP;
static const unsigned int AD = 1u << _bit_AD;
static const unsigned int APD = 1u << _bit_APD;
static const unsigned int FN = 1u << _bit_FN;
static const unsigned int FP = 1u << _bit_FP;
static const unsigned int FD = 1u << _bit_FD;
static const unsigned int FPD = 1u << _bit_FPD;
static const unsigned int FAN = 1u << _bit_FAN;
static const unsigned int FAP = 1u << _bit_FAP;
static const unsigned int FAD = 1u << _bit_FAD;
static const unsigned int FAPD = 1u << _bit_FAPD;
}


#define HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX_DATA(bitMask, idPrefix, normalId, pressedId, disabledId, pressedDisabled, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedPressedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId, focusedActivePressedDisabledId) \
   ::hmibase::widget::appearance::AppearanceAssetIdData(HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_N, idPrefix, normalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_P, idPrefix, pressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_D, idPrefix, disabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_PD, idPrefix, pressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_AN, idPrefix, activeNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_AP, idPrefix, activePressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_AD, idPrefix, activeDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_APD, idPrefix, activePressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FN, idPrefix, focusedNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FP, idPrefix, focusedPressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FD, idPrefix, focusedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FPD, idPrefix, focusedPressedDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FAN, idPrefix, focusedActiveNormalId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FAP, idPrefix, focusedActivePressedId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FAD, idPrefix, focusedActiveDisabledId), \
                                                        HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX(bitMask, mask::_bit_FAPD, idPrefix, focusedActivePressedDisabledId))

#define HMIBASE_APPEARANCE_IMAGE_ASSET_ID_BITMASK_PREFIX(nodeName, bitMask, idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedPressedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId, focusedActivePressedDisabledId) \
   _HMIBASE_APPEARANCE_IMAGE_ASSET_ID(nodeName, HMIBASE_APPEARANCE_ASSET_ID_BITMASK_PREFIX_DATA(bitMask, idPrefix, normalId, pressedId, disabledId, pressedDisabledId, activeNormalId, activePressedId, activeDisabledId, activePressedDisabledId, focusedNormalId, focusedPressedId, focusedDisabledId, focusedPressedDisabledId, focusedActiveNormalId, focusedActivePressedId, focusedActiveDisabledId, focusedActivePressedDisabledId));


/**************************************************************************/
/* Button Color */
/**************************************************************************/
#define HMIBASE_APPEARANCE_COLOR(nodeName, colorData) \
   appContainer->add(nodeName, FEATSTD_NEW(::hmibase::widget::appearance::ColorAppearanceTemplate)(colorData));


/**************************************************************************/
/* Disabled */
#define _HMIBASE_APPEARANCE_NEW_COLOR_DATA_D(normalColor, disabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::create(normalColor, ::Candera::Color(), disabledColor)

#define HMIBASE_APPEARANCE_COLOR_DATA_D(colorData, normalColor, disabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::SharedPointer colorData(_HMIBASE_APPEARANCE_NEW_COLOR_DATA_D(normalColor, disabledColor));

#define HMIBASE_APPEARANCE_COLOR_D(nodeName, normalColor, disabledColor) \
   HMIBASE_APPEARANCE_COLOR(nodeName, _HMIBASE_APPEARANCE_NEW_COLOR_DATA_D(normalColor, disabledColor));


/**************************************************************************/
/* Disabled, Active */
#define _HMIBASE_APPEARANCE_NEW_COLOR_DATA_DA(normalColor, disabledColor, activeNormalColor, activeDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::create(normalColor, ::Candera::Color(), disabledColor, ::Candera::Color(), activeNormalColor, ::Candera::Color(), activeDisabledColor)

#define HMIBASE_APPEARANCE_COLOR_DATA_DA(buttonColorData, normalColor, disabledColor, activeNormalColor, activeDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::SharedPointer buttonColorData(_HMIBASE_APPEARANCE_NEW_COLOR_DATA_DA(normalColor, disabledColor, activeNormalColor, activeDisabledColor));

#define HMIBASE_APPEARANCE_COLOR_DA(nodeName, normalColor, disabledColor, activeNormalColor, activeDisabledColor) \
   HMIBASE_APPEARANCE_COLOR(nodeName, _HMIBASE_APPEARANCE_NEW_COLOR_DATA_DA(normalColor, disabledColor, activeNormalColor, activeDisabledColor));


/**************************************************************************/
/* Pressed */
#define _HMIBASE_APPEARANCE_NEW_COLOR_DATA_P(normalColor, pressedColor) \
   ::hmibase::widget::appearance::AppearanceColorData::create(normalColor, pressedColor)

#define HMIBASE_APPEARANCE_COLOR_DATA_P(colorData, normalColor, pressedColor) \
   ::hmibase::widget::appearance::AppearanceColorData::SharedPointer colorData(_HMIBASE_APPEARANCE_NEW_COLOR_DATA_P(normalColor, pressedColor));

#define HMIBASE_APPEARANCE_COLOR_P(nodeName, normalColor, pressedColor) \
   HMIBASE_APPEARANCE_COLOR(nodeName, _HMIBASE_APPEARANCE_NEW_COLOR_DATA_P(normalColor, pressedColor));


/**************************************************************************/
/* Pressed, Disabled */
#define _HMIBASE_APPEARANCE_NEW_COLOR_DATA_PD(normalColor, pressedColor, disabledColor, pressedDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::create(normalColor, pressedColor, disabledColor, pressedDisabledColor)

#define HMIBASE_APPEARANCE_COLOR_DATA_PD(colorData, normalColor, pressedColor, disabledColor, pressedDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::SharedPointer colorData(_HMIBASE_APPEARANCE_NEW_COLOR_DATA_PD(normalColor, pressedColor, disabledColor, pressedDisabledColor));

#define HMIBASE_APPEARANCE_COLOR_PD(nodeName, normalColor, pressedColor, disabledColor, pressedDisabledColor) \
   HMIBASE_APPEARANCE_COLOR(nodeName, _HMIBASE_APPEARANCE_NEW_COLOR_DATA_PD(normalColor, pressedColor, disabledColor, pressedDisabledColor));


/**************************************************************************/
/* Pressed, Disabled, Active */
#define _HMIBASE_APPEARANCE_NEW_COLOR_DATA_PDA(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::create(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor)

#define HMIBASE_APPEARANCE_COLOR_DATA_PDA(buttonColorData, normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::SharedPointer buttonColorData(_HMIBASE_APPEARANCE_NEW_COLOR_DATA_PDA(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor));

#define HMIBASE_APPEARANCE_COLOR_PDA(nodeName, normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor) \
   HMIBASE_APPEARANCE_COLOR(nodeName, _HMIBASE_APPEARANCE_NEW_COLOR_DATA_PDA(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor));


/**************************************************************************/
/* Pressed, Disabled, Active, Focused */
#define _HMIBASE_APPEARANCE_NEW_COLOR_DATA_PDAF(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor, focusedNormalColor, focusedPressedColor, focusedDisabledColor, focusedDisabledPressedColor, focusedActiveNormalColor, focusedActivePressedColor, focusedActiveDisabledColor, focusedActivePressedDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::create(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor, focusedNormalColor, focusedPressedColor, focusedDisabledColor, focusedDisabledPressedColor, focusedActiveNormalColor, focusedActivePressedColor, focusedActiveDisabledColor, focusedActivePressedDisabledColor)

#define HMIBASE_APPEARANCE_COLOR_DATA_PDAF(colorData, normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor, focusedNormalColor, focusedPressedColor, focusedDisabledColor, focusedDisabledPressedColor, focusedActiveNormalColor, focusedActivePressedColor, focusedActiveDisabledColor, focusedActivePressedDisabledColor) \
   ::hmibase::widget::appearance::AppearanceColorData::SharedPointer colorData(_HMIBASE_APPEARANCE_NEW_COLOR_DATA_PDAF(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor, focusedNormalColor, focusedPressedColor, focusedDisabledColor, focusedDisabledPressedColor, focusedActiveNormalColor, focusedActivePressedColor, focusedActiveDisabledColor, focusedActivePressedDisabledColor));

#define HMIBASE_APPEARANCE_COLOR_PDAF(nodeName, normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor, focusedNormalColor, focusedPressedColor, focusedDisabledColor, focusedDisabledPressedColor, focusedActiveNormalColor, focusedActivePressedColor, focusedActiveDisabledColor, focusedActivePressedDisabledColor) \
   HMIBASE_APPEARANCE_COLOR(nodeName, _HMIBASE_APPEARANCE_NEW_COLOR_DATA_PDAF(normalColor, pressedColor, disabledColor, pressedDisabledColor, activeNormalColor, activePressedColor, activeDisabledColor, activePressedDisabledColor, focusedNormalColor, focusedPressedColor, focusedDisabledColor, focusedDisabledPressedColor, focusedActiveNormalColor, focusedActivePressedColor, focusedActiveDisabledColor, focusedActivePressedDisabledColor));


/**************************************************************************/
/**************************************************************************/
}


}
}


#endif // hmibase_Appearance2D_h
