/* ***************************************************************************************
 * FILE:          Appearance2D.cpp
 * SW-COMPONENT:  HMI-BASE
 *  DESCRIPTION:  Appearance2D is part of HMI-Base Widget Library
 *    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
 *
 * The reproduction, distribution and utilization of this file as well as the
 * communication of its contents to others without express authorization is
 * prohibited. Offenders will be held liable for the payment of damages.
 * All rights reserved in the event of the grant of a patent, utility model or design.
 *
 *************************************************************************************** */

#include "widget2D_std_if.h"
#include "Appearance2D.h"

#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <Widgets/2D/ControlTemplate/TTraverserBase.h>
#include <Widgets/2D/EffectControl/ColorEffectWidget2D.h>
#include <Widgets/utils/ColorPropertyAccessor.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_APPEARANCE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Appearance2D.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace appearance {


/**************************************************************************/
Appearance::Appearance(AppearanceTemplate* tpl) : _template(tpl), _next(NULL)
{
}


/**************************************************************************/
Appearance::~Appearance()
{
   if (_next != NULL)
   {
      FEATSTD_DELETE(_next);
      _next = NULL;
   }

   if (_template != NULL)
   {
      _template->onAppearanceDeleted(this);
   }
}


/**************************************************************************/
Appearance::UpdateResult Appearance::update()
{
   Appearance::UpdateResult result(Appearance::NoChange);
   Appearance* app = this;
   while (app != NULL)
   {
      result = static_cast<Appearance::UpdateResult>(app->updateSelf() | result);
      app = app->_next;
   }
   return result;
}


/**************************************************************************/
AppearanceTemplateContainer::AppearanceTemplateContainer(bool idFilterEnabled) : _idFilterEnabled(idFilterEnabled)
{
}


/**************************************************************************/
AppearanceTemplateContainer::~AppearanceTemplateContainer()
{
   for (MapType::iterator it = _map.begin(); it != _map.end(); ++it)
   {
      Entry* entry = it->second;
      while (entry != NULL)
      {
         Entry* next = entry->_Next;
         FEATSTD_DELETE(entry);
         entry = next;
      }

      it->second = NULL;
   }
}


/**************************************************************************/
AppearanceTemplateContainer::Entry* AppearanceTemplateContainer::get(const IdType& id) const
{
   MapType::const_iterator it = _map.find(Courier::Identifier(id));
   if (it != _map.end())
   {
      return it->second;
   }
   return NULL;
}


/**************************************************************************/
void AppearanceTemplateContainer::add(const NodeAndPropertyId& nodeAndPropertyId, AppearanceTemplate* appTemplate)
{
   if (appTemplate != NULL)
   {
      Entry*& entry = _map[Courier::Identifier(nodeAndPropertyId._NodeId)];

      if (entry == NULL)
      {
         entry = FEATSTD_NEW(Entry)(appTemplate, nodeAndPropertyId._PropertyHint);
      }
      else
      {
         while (entry->_Next != NULL)
         {
            entry = entry->_Next;
         }
         entry->_Next = FEATSTD_NEW(Entry)(appTemplate, nodeAndPropertyId._PropertyHint);
      }
   }
}


/**************************************************************************/
void AppearanceTemplateContainer::clear(const IdType& id)
{
   MapType::iterator it = _map.find(Courier::Identifier(id));
   if (it != _map.end())
   {
      Entry* entry = it->second;
      while (entry != NULL)
      {
         Entry* next = entry->_Next;
         FEATSTD_DELETE(entry);
         entry = next;
      }

      it->second = NULL;

      _map.erase(it);
   }
}


/**************************************************************************/
class AppearanceNodeTraverser : public TTraverserBase<Candera::Node2D>
{
   public:
      /**************************************************************************/
      AppearanceNodeTraverser(Candera::Widget2D& owner, AppearanceTemplateContainer& appTemplateContainer)
         : _owner(owner), _appTemplateContainer(appTemplateContainer), _first(NULL), _current(NULL)
      {
      }

      /**************************************************************************/
      ~AppearanceNodeTraverser()
      {
         _first = NULL;
         _current = NULL;
      }

      /**************************************************************************/
      Appearance* getAppearance()
      {
         return _first;
      }

   protected:
      /**************************************************************************/
      virtual TraverserAction ProcessNode(Candera::Node2D& node)
      {
         AppearanceTemplateContainer::Entry* entry = NULL;

         //for widget's node check first an app template with empty name
         if (&node == _owner.GetNode())
         {
            entry = _appTemplateContainer.get(HMIBASE_APPEARANCE_NODE_NAME_WIDGET_NODE);
         }
         if ((entry == NULL) && (!_appTemplateContainer.isIdFilterEnabled() || _appTemplateContainer.matchesIdFilter(node.GetName())))
         {
            entry = _appTemplateContainer.get(node.GetName());
         }
         while (entry != NULL)
         {
            if (entry->_Template != NULL)
            {
               Appearance* app = entry->_Template->create(_owner, node, entry->_PropertyHint);
               if (app != NULL)
               {
                  if (_current != NULL)
                  {
                     _current->_next = app;
                     _current = app;
                  }
                  else
                  {
                     _current = app;
                     _first = app;
                  }
               }
            }
            entry = entry->_Next;
         }
         return ProceedTraversing;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(AppearanceNodeTraverser);

      Candera::Widget2D& _owner;
      AppearanceTemplateContainer& _appTemplateContainer;

      Appearance* _first;
      Appearance* _current;
};


/**************************************************************************/
void AppearanceTemplateContainer::enableIdFilter(bool value)
{
   _idFilterEnabled = value;
}


/**************************************************************************/
bool AppearanceTemplateContainer::isIdFilterEnabled() const
{
   return _idFilterEnabled;
}


/**************************************************************************/
bool AppearanceTemplateContainer::matchesIdFilter(const IdType& id) const
{
   return (id != NULL) && (id[0] == '_');
}


/**************************************************************************/
Appearance* AppearanceTemplateContainer::createAppearance(Candera::Widget2D& widget)
{
   if (widget.GetNode() == NULL)
   {
      return NULL;
   }

   AppearanceNodeTraverser traverser(widget, *this);
   traverser.Traverse(*widget.GetNode());
   return traverser.getAppearance();
}


/**************************************************************************/
template <> Candera::BitmapBrush* AppearanceHelper::getTarget<Candera::BitmapBrush>(Candera::Node2D& node, int /*propertyHint*/)
{
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(&node);
   if ((renderNode != NULL) && (renderNode->GetEffect(0) != NULL))
   {
      return Candera::Dynamic_Cast<Candera::BitmapBrush*>(renderNode->GetEffect(0)->GetBrushEffect2D());
   }

   return NULL;
}


/**************************************************************************/
template <> Candera::ColorProperty* AppearanceHelper::getTarget<Candera::ColorProperty>(Candera::Node2D& node, int propertyHint)
{
   return hmibase::widget::utils::EffectUtils::getColorProperty(&node, static_cast<hmibase::widget::utils::EffectUtils::colorid::ENUM>(propertyHint));
}


/**************************************************************************/
ImageWrapper::ImageWrapper(TId id) : _id(id), _initialized(false)
{
}


/**************************************************************************/
ImageWrapper::ImageWrapper(Candera::MemoryManagement::SharedPointer<Candera::Image2D> image) : _id(TId()), _image(image), _initialized(true)
{
}


/**************************************************************************/
void ImageWrapper::init()
{
   if (_image.PointsToNull() && !_initialized)
   {
      _initialized = true;
      if (_id != TId())
      {
         _image = Candera::DefaultAssetProvider::GetInstance().GetBitmapImage2DById(_id);
      }
   }
}


/**************************************************************************/
static Appearance::UpdateResult updateAppearance(BaseWidget2D& owner, Candera::BitmapBrush& target, Candera::MemoryManagement::SharedPointer<Candera::Image2D>& image)
{
   Candera::Image2D* oldImage = target.Image().Get();

   //if both bitmaps are NULL or they point to same image=> do nothing
   if (oldImage == image.GetPointerToSharedInstance())
   {
      return Appearance::NoChange;
   }

   if (image.PointsToNull())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "updateAppearance image=null widget=%s", HMIBASE_TO_STRING_VW(&owner)));
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "updateAppearance %s", HMIBASE_TO_STRING_VW(&owner)));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "updateAppearance image=%s", HMIBASE_TO_STRING(image.GetPointerToSharedInstance())));
   }
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "updateAppearance node=%s", HMIBASE_TO_STRING_SPN2D(owner.GetNode())));

   bool invalidateLayout = false;
   //if one of the bitmaps is NULL and the other one non NULL or if size is different=>Invalidate layout
   if ((oldImage == NULL)
         || image.PointsToNull()
         || (oldImage->GetWidth() != image->GetWidth())
         || (oldImage->GetHeight() != image->GetHeight()))
   {
      invalidateLayout = true;
   }

   target.Unload();
   target.Image().Set(image);
   target.Upload();

   return invalidateLayout ? Appearance::LayoutAndRenderRequired : Appearance::RenderRequired;
}


/**************************************************************************/
template <> Appearance::UpdateResult ImageAppearance::updateSelf()
{
   if (_data.PointsToNull())
   {
      return NoChange;
   }

   return updateAppearance(_owner, _target, _data->getData(_owner.GetAppearanceState()));
}


/**************************************************************************/
template <> Appearance::UpdateResult ImageWrapperAppearance::updateSelf()
{
   if (_data.PointsToNull())
   {
      return NoChange;
   }

   ImageWrapper& imageWrapper = _data->getData(_owner.GetAppearanceState());
   imageWrapper.init();
   return updateAppearance(_owner, _target, imageWrapper._image);
}


/**************************************************************************/
ImageWrapperAppearance* ImageIdAppearanceTemplate::createImpl(OwnerType& owner, TargetType& target)
{
   if (_data.PointsToNull())
   {
      _data = AppearanceImageWrapperData::create(_templateData.Normal, _templateData.Pressed, _templateData.Disabled, _templateData.PressedDisabled,
              _templateData.ActiveNormal, _templateData.ActivePressed, _templateData.ActiveDisabled, _templateData.ActivePressedDisabled,
              _templateData.FocusedNormal, _templateData.FocusedPressed, _templateData.FocusedDisabled, _templateData.FocusedPressedDisabled,
              _templateData.FocusedActiveNormal, _templateData.FocusedActivePressed, _templateData.FocusedActiveDisabled, _templateData.FocusedActivePressedDisabled);
   }

   return FEATSTD_NEW(ImageWrapperAppearance)(owner, target, _data, this);
}


/**************************************************************************/
void ImageIdAppearanceTemplate::onAppearanceDeleted(Appearance* /*app*/)
{
   if (!_data.PointsToNull() && FeatStd::MemoryManagement::Internal::IsSoleReference(_data))
   {
      _data.Release();
   }
}


/**************************************************************************/
template <> Appearance::UpdateResult ColorAppearance::updateSelf()
{
   if (_data.PointsToNull())
   {
      return NoChange;
   }

   const Candera::Color& color = _data->getData(_owner.GetAppearanceState());
   if (Candera::Color(_target.Get()) != color)
   {
      _target.Set(color.GetData());
      return RenderRequired;
   }

   return NoChange;
}


/**************************************************************************/
template <> Appearance::UpdateResult ColorAccessorAppearance::updateSelf()
{
   if (_data.PointsToNull())
   {
      return NoChange;
   }

   const Candera::Color& color = _data->getData(_owner.GetAppearanceState());
   if (_target.Get() != color)
   {
      _target.Set(color);
      return RenderRequired;
   }

   return NoChange;
}


///**************************************************************************/
//template <typename TRes>
//class ResourceArray
//{
//public:
//   typedef TRes ResType;
//
//   virtual ResType& get(size_t index) = 0;
//};
//
//
///**************************************************************************/
//class BitmapArray : public ResourceArray<Candera::MemoryManagement::SharedPointer<Candera::BitmapImage2D>>
//{
//public:
//   typedef const char* TName;
//
//   virtual ResType& get(size_t index)
//   {
//      if (index >= _array.size())
//      {
//         static ResType _invalid;
//         return _invalid;
//      }
//
//      Data& data = _array[index];
//      if (data._image.PointsToNull() && !data._checked)
//      {
//         data._checked = true;
//         if (data._name != TName())
//         {
//            //Candera::Id canderaId = Candera::Internal::AssetProviderFunctions::GetIdByName(&Candera::DefaultAssetProvider::GetInstance(), Candera::BitmapLib, data._id);
//            //data._image = Candera::DefaultAssetProvider::GetInstance().GetBitmapImage2DById(canderaId);
//            data._image = ImageLoader::getAssetBitmapImage(data._name);
//         }
//      }
//      return data._image;
//   }
//
//   BitmapArray& addId(TName name)
//   {
//      _array.push_back(Data(name));
//      return *this;
//   }
//
//protected:
//   struct Data
//   {
//      explicit Data(TName id = TName()) : _name(id), _checked(false)
//      {
//      }
//
//      TName _name;
//      ResType _image;
//      bool _checked;
//   };
//   typedef std::vector<Data> ArrayType;
//   ArrayType _array;
//};
//
}


}
}
