/* ***************************************************************************************
* FILE:          AnimationControlWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationControlWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "AnimationControlWidget2D.h"
#include <Courier/Visualization/ViewFacade.h>
#include <View/CGI/CgiExtensions/AppViewHandler.h>
#include <View/CGI/CgiExtensions/ArrayUtils.h>
#include <View/CGI/CgiExtensions/LayoutSizePropertySetter.h>
#include <CanderaAssetLoader/AssetLoaderBase/AnimationPropertySetterFactory.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_ANIMATION
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AnimationControlWidget2D.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace animation {

CGI_WIDGET_RTTI_DEFINITION(AnimationControlWidget2D);

using namespace Candera::Animation;

/*****************************************************************************/
AnimationControlWidget2D::AnimationControlWidget2D() : _invalid(true), _animationWrapper(AppViewHandler::getInstance())
{
}


/*****************************************************************************/
AnimationControlWidget2D::~AnimationControlWidget2D()
{
}


/*****************************************************************************/
bool AnimationControlWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const AnimationControlWidget2D* original = CLONEABLE_WIDGET_CAST<const AnimationControlWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetAnimationPlayerSource(original->GetAnimationPlayerSource());
      //intentionally commented
      //SetAnimation(controlTemplateMap.ResolveWidgetClone(original->GetAnimation()));

      //Start Mode, Should Run and Repeat Count
      SetStartMode(original->GetStartMode());
      SetShouldRun(original->GetShouldRun());
      SetRepeatCountEnabled(original->GetRepeatCountEnabled());
      SetRepeatCount(original->GetRepeatCount());
      //Repeat Mode, Speed Factor and Direction
      SetRepeatModeEnabled(original->GetRepeatModeEnabled());
      SetRepeatMode(original->GetRepeatMode());
      SetSpeedFactorEnabled(original->GetSpeedFactorEnabled());
      SetSpeedFactor(original->GetSpeedFactor());
      SetDirectionEnabled(original->GetDirectionEnabled());
      SetDirection(original->GetDirection());
      //Start Time and Duration
      SetStartTimeEnabled(original->GetStartTimeEnabled());
      SetStartTime(original->GetStartTime());
      SetDurationEnabled(original->GetDurationEnabled());
      SetDuration(original->GetDuration());
      //Animated Property Source
      SetAnimatedPropertyEnabled(original->GetAnimatedPropertyEnabled());
      //Property Setter
      SetProperty(original->GetProperty());
      SetWidget(controlTemplateMap.ResolveWidgetClone(original->GetWidget()));
      SetWidgetProperty(original->GetWidgetProperty());
      SetChannels(original->GetChannels());
      //Interpolation
      SetInterpolationStrategy(original->GetInterpolationStrategy());
      SetEaseParams(original->GetEaseParams());
      //Sequence Times
      SetKeyframeCount(original->GetKeyframeCount());
      if (!hmibase::util::Arrays::equals(GetSequenceTimes(), original->GetSequenceTimes()))
      {
         SetSequenceTimes(original->GetSequenceTimes());
      }
      //Keyframe Values
      SetCurrentFirstKeyframeEnabled(original->GetCurrentFirstKeyframeEnabled());
      if (!hmibase::util::Arrays::equals(GetKeyframeValues(), original->GetKeyframeValues()))
      {
         SetKeyframeValues(original->GetKeyframeValues());
      }

      cloned = true;
   }
   return cloned;
}


/*****************************************************************************/
void AnimationControlWidget2D::Finalize()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Finalize %s", HMIBASE_TO_STRING_VW(this)));

   if (_animationWrapper.getPlayer() != NULL)
   {
      _animationWrapper.stop();
      _animationWrapper.removeListener(this);
      _animatedPropertyWrapper.removeFromPlayer(_animationWrapper.getPlayer());
   }

   Base::Finalize();
}


/*****************************************************************************/
void AnimationControlWidget2D::Update()
{
   if (_animationWrapper.isEnabled())
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update isRunning %s", HMIBASE_TO_STRING_VW(this)));
   }

   bool prepared = false;
   if (_invalid && (GetNode() != NULL) && (GetParentView() != NULL))
   {
      prepared = prepareAnimation();
      _invalid = false;
   }

   if ((GetStartMode() == enStartMode::Property) && (GetShouldRun() != _animationWrapper.isEnabled()))
   {
      if (GetShouldRun())
      {
         if (prepared || prepareAnimation())
         {
            _animationWrapper.start(Courier::AnimationProperties(), false);
         }
      }
      else
      {
         _animationWrapper.stop();
      }
   }
}


/*****************************************************************************/
void AnimationControlWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnParentViewRenderingEnabled enable=%u %s",
                       enable, HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnParentViewActivate(bool activate)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnParentViewActivate activate=%u %s",
                       activate, HMIBASE_TO_STRING_VW(this)));

   if (activate)
   {
      prepareAnimation();
   }
}


/*****************************************************************************/
void AnimationControlWidget2D::OnParentViewLoad(bool load)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnParentViewLoad load=%u %s",
                       load, HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnChanged(Courier::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   _invalid = true;
}


/*****************************************************************************/
bool AnimationControlWidget2D::prepareAnimation()
{
   bool result = true;

   //detach from old player
   if (_animationWrapper.getPlayer() != NULL)
   {
      _animationWrapper.stop();
      _animationWrapper.removeListener(this);
      _animatedPropertyWrapper.removeFromPlayer(_animationWrapper.getPlayer());
   }

   if (isExternalAnimationPlayerUsed())
   {
      AnimationPlayer* player = dynamic_cast<AnimationPlayer*>(GetAnimation().GetPointerToSharedInstance());
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PrepareAnimation externalAnimation=[%p.%50s] %s",
                          player, player != NULL ? player->GetName() : "null",
                          HMIBASE_TO_STRING_VW(this)));

      _animationWrapper.setPlayer(AnimationPlayer::SharedPointer(player));
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PrepareAnimation internalAnimation %s", HMIBASE_TO_STRING_VW(this)));

      _animationWrapper.init();
   }

   if (_animationWrapper.getPlayer() != NULL)
   {
      _animationWrapper.addListener(this);

      Courier::AnimationProperties animationProperties(getAnimationProperties());

      if (isAnimatedPropertySectionVisible())
      {
         size_t channelCount(0);
         const size_t MAX_CHANNEL_COUNT(4);
         FeatStd::Float currentValue[MAX_CHANNEL_COUNT];

         AnimationPropertySetter::SharedPointer propertySetter = createPropertySetter(channelCount, currentValue);
         if (!propertySetter.PointsToNull())
         {
            if (channelCount > MAX_CHANNEL_COUNT)
            {
               channelCount = MAX_CHANNEL_COUNT;
            }

            size_t keyframeCount = GetKeyframeCount();

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PrepareAnimation kfCount=%u seqTimeCount=%u valuesCount=%u crtFirstKf=%u int=%u %s",
                                GetKeyframeCount(),
                                GetSequenceTimes().GetCount(),
                                GetKeyframeValues().GetCount(),
                                GetCurrentFirstKeyframeEnabled(),
                                GetInterpolationStrategy(),
                                HMIBASE_TO_STRING_VW(this)));

            if ((keyframeCount <= GetSequenceTimes().GetCount()) && (keyframeCount * channelCount <= GetKeyframeValues().GetCount()))
            {
               if (keyframeCount != _animatedPropertyWrapper.getKeyframeCount() || channelCount != _animatedPropertyWrapper.getComponentCount())
               {
                  Candera::Animation::InterpolationStrategy::SharedPointer interpolationStrategy;

                  switch (GetInterpolationStrategy())
                  {
                     default:
                     case enInterpolationStrategy::Linear:
                        interpolationStrategy = LinearInterpolationStrategy::Create();
                        break;
                     case enInterpolationStrategy::Step:
                        interpolationStrategy = StepInterpolationStrategy::Create();
                        break;
                     case enInterpolationStrategy::Spline:
                        interpolationStrategy = SplineInterpolationStrategy::Create();
                        break;
                  }

                  _animatedPropertyWrapper.init(keyframeCount, channelCount, LinearInterpolationStrategy::Create());
               }
               for (size_t keyframeIndex = 0; keyframeIndex < keyframeCount; ++keyframeIndex)
               {
                  _animatedPropertyWrapper.setSequenceTime(keyframeIndex, GetSequenceTimes().Get(static_cast<FeatStd::UInt>(keyframeIndex)));

                  if ((keyframeIndex > 0) || !GetCurrentFirstKeyframeEnabled())
                  {
                     for (size_t channelIndex = 0; channelIndex < channelCount; ++channelIndex)
                     {
                        _animatedPropertyWrapper.setKeyframeValue(keyframeIndex, channelIndex, GetKeyframeValues().Get(static_cast<FeatStd::UInt>(keyframeIndex * channelCount + channelIndex)));
                     }
                  }
               }

               if (GetCurrentFirstKeyframeEnabled())
               {
                  for (size_t channelIndex = 0; channelIndex < channelCount; ++channelIndex)
                  {
                     _animatedPropertyWrapper.setKeyframeValue(0, channelIndex, currentValue[channelIndex]);
                  }
               }

               _animatedPropertyWrapper.setPropertySetter(propertySetter);

               result = _animatedPropertyWrapper.addToPlayer(_animationWrapper.getPlayer());

               if (!GetDurationEnabled())
               {
                  animationProperties.SetSequenceDurationMs(_animationWrapper.getMaximumDuration());
               }
            }
         }
         else
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PrepareAnimation failed to create setter property=%u %s",
                                GetProperty(), HMIBASE_TO_STRING_VW(this)));
            result = false;
         }
      }

      _animationWrapper.applyProperties(animationProperties);
   }
   return result && (_animationWrapper.getPlayer() != NULL)
          && (_animationWrapper.getPlayer()->GetController() != NULL)
          && (_animationWrapper.getPlayer()->GetController()->GetNumberOfProperties() > 0);
}


/*****************************************************************************/
Courier::AnimationProperties AnimationControlWidget2D::getAnimationProperties()
{
   Courier::AnimationProperties props;

   if (isSpeedFactorPropertyVisible())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetAnimationProperties speedFactor=%f %s",
                          GetSpeedFactor(), HMIBASE_TO_STRING_VW(this)));

      props.SetSpeedFactor(GetSpeedFactor());
   }

   if (isShouldRunPropertyVisible())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetAnimationProperties repeatCount=0 %s", HMIBASE_TO_STRING_VW(this)));
      props.SetRepeatCount(0);
   }
   else if (isRepeatCountPropertyVisible())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetAnimationProperties repeatCount=%u %s",
                          GetRepeatCount(), HMIBASE_TO_STRING_VW(this)));

      props.SetRepeatCount(GetRepeatCount());
   }
   else
   {
      //nothing to do
   }

   if (isRepeatModePropertyVisible())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetAnimationProperties repeatMode=%10s %s",
                          (GetRepeatMode() == enRepeatMode::Bounce) ? "Bounce" : "Replay",
                          HMIBASE_TO_STRING_VW(this)));

      props.SetRepeatMode((GetRepeatMode() == enRepeatMode::Bounce) ? AnimationPlayer::Bounce : AnimationPlayer::Replay);
   }

   if (isDirectionPropertyVisible())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetAnimationProperties direction=%10s %s",
                          (GetDirection() == enDirection::Reverse) ? "Reverse" : "Forward",
                          HMIBASE_TO_STRING_VW(this)));

      props.SetDirection((GetDirection() == enDirection::Reverse) ? AnimationPlayer::Reverse : AnimationPlayer::Forward);
   }

   if (GetStartTimeEnabled())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetAnimationProperties startTime=%d %s",
                          GetStartTime(), HMIBASE_TO_STRING_VW(this)));

      props.SetSequenceStartTimeMs(GetStartTime());
   }

   if (GetDurationEnabled())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetAnimationProperties duration=%d %s",
                          GetDuration(), HMIBASE_TO_STRING_VW(this)));

      props.SetSequenceDurationMs(GetDuration());
   }

   return props;
}


/*****************************************************************************/
static AnimationPropertySetter::SharedPointer createPositionPropertySetter(Candera::Node2D& node, enChannels::Enum channels, size_t& channelCount, FeatStd::Float* currentValue)
{
   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_01:
         channelCount = 2;
         currentValue[0] = node.GetPosition().GetX();
         currentValue[1] = node.GetPosition().GetY();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Transformable2DTranslatePropertySetter>(&node);

      case enChannels::Ch_0:
         channelCount = 1;
         currentValue[0] = node.GetPosition().GetX();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Transformable2DTranslateXPropertySetter>(&node);

      case enChannels::Ch_1:
         channelCount = 1;
         currentValue[0] = node.GetPosition().GetY();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Transformable2DTranslateYPropertySetter>(&node);

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreatePositionPropertySetter Invalid Channels %u", channels));
         return AnimationPropertySetter::SharedPointer();
   }
}


/*****************************************************************************/
static AnimationPropertySetter::SharedPointer createRotationPropertySetter(Candera::Node2D& node, enChannels::Enum channels, size_t& channelCount, FeatStd::Float* currentValue)
{
   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_0:
         channelCount = 1;
         currentValue[0] = node.GetRotation();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Transformable2DRotatePropertySetter>(&node);

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreateRotationPropertySetter Invalid Channels %u", channels));
         return AnimationPropertySetter::SharedPointer();
   }
}


/*****************************************************************************/
static AnimationPropertySetter::SharedPointer createScalePropertySetter(Candera::Node2D& node, enChannels::Enum channels, size_t& channelCount, FeatStd::Float* currentValue)
{
   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_01:
         channelCount = 2;
         currentValue[0] = node.GetScale().GetX();
         currentValue[1] = node.GetScale().GetY();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Transformable2DScalePropertySetter>(&node);

      case enChannels::Ch_0:
         channelCount = 1;
         currentValue[0] = node.GetScale().GetX();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Transformable2DScaleXPropertySetter>(&node);

      case enChannels::Ch_1:
         channelCount = 1;
         currentValue[0] = node.GetScale().GetY();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Transformable2DScaleYPropertySetter>(&node);

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreateScalePropertySetter Invalid Channels %u", channels));
         return AnimationPropertySetter::SharedPointer();
   }
}


/*****************************************************************************/
static AnimationPropertySetter::SharedPointer createAlphaValuePropertySetter(Candera::Node2D& node, enChannels::Enum channels, size_t& channelCount, FeatStd::Float* currentValue)
{
   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_0:
         channelCount = 1;
         currentValue[0] = node.GetAlphaValue();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<AlphaNode2DPropertySetter>(&node);

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreateAlphaValuePropertySetter Invalid Channels %u", channels));
         return AnimationPropertySetter::SharedPointer();
   }
}


/*****************************************************************************/
static AnimationPropertySetter::SharedPointer createVisiblePropertySetter(Candera::Node2D& node, enChannels::Enum channels, size_t& channelCount, FeatStd::Float* currentValue)
{
   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_0:
         channelCount = 1;
         currentValue[0] = node.GetAlphaValue();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<RenderingEnabledNode2DPropertySetter>(&node);

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreateVisiblePropertySetter Invalid Channels %u", channels));
         return AnimationPropertySetter::SharedPointer();
   }
}


/*****************************************************************************/
static AnimationPropertySetter::SharedPointer createMarginPropertySetter(Candera::Node2D& node, enChannels::Enum channels, size_t& channelCount, FeatStd::Float* currentValue)
{
   Candera::Margin margin(Candera::Layouter::GetMargin(node));

   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_0123:
         channelCount = 4;
         currentValue[0] = margin.GetLeft();
         currentValue[1] = margin.GetTop();
         currentValue[2] = margin.GetRight();
         currentValue[3] = margin.GetBottom();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<LayoutMarginsPropertySetter>(&node);

      case enChannels::Ch_0:
         channelCount = 1;
         currentValue[0] = margin.GetLeft();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<LayoutMarginsLeftPropertySetter>(&node);

      case enChannels::Ch_1:
         channelCount = 1;
         currentValue[0] = margin.GetTop();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<LayoutMarginsTopPropertySetter>(&node);

      case enChannels::Ch_2:
         channelCount = 1;
         currentValue[0] = margin.GetRight();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<LayoutMarginsRightPropertySetter>(&node);

      case enChannels::Ch_3:
         channelCount = 1;
         currentValue[0] = margin.GetBottom();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<LayoutMarginsBottomPropertySetter>(&node);

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreateMarginPropertySetter Invalid Channels %u", channels));
         return AnimationPropertySetter::SharedPointer();
   }
}


/*****************************************************************************/
static AnimationPropertySetter::SharedPointer createSizePropertySetter(Candera::Node2D& node, enChannels::Enum channels, size_t& channelCount, FeatStd::Float* currentValue)
{
   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_01:
         channelCount = 2;
         currentValue[0] = Candera::Layouter::GetSize(node).GetX();
         currentValue[1] = Candera::Layouter::GetSize(node).GetY();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<hmibase::view::animation::LayoutSizePropertySetter>(&node);

      case enChannels::Ch_0:
         channelCount = 1;
         currentValue[0] = Candera::Layouter::GetSize(node).GetX();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<hmibase::view::animation::LayoutSizeXPropertySetter>(&node);

      case enChannels::Ch_1:
         channelCount = 1;
         currentValue[0] = Candera::Layouter::GetSize(node).GetY();
         return ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<hmibase::view::animation::LayoutSizeYPropertySetter>(&node);

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreateSizePropertySetter Invalid Channels %u", channels));
         return AnimationPropertySetter::SharedPointer();
   }
}


/*****************************************************************************/
static void getWidgetChannelList(enChannels::Enum channels, size_t& channelCount, FeatStd::Internal::Vector<Candera::Internal::AnimatedPropertyChannels>& apcList)
{
   // for widget properties the actual values are quite flexible: X, Red, Left and a few others are used for 1st channel; Y, Green, Top for the 2nd channel; etc.
   switch (channels)
   {
      case enChannels::Default:
      case enChannels::Ch_0:
         channelCount = 1;
         apcList.Add(Candera::Internal::apcDefault);
         break;

      case enChannels::Ch_1:
         channelCount = 1;
         apcList.Add(Candera::Internal::apcY);
         break;

      case enChannels::Ch_2:
         channelCount = 1;
         apcList.Add(Candera::Internal::apcZ);
         break;

      case enChannels::Ch_3:
         channelCount = 1;
         apcList.Add(Candera::Internal::apcW);
         break;

      case enChannels::Ch_01:
         channelCount = 2;
         apcList.Add(Candera::Internal::apcX);
         apcList.Add(Candera::Internal::apcY);
         break;

      case enChannels::Ch_012:
         channelCount = 3;
         apcList.Add(Candera::Internal::apcX);
         apcList.Add(Candera::Internal::apcY);
         apcList.Add(Candera::Internal::apcZ);
         break;

      case enChannels::Ch_0123:
         channelCount = 4;
         apcList.Add(Candera::Internal::apcX);
         apcList.Add(Candera::Internal::apcY);
         apcList.Add(Candera::Internal::apcZ);
         apcList.Add(Candera::Internal::apcW);
         break;

      case enChannels::Ch_12:
         channelCount = 2;
         apcList.Add(Candera::Internal::apcY);
         apcList.Add(Candera::Internal::apcZ);
         break;

      case enChannels::Ch_123:
         channelCount = 3;
         apcList.Add(Candera::Internal::apcY);
         apcList.Add(Candera::Internal::apcZ);
         apcList.Add(Candera::Internal::apcW);
         break;

      case enChannels::Ch_23:
         channelCount = 2;
         apcList.Add(Candera::Internal::apcZ);
         apcList.Add(Candera::Internal::apcW);
         break;

      default:
         break;
   }
}


/*****************************************************************************/
AnimationPropertySetter::SharedPointer AnimationControlWidget2D::createPropertySetter(size_t& channelCount, FeatStd::Float* currentValue)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "CreatePropertySetter property=%u %s",
                       GetProperty(), HMIBASE_TO_STRING_VW(this)));

   switch (GetProperty())
   {
      case enProperty::Position:
         return createPositionPropertySetter(*GetNode(), GetChannels(), channelCount, currentValue);

      case enProperty::Rotation:
         return createRotationPropertySetter(*GetNode(), GetChannels(), channelCount, currentValue);

      case enProperty::Scale:
         return createScalePropertySetter(*GetNode(), GetChannels(), channelCount, currentValue);

      case enProperty::AlphaValue:
         return createAlphaValuePropertySetter(*GetNode(), GetChannels(), channelCount, currentValue);

      case enProperty::Visible:
         return createVisiblePropertySetter(*GetNode(), GetChannels(), channelCount, currentValue);

      case enProperty::Size:
         return createSizePropertySetter(*GetNode(), GetChannels(), channelCount, currentValue);
         break;

      case enProperty::Margin:
         return createMarginPropertySetter(*GetNode(), GetChannels(), channelCount, currentValue);

      case enProperty::WidgetProperty:
      {
         Candera::WidgetBase* widget = GetWidget();
         if (widget != NULL)
         {
            FeatStd::Internal::Vector<Candera::Internal::AnimatedPropertyChannels> apcList;
            getWidgetChannelList(GetChannels(), channelCount, apcList);
            currentValue[0] = currentValue[1] = currentValue[2] = currentValue[3] = 0.0f;

            AnimationPropertySetter::SharedPointer propertySetter;
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetWidgetProperty());
            propertySetter = Candera::Internal::AnimationPropertySetterFactory::CreatePropertyMetaInfoSetter(widget->GetMetaInfo(), widget, GetWidgetProperty().GetCString(), apcList);
            if (propertySetter.PointsToNull())
            {
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreatePropertySetter Failed to create widget property setter (%50s.%10s) %s",
                                  HMIBASE_TO_STRING_W(widget),
                                  GetWidgetProperty().GetCString(),
                                  HMIBASE_TO_STRING_VW(this)));
            }
            SECURE_FEATSTD_STRING_ACCESS_END();
            return propertySetter;
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreatePropertySetter No widget to animate %s", HMIBASE_TO_STRING_VW(this)));
         }
         break;
      }

      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "CreatePropertySetter Invalid property %u %s",
                            GetProperty(), HMIBASE_TO_STRING_VW(this)));
         break;
   }

   return AnimationPropertySetter::SharedPointer();
}


/*****************************************************************************/
void AnimationControlWidget2D::OnPastEnd(AnimationPlayerBase*, FeatStd::Int32 completedIterationsCount)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnParentViewRenderingEnabled iterations=%d %s",
                       completedIterationsCount, HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnStartAnimation(AnimationPlayerBase*)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnStartAnimation %s", HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnStopAnimation(AnimationPlayerBase*)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnStopAnimation %s", HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnFinishAnimation(AnimationPlayerBase*)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnFinishAnimation %s", HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnResumeAnimation(AnimationPlayerBase*)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnResumeAnimation %s", HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnPauseAnimation(AnimationPlayerBase*)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnPauseAnimation %s", HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void AnimationControlWidget2D::OnDirectionChange(AnimationPlayerBase*, FeatStd::Int direction)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnDirectionChange direction=%u %s", direction, HMIBASE_TO_STRING_VW(this)));
}


}
}


}
