/* ***************************************************************************************
* FILE:          AnimationWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/Animation/AnimationWidget2D.h"
#include "CanderaAssetLoader/AssetLoaderBase/AnimationPropertySetterFactory.h"
#include "hmibase/util/StringUtils.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateBinding.h"
#include "AnimationBaseHelper.h"

using namespace Candera;
using namespace Courier;
using namespace Candera::Internal;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_ANIMATION
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AnimationWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(AnimationWidget2D)

// ------------------------------------------------------------------------
AnimationWidget2D::AnimationWidget2D():
   _node(NULL),
   _animationTimeDispatcher(NULL),
   _animationController(NULL),
   _animationKfsTime(NULL),
   _animationDuration(1000),
   _animationKfValues(NULL),
   _componentCount(0),
   _delimeterCount(0),
   _invalidInputData(false),
   _animationPlayer(NULL),
   _shouldRun(false),
   _shouldStop(false)
{
   _animationKeyframeSequence = Candera::Animation::AnimationKeyframeSequence::Create();
   _animationBlendedProperty = Candera::Animation::AnimationBlendedProperty::Create();
   _animationController = Candera::Animation::AnimationController::Create();
   _animationPlayer = Candera::Animation::AnimationPlayer::Create();
}


// ------------------------------------------------------------------------
AnimationWidget2D::~AnimationWidget2D()
{
   //_node = NULL;

   if (NULL != _animationKfsTime)
   {
      FEATSTD_SAFE_DELETE_ARRAY(_animationKfsTime);
   }

   if (!_animationTimeDispatcher.PointsToNull())

   {
      if (!_animationPlayer.PointsToNull())
      {
         AnimationBaseHelper::removePlayerFromAnimationTimeDispatcher(_animationPlayer, _animationTimeDispatcher);
         removeAnimationListener(this);
         _animationPlayer.Release();
      }
   }

   if (_animationKfValues != NULL)
   {
      FEATSTD_SAFE_DELETE_ARRAY(_animationKfValues);
   }

   _animationKfsTime         = NULL;
   _animationKfValues        = NULL;

   _invalidInputData = false;
}   //lint !e1579 _node freed in removeAnimationListener;


// ------------------------------------------------------------------------
void AnimationWidget2D::InitWidget()
{
   Base::InitWidget();
   _node = GetNode();
   initAnimation();
   assignInterpolationStategy();
}


// ------------------------------------------------------------------------
void AnimationWidget2D::Update()
{
   Base::Update();

   if (_shouldRun)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationWidget2D : Update startAnimation for [%s]", GetLegacyName()));
      (void)startAnimation();
      _shouldRun = false;
   }
   if (_shouldStop)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationWidget2D : Update stopAnimation for [%s]", GetLegacyName()));
      (void)stopAnimation();
      _shouldStop = false;
   }

   if (!_animationPlayer.PointsToNull() && _animationPlayer->IsEnabled())
   {
      Invalidate(true); // force the invalidation, required to update the Rendering Animation
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::initAnimation()
{
   if (initAnimationProperties())
   {
      assignKeyFrameTimeSequence();
      assignKeyFrameValues();
      assignAnimationKfSequence();
      initAnimationController();
      initAnimationPlayer();
   }
}


// ------------------------------------------------------------------------
bool AnimationWidget2D::OnMessage(const Courier::Message& msg)
{
   bool bIsMsgConsumed = Base::OnMessage(msg);
   if (bIsMsgConsumed)
   {
      return true;
   }

   if (msg.GetId() == AnimationWidgetReqMsg::ID)
   {
      const AnimationWidgetReqMsg* animationReqMsg = message_cast<const AnimationWidgetReqMsg*>(&msg);

      if ((animationReqMsg != NULL) &&
            (Courier::Identifier(GetLegacyName()) == animationReqMsg->GetSender()) &&
            (GetParentView()->GetId() == animationReqMsg->GetView()))
      {
         switch (animationReqMsg->GetAnimationAction())
         {
            case AnimationAction ::Start:
               _shouldRun = true;
               bIsMsgConsumed = true;
               break;

            case AnimationAction ::Stop:
               _shouldStop = true;
               bIsMsgConsumed = true;
               break;

            default:
               break;
         }
      } // ![switch case]
   } // valid message

   return bIsMsgConsumed;
}


// ------------------------------------------------------------------------
void AnimationWidget2D::OnAnimationTimeDispatcherChanged()
{
   _animationTimeDispatcher = GetAnimationTimeDispatcher();

   if ((!_animationTimeDispatcher.PointsToNull()) && (!_animationPlayer.PointsToNull()))

   {
      _animationTimeDispatcher->AddPlayer(_animationPlayer);
      removeAnimationListener(this);
      setAnimationListener(this);
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::assignKeyFrameTimeSequence()
{
   if (GetSequenceTime().GetCharCount() == 0)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] SequenceTime input data found to be NULL", GetLegacyName()));
      _invalidInputData = true;
   }
   else
   {
      if (_animationKfsTime != NULL)
      {
         FEATSTD_SAFE_DELETE_ARRAY(_animationKfsTime);
      }
      _animationKfsTime = FEATSTD_NEW_ARRAY(Candera::Animation::SequenceTimeType, GetKeyFrameCount());
      memset(_animationKfsTime, 0, sizeof(Candera::Animation::SequenceTimeType) * GetKeyFrameCount());
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetSequenceTime());
      _delimeterCount = extractAnimationKeyFrames<Candera::Animation::SequenceTimeType>(GetSequenceTime().GetCString(), _animationKfsTime);
      SECURE_FEATSTD_STRING_ACCESS_END();
      _animationDuration = _animationKfsTime[GetKeyFrameCount() - 1];

      // set the Animation duration based on the values calculated
      AnimationBaseHelper::setAnimationSequenceDuration(_animationPlayer, _animationDuration);

      if ((_delimeterCount + 1) != GetKeyFrameCount())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] KeyFrameCount doesnot match the KeyFrameTimeSequence", GetLegacyName()));
      }
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::assignKeyFrameValues()
{
   switch (GetPropertyAnimation())
   {
      case AnimateAlphaValue:
      case AnimateRotation:
      case AnimateRendering:
      case AnimateWidgetProperty:
      {
         if (GetChannel1().GetCharCount() == 0)
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] KeyFrameValue Channel1 input data found to be NULL", GetLegacyName()));
            _invalidInputData = true;
         }
         else
         {
            if (_animationKfValues != NULL)
            {
               FEATSTD_SAFE_DELETE_ARRAY(_animationKfValues);
            }

            _animationKfValues = FEATSTD_NEW_ARRAY(Candera::Float, GetKeyFrameCount());
            memset(_animationKfValues, 0, sizeof(Candera::Float) * GetKeyFrameCount());
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetChannel1());
            _delimeterCount = extractAnimationKeyFrames<Candera::Float>(GetChannel1().GetCString(), _animationKfValues);
            SECURE_FEATSTD_STRING_ACCESS_END();
            _componentCount = 1;
         }
      }
      break;

      case AnimatePosition:
      case AnimateScale:
      {
         if (GetChannel1().GetCharCount() == 0 || GetChannel2().GetCharCount() == 0)
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Channel %d ", (GetChannel1().GetCharCount() == 0) ? 1 : 2));
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "KeyFrameValue input data found to be NULL [%s] ", GetLegacyName()));
            _invalidInputData = true;
         }
         else
         {
            if (_animationKfValues != NULL)
            {
               FEATSTD_SAFE_DELETE_ARRAY(_animationKfValues);
            }
            _animationKfValues = FEATSTD_NEW_ARRAY(Candera::Float, 2 * GetKeyFrameCount());

            if (_animationKfValues != 0)
            {
               memset(_animationKfValues, 0, sizeof(Candera::Float) * 2 * GetKeyFrameCount());
               Candera::Float* temp = FEATSTD_NEW_ARRAY(Candera::Float, 2 * GetKeyFrameCount());

               if (temp != 0)
               {
                  memset(temp, 0, sizeof(Candera::Float) * 2 * GetKeyFrameCount());
                  SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetChannel1());
                  (void)extractAnimationKeyFrames<Candera::Float>(GetChannel1().GetCString(), temp);
                  SECURE_FEATSTD_STRING_ACCESS_END();
                  SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetChannel2());
                  _delimeterCount = extractAnimationKeyFrames<Candera::Float>(GetChannel2().GetCString(), temp + GetKeyFrameCount());
                  SECURE_FEATSTD_STRING_ACCESS_END();
                  for (UInt32 i = 0; i < GetKeyFrameCount(); i++)
                  {
                     _animationKfValues[i * 2] = temp[i];
                     _animationKfValues[(i * 2) + 1] = temp[i + GetKeyFrameCount()];
                  }

                  FEATSTD_SAFE_DELETE_ARRAY(temp);
               }
               _componentCount = 2;  //Scale X,Y
            }
         }
      }
      break;

      default:
         break;
   }

   if ((_delimeterCount + 1) != GetKeyFrameCount())
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] KeyFrameCount doesnot match the KeyFrameValue", GetLegacyName()));
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::initAnimationPlayer()
{
   if ((!_animationPlayer.PointsToNull()) && (_animationController != NULL))
   {
      AnimationBaseHelper::setAnimationController(_animationPlayer, _animationController);
      AnimationBaseHelper::setupAnimationPlayer(
         _animationPlayer,
         Candera::Animation::AnimationPlayer::PlayDirection(_animationPlayer->GetDirection()),
         GetRepeatTimes(),
         GetSpeedFactor(),
         _animationDuration
      );

      if (GetToggleDirection())
      {
         _animationPlayer->SetRepeatMode(Candera::Animation::AnimationPlayer::Bounce); ///< Reverse the animation direction with each repeat.
      }
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::initAnimationController()
{
   if ((_animationController != NULL) && (_animationBlendedProperty != NULL) && !_invalidInputData)
   {
      _animationController->AddProperty(_animationBlendedProperty);
      _animationController->SetSequenceReferenceTime(0);
   }
}


// ------------------------------------------------------------------------
bool AnimationWidget2D::initAnimationProperties()
{
   if (_animationBlendedProperty != NULL)
   {
      if (!AnimationBaseHelper::createAnimationPropertySetter(_animationBlendedProperty, _node, static_cast<enAnimationProperty>(GetPropertyAnimation())))
      {
         if (GetPropertyAnimation() == AnimateWidgetProperty)
         {
            Candera::WidgetBase* widget = GetWidgetToAnimate();
            if (widget == 0)
            {
               _invalidInputData = true;
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] Widget To Animate found to be NULL", GetLegacyName()));
            }
            else
            {
               FeatStd::Internal::Vector<AnimatedPropertyChannels> apcList;
               apcList.Add(static_cast<AnimatedPropertyChannels>(apcDefault));
               Candera::Animation::AnimationPropertySetter::SharedPointer propertysetterForWidget;
               SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetWidgetPropertyNameToAnimate());
               propertysetterForWidget = AnimationPropertySetterFactory::CreatePropertyMetaInfoSetter(widget->GetMetaInfo(), widget, GetWidgetPropertyNameToAnimate().GetCString(), apcList);
               SECURE_FEATSTD_STRING_ACCESS_END();
               if (propertysetterForWidget.PointsToNull())
               {
                  ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] AnimationPropertySetter found to be NULL", GetLegacyName()));
                  _invalidInputData = true;
               }
               else
               {
                  _animationBlendedProperty->SetAnimationPropertySetter(propertysetterForWidget);
               }
            }
         }
      } // ![End of Switch case]

      (void)AnimationBaseHelper::addKeyframeSequence(_animationBlendedProperty, _animationKeyframeSequence);
   }

   return ! _invalidInputData;
}


// ------------------------------------------------------------------------
void AnimationWidget2D::assignAnimationKfSequence()
{
   if (_animationKeyframeSequence != NULL && !_invalidInputData)
   {
      _animationKeyframeSequence->SetKeyframes(_componentCount, (Int32)GetKeyFrameCount(), _animationKfsTime, 0, _animationKfValues, 0);
   }
}


// ------------------------------------------------------------------------


void AnimationWidget2D::OnPastEnd(Animation::AnimationPlayerBase* animationPlayer, Candera::Int32 completedIterationsCount)
{
   FEATSTD_UNUSED(animationPlayer);
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Animation completedIterationCount:='%d' for [%s]", completedIterationsCount, GetLegacyName()));
   postAnimationIndMsg(completedIterationsCount);

   Invalidate();
}


// ------------------------------------------------------------------------
void AnimationWidget2D::setAnimationListener(Candera::Animation::AnimationPlayerListener* listener)
{
   if (AnimationBaseHelper::addListenerToPlayer(_animationPlayer, listener))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationPlayerListener[%p] added to AnimationPlayer[%p]", listener, _animationPlayer.GetPointerToSharedInstance()));
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "AnimationPlayerListener not added to AnimationPlayer"));
   }
}


// ------------------------------------------------------------------------
bool AnimationWidget2D::removeAnimationListener(Candera::Animation::AnimationPlayerListener* listener)
{
   if (AnimationBaseHelper::removeListenerFromPlayer(_animationPlayer, listener))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationPlayerListener[%p] removed from AnimationPlayer[%p]", listener, _animationPlayer.GetPointerToSharedInstance()));
      return true;
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationPlayerListener not removed from AnimationPlayer"));
      return false;
   }
}


// ------------------------------------------------------------------------
bool AnimationWidget2D::isAnimationEnabled()
{
   return (!_animationPlayer.PointsToNull()) ? _animationPlayer->IsEnabled() : false;
}


// ------------------------------------------------------------------------
bool AnimationWidget2D::startAnimation()
{
   bool retVal = false;

   if (!_animationPlayer.PointsToNull())
   {
      if (_invalidInputData)
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] Animation not started due to invalid input data", GetLegacyName()));
      }
      else
      {
         if (_animationPlayer->IsEnabled())
         {
            _animationPlayer->Stop();
         }

         retVal = _animationPlayer->Start();
      }
   }

   if (retVal)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Animation started successfully with Direction %d for [%s] ", GetDirection(), GetLegacyName()));
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] Animation failed to Start", GetLegacyName()));
   }
   return retVal;
}


// ------------------------------------------------------------------------
bool AnimationWidget2D::stopAnimation()
{
   bool retValue = false;

   if (!_animationPlayer.PointsToNull())
   {
      if (_animationPlayer->IsEnabled())
      {
         retValue = _animationPlayer->Stop();

         if (retValue)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "[%s] Animation stopped successfully", GetLegacyName()));
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "[%s] Animation failed to Stop", GetLegacyName()));
         }
      }
   }

   return retValue;
}


// ------------------------------------------------------------------------
void AnimationWidget2D::OnChanged(UInt32 propertyId)
{
   switch (propertyId)
   {
      //this widget has notifierPolicy=onUpdate which means that changing its specific animation related properties will trigger Update so OnChanged will be called from Update instead from the setter
      //for the other properties like Visible, Enabled => OnChanged is called from the setter
      case VisiblePropertyId:
      case EnabledPropertyId:
         Base::OnChanged(propertyId);
         break;

      case SequenceTimePropertyId:
         assignKeyFrameTimeSequence();
         assignAnimationKfSequence();
         break;

      case Channel1PropertyId:
      case Channel2PropertyId:
         assignKeyFrameValues();
         assignAnimationKfSequence();
         break;

      case ShouldRunPropertyId:
         if (GetShouldRun())
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationWidget2D : OnChanged ShouldRun set to true for [%s]", GetLegacyName()));
            _shouldRun = true;
         }
         else
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationWidget2D : OnChanged ShouldRun set to false for [%s]", GetLegacyName()));
            _shouldStop = true;
         }
         break;

      case DirectionPropertyId:
         if (!_animationPlayer.PointsToNull() && (!_animationPlayer->IsEnabled()))
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationWidget2D : OnChanged Direction %d for [%s]", GetDirection(), GetLegacyName()));
            _animationPlayer->SetDirection((Candera::Animation::AnimationPlayer::PlayDirection)GetDirection());
         }
         break;

      default:
         break;
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::assignInterpolationStategy()
{
   switch (GetInterpolationStrategy())
   {
      case InterpolationStrategyLinear:
         AnimationBaseHelper::setInterpolationStrategy(AnimationBaseHelper::createInterpolationStrategy<Candera::Animation::LinearInterpolationStrategy>(), _animationKeyframeSequence);
         break;

      case InterpolationStrategySpline:
         AnimationBaseHelper::setInterpolationStrategy(AnimationBaseHelper::createInterpolationStrategy<Candera::Animation::SplineInterpolationStrategy>(), _animationKeyframeSequence);
         break;

      case InterpolationStrategyStep:
         AnimationBaseHelper::setInterpolationStrategy(AnimationBaseHelper::createInterpolationStrategy<Candera::Animation::StepInterpolationStrategy>(), _animationKeyframeSequence);
         break;

      case InterpolationStrategyEase:
      {
         Candera::Animation::EaseInterpolationStrategy::SharedPointer easeStategy = Animation::EaseInterpolationStrategy::Create();
         configureEaseInterpolation(easeStategy);
         AnimationBaseHelper::setInterpolationStrategy(easeStategy, _animationKeyframeSequence);
      }
      break;

      default:
         AnimationBaseHelper::setInterpolationStrategy(AnimationBaseHelper::createInterpolationStrategy<Candera::Animation::LinearInterpolationStrategy>(), _animationKeyframeSequence);
         break;
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   if (enable)
   {
      if (GetShouldRun())
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationWidget2D : start Animation OnParentViewRenderingEnabled for [%s]", GetLegacyName()));
         startAnimation();
      }
   }
   Base::OnParentViewRenderingEnabled(enable);
}


// ------------------------------------------------------------------------
void AnimationWidget2D::OnParentViewActivate(bool activate)
{
   if (activate == false)
   {
      stopAnimation();  // Stop the animation if playing
      if (!_animationPlayer.PointsToNull())
      {
         _animationPlayer->SetDirection((Candera::Animation::AnimationPlayer::PlayDirection)GetDirection());  // Reset the original Animation play direction
      }
   }
   Base::OnParentViewActivate(activate);
}


// ------------------------------------------------------------------------
void AnimationWidget2D::OnNodeChanged()
{
   if (GetNode() != NULL && _animationBlendedProperty != NULL)
   {
      _node = GetNode();
      AnimationBaseHelper::assignNodeToAnimate(_animationBlendedProperty, _node);
   }
}


// ------------------------------------------------------------------------
void AnimationWidget2D::postAnimationIndMsg(Candera::Int32 cnt)
{
   Courier::Message* msg = COURIER_MESSAGE_NEW(AnimationWidgetIndMsg)(
                              GetParentView()->GetId(),
                              Courier::Identifier(GetLegacyName()),
                              cnt
                           );

   if (msg != 0)
   {
      msg->Post();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationWidgetIndMsg posted for %s", GetLegacyName()));
   }
}


// ------------------------------------------------------------------------
template <class TAnimationKeyFrame>
Candera::UInt32 AnimationWidget2D::extractAnimationKeyFrames(Candera::String stringToDecode, TAnimationKeyFrame* Tdata)
{
   if (Tdata != 0)
   {
      std::vector<std::string> splitData;
      splitData.clear();
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(stringToDecode);
      splitData = ::hmibase::util::split(stringToDecode.GetCString(), ';');
      SECURE_FEATSTD_STRING_ACCESS_END();
      Candera::UInt32 i = 0;
      std::stringstream data;

      while (i < GetKeyFrameCount() && i < splitData.size())
      {
         data << splitData[i].c_str();
         data >> Tdata[i];
         data.clear();
         i++;
      }

      return static_cast<Candera::UInt32>(splitData.size() > 0 ? splitData.size() - 1 : 0);
   }
   return 0;
}


// ------------------------------------------------------------------------
void AnimationWidget2D::configureEaseInterpolation(Candera::Animation::EaseInterpolationStrategy::SharedPointer strategy)
{
   if (! strategy.PointsToNull())
   {
      strategy->SetEaseDirection(static_cast<Animation::EaseInterpolationStrategy::EaseDirection>(GetEaseDirection()));

      Animation::AbstractEasingFunction::SharedPointer easeFunction;

      switch (GetFunctionType())
      {
         case EaseBack:
         {
            Animation::BackEaseFunction::SharedPointer backFunction = Animation::BackEaseFunction::Create();
            if (backFunction != 0)
            {
               backFunction->SetAmplitude(GetAmplitude());
               easeFunction = backFunction;
            }
         }
         break;

         case EaseBounce:
         {
            Animation::BounceEaseFunction::SharedPointer bounceFunction = Animation::BounceEaseFunction::Create();
            if (bounceFunction != 0)
            {
               bounceFunction->SetBounceCount(GetBounceCount());
               bounceFunction->SetRestitutionCoefficient(GetRestitutionCoefficient());
               easeFunction = bounceFunction;
            }
         }

         break;

         case EaseElastic:
         {
            Animation::ElasticEaseFunction::SharedPointer elasticFunction = Animation::ElasticEaseFunction::Create();
            if (elasticFunction != 0)
            {
               elasticFunction->SetOscillationCount(GetOscillationCount());
               elasticFunction->SetExponent(GetExponentElastic());
            }
            easeFunction = elasticFunction;
         }
         break;

         case EaseExponential:
         {
            Animation::ExponentialEaseFunction::SharedPointer exponentialFunction = Animation::ExponentialEaseFunction::Create();
            if (exponentialFunction != 0)
            {
               exponentialFunction->SetExponent(GetExponent());
            }
            easeFunction = exponentialFunction;
         }
         break;

         case EasePower:
         {
            Animation::PowerEaseFunction::SharedPointer powerFunction = Animation::PowerEaseFunction::Create();
            if (powerFunction != 0)
            {
               powerFunction->SetPower(GetPower());
            }
            easeFunction = powerFunction;
         }
         break;

         default:
            break;
      }

      strategy->SetEaseFunction(easeFunction);
   }
}


bool AnimationWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const AnimationWidget2D* original = CLONEABLE_WIDGET_CAST<const AnimationWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetPropertyAnimation(original->GetPropertyAnimation());
      SetSpeedFactor(original->GetSpeedFactor());
      SetToggleDirection(original->GetToggleDirection());
      SetDirection(original->GetDirection());
      SetInterpolationStrategy(original->GetInterpolationStrategy());
      SetShouldRun(original->GetShouldRun());
      SetRepeatTimes(original->GetRepeatTimes());
      SetKeyFrameCount(original->GetKeyFrameCount());
      SetSequenceTime(original->GetSequenceTime());
      SetChannel1(original->GetChannel1());
      SetChannel2(original->GetChannel2());
      _shouldRun = original->_shouldRun;
      _shouldStop = original->_shouldStop;

      if (GetPropertyAnimation() == AnimateWidgetProperty)
      {
         SetWidgetToAnimate(controlTemplateMap.ResolveWidgetClone(original->GetWidgetToAnimate()));
         SetWidgetPropertyNameToAnimate(original->GetWidgetPropertyNameToAnimate());
      }
      cloned = true;
   }
   return cloned;
}


#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
template<> ::FeatStd::UInt32 StringBufferAppender<Candera::enAnimationPlayDirection>::Append(::FeatStd::StringBuffer& stringBuffer, Candera::enAnimationPlayDirection const& object)
{
   FEATSTD_UNUSED(object);
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("::Candera::enAnimationPlayDirection {");
   tcharCount += stringBuffer.Append("todo");
   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


} //namespace FeatStd
#endif
