/* ***************************************************************************************
* FILE:          AnimationHelperInterface.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationHelperInterface.cpp is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "AnimationHelperInterface.h"

using namespace Candera::MemoryManagement;
using namespace Candera::Animation;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_OPS
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AnimationHelperInterface.cpp.trc.h"
#endif


AnimationHelperInterface::AnimationHelperInterface() :
   _nodeAnimationPlayerMap()
{
   _vecAnimationPlayers.Clear();
   _componentCount = 1;
}


AnimationHelperInterface::~AnimationHelperInterface()
{
   //Animation
   if (_vecAnimationPlayers.Size() != 0)
   {
      AnimationElementsContainer* animationElementBlock;
      for (std::size_t index = 0; index < _vecAnimationPlayers.Size(); index++)
      {
         animationElementBlock = _vecAnimationPlayers[index];
         if (animationElementBlock != NULL)
         {
            AnimationBaseHelper::removePlayerFromAnimationTimeDispatcher(animationElementBlock->getAnimationPlayer(), _animationTimeDispatcher);
            AnimationBaseHelper::removeListenerFromPlayer(animationElementBlock->getAnimationPlayer(), this);
            FEATSTD_DELETE(animationElementBlock); // this should call the destructor of AnimationElementsContainer
         }
      }
   }

   _vecAnimationPlayers.Clear();
   _nodeAnimationPlayerMap.clear();
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void AnimationHelperInterface::OnPastEnd(Candera::Animation::AnimationPlayerBase* animationPlayer, Candera::Int32 completedIterationsCount)
{
   AnimationElementsContainer* animationPlayerObj = NULL;
   for (std::size_t index = 0; index < _vecAnimationPlayers.Size(); index++)
   {
      animationPlayerObj = _vecAnimationPlayers[index];
      if ((animationPlayerObj != 0) && (animationPlayer == (Candera::Animation::AnimationPlayerBase*)animationPlayerObj->getAnimationPlayer().GetPointerToSharedInstance()))
      {
         if (animationPlayer->GetRepeatCount() == completedIterationsCount)
         {
            animationPlayerObj->setAnimationPlaying(false);
            break;
         }
      }
   }
   return;
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void AnimationHelperInterface::setTimeDispatcher(Candera::Animation::AnimationTimeDispatcher::SharedPointer pAnimationTimeDispatcher)
{
   _animationTimeDispatcher = pAnimationTimeDispatcher;
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void AnimationHelperInterface::startAnimation(
   Candera::Node2D* pNode,
   FeatStd::Int32 u32AnimateDuration,
   const float* animationStartValue,
   const float* animationEndValue,
   enAnimationProperty type,
   Candera::UInt32 repeatCount
)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationHelperInterface : startAnimation request received"));

   if (animationStartValue == 0 || animationEndValue == 0 || pNode == NULL)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "AnimationHelperInterface : Start animation input values are NULL"));
      return;
   }
   AnimationElementsContainer* animationElementBlock = 0;
   bool useExistingAnimation = false;
   if ((_vecAnimationPlayers.Size() > 0) && (isAnyAnimationPlayerFree()))
   {
      // Handling for  the Node,  enAnimationProperty type, animation duration, repeat count, animation end value is handled	for already created animation object.
      // TODO : Handle the start and End values of the Animation Sequence, currently it is always starting from 0 and animationDuration is taken as end Value
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "AnimationHelperInterface : startAnimation Create new Animation elements"));
      animationElementBlock = getFreeanimationElementBlock();
      useExistingAnimation = true;
   }
   else
   {
      animationElementBlock = FEATSTD_NEW(AnimationElementsContainer);
      _vecAnimationPlayers.Add(animationElementBlock);
   }

   if (animationElementBlock != NULL)
   {
      // Based on the Animation type, allocate memory for _animationValues to store Animation Values

      if (animationElementBlock->_animationValues != NULL)
      {
         FEATSTD_DELETE_ARRAY(animationElementBlock->_animationValues);
      }

      if (type == enAnimatePosition || type == enAnimateScale)
      {
         _componentCount = 2;
         animationElementBlock->_animationValues = FEATSTD_NEW_ARRAY(Candera::Float, 4);
         if (animationElementBlock->_animationValues != 0)
         {
            memset(animationElementBlock->_animationValues, 0, (sizeof(FeatStd::Float) * 4));
            for (FeatStd::UInt32 i = 0; i < 2; i++)
            {
               animationElementBlock->_animationValues[i] = animationStartValue[i];
               animationElementBlock->_animationValues[i + 2] = animationEndValue[i];
            }
         }
      }
      else
      {
         _componentCount = 1;
         animationElementBlock->_animationValues = FEATSTD_NEW_ARRAY(Candera::Float, 2);

         if (animationElementBlock->_animationValues != NULL)
         {
            memset(animationElementBlock->_animationValues, 0, (sizeof(FeatStd::Float) * 2));
            animationElementBlock->_animationValues[0] = *(animationStartValue);
            animationElementBlock->_animationValues[1] = *(animationEndValue);
         }
      }

      animationElementBlock->_animationSequenceTime[0] = 0; // TODO : Starting value may not be 0
      animationElementBlock->_animationSequenceTime[1] = u32AnimateDuration;

      if (useExistingAnimation)
      {
         AnimationBaseHelper::assignNodeToAnimate(animationElementBlock->getAnimationBlendedProperty(), pNode);
         AnimationBaseHelper::setAnimationSequenceDuration(animationElementBlock->getAnimationPlayer(), u32AnimateDuration);
         AnimationBaseHelper::setAnimationRepeatCount(animationElementBlock->getAnimationPlayer(), repeatCount);
         NodeAnimationPlayerMap::iterator iter = _nodeAnimationPlayerMap.begin();
         while (iter != _nodeAnimationPlayerMap.end())
         {
            if (iter->second == animationElementBlock)
            {
               NodeAnimationPlayerMap::iterator toErase = iter; // delete the existing node in the map
               ++iter;
               _nodeAnimationPlayerMap.erase(toErase);
               break;
            }
            else
            {
               ++iter;
            }
         }
         _nodeAnimationPlayerMap[pNode] = animationElementBlock; // add the new node to the map
      }
      else
      {
         animationElementBlock->setKeyFrameSequence(AnimationBaseHelper::createAnimationKeyFrameSequence()); // For New Animation
      }

      if (animationElementBlock->getKeyFrameSequence().PointsToNull())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "AnimationHelperInterface : No keyframe sequence in animation element block"));
         return;
      }
      animationElementBlock->getKeyFrameSequence()->SetKeyframes(
         _componentCount,
         2,        // currently it is restricted to only two values
         animationElementBlock->_animationSequenceTime,
         0,
         animationElementBlock->_animationValues,
         0);

      animationElementBlock->getKeyFrameSequence()->SetInterpolationStrategy(AnimationBaseHelper::createInterpolationStrategy<LinearInterpolationStrategy>());
   }

   if (! useExistingAnimation)
   {
      createAnimationProperties(animationElementBlock, pNode, type);

      bool result = false;
      AnimationController::SharedPointer controller = AnimationBaseHelper::createAnimationController();
      if (!controller.PointsToNull())
      {
         animationElementBlock->setAnimationController(controller);
         AnimationBaseHelper::addPropertyToAnimationController(controller, animationElementBlock->getAnimationBlendedProperty());
         controller->SetSequenceReferenceTime(0);
         result = true;
      }

      if (result)
      {
         AnimationPlayer::SharedPointer player = AnimationBaseHelper::createAnimationPlayer();
         if (!player.PointsToNull())
         {
            AnimationBaseHelper::setupAnimationPlayer(player,
                  Candera::Animation::AnimationPlayer::Forward,
                  repeatCount, // repeat count
                  1.0f, // speed factor
                  u32AnimateDuration); // duration

            AnimationBaseHelper::addPlayerToAnimationTimeDispatcher(player, _animationTimeDispatcher);
            AnimationBaseHelper::addListenerToPlayer(player, this);

            AnimationBaseHelper::setAnimationController(player, animationElementBlock->getAnimationController());

            animationElementBlock->setAnimationPlayer(player);

            // Updating the map
            _nodeAnimationPlayerMap[pNode] = animationElementBlock;
         }
      }
   }

   startAnimationAttachedToNode(pNode);
}


/******************************************************************************
*       Function      : CreateNodeRenderingAnimationProperties
*       Description   :
*       Parameters     :
*       Return        : void
*****************************************************************************/
void AnimationHelperInterface::createAnimationProperties(AnimationElementsContainer* animationElementBlock, Candera::Node2D* pNode, enAnimationProperty animationType)
{
   if (animationElementBlock != NULL  && pNode != NULL)
   {
      animationElementBlock->setAnimationBlendedProperty(AnimationBaseHelper::createAnimationBlendedProperty());

      if (!animationElementBlock->getAnimationBlendedProperty().PointsToNull())
      {
         AnimationBaseHelper::addKeyframeSequence(animationElementBlock->getAnimationBlendedProperty(), animationElementBlock->getKeyFrameSequence());
         AnimationBaseHelper::createAnimationPropertySetter(animationElementBlock->getAnimationBlendedProperty(), pNode, animationType);

         /*AnimationBaseHelper::createPropertySetter<Candera::Animation::RenderingEnabledNode2DPropertySetter>(animationPlayerObj->getAnimationBlendedProperty());
         AnimationBaseHelper::assignNodeToAnimate(animationPlayerObj->getAnimationBlendedProperty(), pNode);*/
      }
   }
}


/******************************************************************************
*       Function      : StartAnimation
*       Description   :
*       Parameters     :Candera::RenderNode*
*       Return        : bool
*****************************************************************************/
bool AnimationHelperInterface::startAnimationAttachedToNode(Candera::Node2D* pNode)
{
   bool bRet = false;
   if (pNode != 0)
   {
      // To find the AnimationPlayer corresponding to Node2D*
      NodeAnimationPlayerMap::iterator iter = _nodeAnimationPlayerMap.find(pNode);
      if (iter != _nodeAnimationPlayerMap.end() && ((iter->second) != NULL))
      {
         SharedPointer<AnimationPlayer> player = (iter->second)->getAnimationPlayer();

         if (!player.PointsToNull())
         {
            if (player->Start())
            {
               (iter->second)->setAnimationPlaying(true);
               bRet = true;
            }
            else
            {
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Animation failed to Start"));
            }
         }
      }
   }
   return bRet;
}


/******************************************************************************
*       Function      : StopAnimation
*       Description   :
*       Parameters     :Candera::RenderNode*
*       Return        : bool
*****************************************************************************/
bool AnimationHelperInterface::stopAnimation(Candera::Node2D* pNode)
{
   bool bRet = false;

   // To find the AnimationPlayer corresponding to Node2D*
   NodeAnimationPlayerMap::iterator iter = _nodeAnimationPlayerMap.find(pNode);
   if (iter != _nodeAnimationPlayerMap.end() && ((iter->second) != NULL))
   {
      SharedPointer<AnimationPlayer> player = (iter->second)->getAnimationPlayer();

      if (!player.PointsToNull())
      {
         if (player->Stop())
         {
            (iter->second)->setAnimationPlaying(false);
            bRet = true;
         }
      }
   }

   return bRet;
}


/******************************************************************************
*       Function      : IsAnimationEnabledOnNode
*       Description   :
*       Parameters     :Candera::RenderNode*
*       Return        : bool
*****************************************************************************/
bool AnimationHelperInterface::isAnimationEnabledOnNode(Candera::Node2D* pNode)
{
   bool bRet = false;

   // To find the AnimationPlayer corresponding to Node2D*
   NodeAnimationPlayerMap::iterator iter = _nodeAnimationPlayerMap.find(pNode);
   if (iter != _nodeAnimationPlayerMap.end() && ((iter->second) != NULL))
   {
      SharedPointer<AnimationPlayer> player = (iter->second)->getAnimationPlayer();

      if (!player.PointsToNull())
      {
         if (player->IsEnabled())
         {
            bRet = true;
         }
      }
   }

   return bRet;
}


/******************************************************************************
*       Function      : IsAnimationOnAnyPlayerRunning
*       Description   :
*       Parameters     :
*       Return        : bool
*****************************************************************************/

bool AnimationHelperInterface::isAnimationOnAnyPlayerRunning()
{
   bool bRet = false;
   AnimationElementsContainer* animationElementBlock;
   for (std::size_t index = 0; index < _vecAnimationPlayers.Size() && (!bRet); index++)
   {
      animationElementBlock = _vecAnimationPlayers[index];

      if (animationElementBlock != 0)
      {
         if (!animationElementBlock->getAnimationPlayer().PointsToNull() && (animationElementBlock->getAnimationPlayer()->IsEnabled()))
         {
            bRet = true;
            break;
         }
      }
   }
   return bRet;
}


/******************************************************************************
*       Function      : isAnyAnimationPlayerFree
*       Description   :
*       Parameters    :
*       Return        : bool
*****************************************************************************/
bool AnimationHelperInterface::isAnyAnimationPlayerFree()
{
   bool bRet = false;
   AnimationElementsContainer* animationElementBlock;
   if (_vecAnimationPlayers.Size() > 0)
   {
      for (std::size_t index = 0; index < _vecAnimationPlayers.Size(); index++)
      {
         animationElementBlock = _vecAnimationPlayers[index];
         if ((animationElementBlock != NULL) && (animationElementBlock->isAnimationPlaying() == false))
         {
            bRet = true;
            break;
         }
      }
   }
   return bRet;
}


void AnimationHelperInterface::stopAllExistingAnimations()
{
   AnimationElementsContainer* animationElementBlock = NULL;
   for (std::size_t index = 0; index < _vecAnimationPlayers.Size(); index++)
   {
      animationElementBlock = _vecAnimationPlayers[index];
      if (animationElementBlock != 0)
      {
         (void)animationElementBlock->stopAllAnimations();
      }
   }
}


AnimationElementsContainer* AnimationHelperInterface::getFreeanimationElementBlock()
{
   AnimationElementsContainer* animationElementBlock = NULL;
   if (_vecAnimationPlayers.Size() > 0)
   {
      for (std::size_t index = 0; index < _vecAnimationPlayers.Size(); index++)
      {
         animationElementBlock = _vecAnimationPlayers[index];
         if (animationElementBlock->isAnimationPlaying() == false)
         {
            break;
         }
      }
   }
   return animationElementBlock;
}


AnimationElementsContainer::~AnimationElementsContainer()
{
   if (!_animationController.PointsToNull())
   {
      _animationController.Release(); // Releasing the controller
   }
   if (!_animationPlayer.PointsToNull())
   {
      _animationPlayer.Release();   // Releasing the player
   }
   _isAnimationPlaying = false;
   if (_animationValues != NULL)
   {
      FEATSTD_DELETE_ARRAY(_animationValues);
   }
   _animationValues = 0;
}


bool AnimationElementsContainer::isAnimationPlaying()
{
   return _isAnimationPlaying;
}


void AnimationElementsContainer::setAnimationPlaying(bool bStatus)
{
   _isAnimationPlaying = bStatus;
}


bool AnimationElementsContainer::stopAllAnimations()
{
   if (!_animationPlayer.PointsToNull())
   {
      return _animationPlayer->Stop();
   }
   return false;
}
