/* ***************************************************************************************
* FILE:          DefaultAdorner.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DefaultAdorner.h is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(DefaultAdorner_h)
#define DefaultAdorner_h

#include "Adorner.h"
#include <View/CGI/CgiExtensions/AnimationWrapper.h>

namespace hmibase {
namespace widget {
namespace adorner {


/*****************************************************************************/
/* Wraps the adorner so we can be notified if it is being deleted.
*/
/*****************************************************************************/
class AdornerNode : public Candera::Group2D
{
      FEATSTD_TYPEDEF_BASE(Candera::Group2D);

   public:
      virtual ~AdornerNode();

      //virtual void DisposeSelf();

      Adorner::SharedPointer AdornerHandle;
};


/*****************************************************************************/
class DefaultAdorner : public Adorner
{
      FEATSTD_TYPEDEF_BASE(Adorner);

   public:
      CANDERA_RTTI_DECLARATION(DefaultAdorner);

      DefaultAdorner(Candera::Widget2D* sourceWidget, Candera::Node2D* adornerNode);
      virtual ~DefaultAdorner();

      /*****************************************************************************/
      virtual void onNodeDeleted(Candera::Node2D* node);

      /*****************************************************************************/
      virtual void show();
      virtual void hide();
      virtual void destroy();
      virtual void move(const Candera::Vector2& delta);

   protected:
      void invalidateView();
      void detachAnimation();

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(DefaultAdorner);
};


/*****************************************************************************/
/* Default implementation for adorner animations which animates the position of the adorner.
* It creates internally an animation with 2 keyframes and linear interpolation.
*/
/*****************************************************************************/
class DefaultAdornerAnimation : public AdornerAnimation, public Candera::Animation::AnimationPlayerListener
{
      FEATSTD_TYPEDEF_BASE(AdornerAnimation);

   public:
      CANDERA_RTTI_DECLARATION(DefaultAdornerAnimation);

      DefaultAdornerAnimation(AppViewHandler& appViewHandler);
      virtual ~DefaultAdornerAnimation();

      /*****************************************************************************/
      /* Animation control */
      virtual bool start();
      virtual void stop();
      virtual void reset();
      virtual bool isRunning() const;

      /*****************************************************************************/
      /* Attach/detach adorner */
      virtual bool attachToAdorner(Adorner::SharedPointer adorner);
      virtual void detachFromAdorner();
      virtual void onAdornerNodeDeleted(Candera::Node2D* adornerNode);

      /*****************************************************************************/
      /* Animation configuration */
      virtual Candera::Animation::SequenceTimeType getDuration() const;
      virtual void setDuration(Candera::Animation::SequenceTimeType duration);

      virtual Candera::Vector2 getBeginValues() const;
      virtual void setBeginValues(const Candera::Vector2& values);

      virtual Candera::Vector2 getEndValues() const;
      virtual void setEndValues(const Candera::Vector2& values);

      virtual void addListener(AnimationListener& listener);
      virtual void removeListener(AnimationListener& listener);

      //overrides from Candera::Animation::AnimationPlayerListener
      virtual void OnStartAnimation(Candera::Animation::AnimationPlayerBase* animationPlayer);
      virtual void OnStopAnimation(Candera::Animation::AnimationPlayerBase* animationPlayer);
      virtual void OnFinishAnimation(Candera::Animation::AnimationPlayerBase* animationPlayer);
      virtual void OnPastEnd(Candera::Animation::AnimationPlayerBase* animationPlayer, Candera::Int32 completedIterationsCount);

   protected:
      virtual void notifyAnimationEnded();

      ::hmibase::view::AnimationWrapper _animationWrapper;
      ::hmibase::view::AnimatedPropertyWrapper _animatedPropertyWrapper;
      AnimationListener* _listener;

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(DefaultAdornerAnimation);
};


/*****************************************************************************/
class DefaultAdornerManager : public AdornerManager
{
   public:
      DefaultAdornerManager();
      virtual ~DefaultAdornerManager() {}

      /* Creates a new adorner for the specified widget.
      * If an adorner node is specified it will be used, otherwise the widget node will be used.
      * If an adorner container is specified, the adorner will be added to it, otherwise it will be added to default location in the scene.
      */
      virtual Adorner::SharedPointer createAdorner(Candera::Widget2D& widget, Candera::Node2D* adornerNode, Candera::Node2D* adornerContainer);
      virtual void destroyAdorner(Adorner::SharedPointer adorner);
      virtual void showAdorner(Adorner::SharedPointer adorner);
      virtual void hideAdorner(Adorner::SharedPointer adorner);
      virtual void moveAdorner(Adorner::SharedPointer adorner, const Candera::Vector2& delta);

      /*****************************************************************************/
      /* Animation control */
      virtual AdornerAnimation* createAnimation(Candera::Widget2D& widget);
      virtual void destroyAnimation(AdornerAnimation* animation);
      virtual bool startAnimation(AdornerAnimation& animation);
      virtual void stopAnimation(AdornerAnimation& animation);
      virtual void resetAnimation(AdornerAnimation& animation);

      /*****************************************************************************/
      /* Attach/detach adorner */
      virtual bool attachAnimationToAdorner(AdornerAnimation& animation, Adorner::SharedPointer adorner);
      virtual void detachAnimationFromAdorner(AdornerAnimation& animation);

      /*****************************************************************************/
      /* Animation configuration */
      virtual void setAnimationDuration(AdornerAnimation& animation, Candera::Animation::SequenceTimeType duration);
      virtual void setAnimationBeginValues(AdornerAnimation& animation, const Candera::Vector2& values);
      virtual void setAnimationEndValues(AdornerAnimation& animation, const Candera::Vector2& values);

   protected:
      virtual Candera::Node2D* cloneNode(Candera::Node2D& node);
      virtual Adorner* newAdorner(Candera::Widget2D& widget, Candera::Node2D* adornerNode);
      virtual Candera::Node2D* prepareAdornerNodeContent(Candera::Widget2D& widget, Candera::Node2D* adornerNodeTemplate);
      virtual AdornerNode* prepareAdornerNode(Candera::Widget2D& widget, Candera::Node2D* adornerNodeContent, Candera::Node2D* adornerContainer);
      virtual Adorner::SharedPointer prepareAdorner(Candera::Widget2D& widget, AdornerNode* adornerNode);
      virtual Candera::Rectangle calculateAdornerNodeBounds(Candera::Widget2D& widget, AdornerNode* adornerNode, Candera::Node2D* adornerNodeContent, Candera::Node2D* adornerContainer);

      virtual AdornerAnimation* newAnimation(Candera::Widget2D& widget);
      virtual bool prepareAnimation(AdornerAnimation* animation, Candera::Widget2D& widget);

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(DefaultAdornerManager);
};


}//adorner
}//widget
}//hmibase

#endif
