/* ***************************************************************************************
* FILE:          DefaultAdorner.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Adorner.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "DefaultAdorner.h"

#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/Engine2D/Core/Group2D.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrush.h>
#include <CanderaAssetLoader/AssetLoaderBase/AnimationPropertySetterFactory.h>
#include <View/CGI/CgiExtensions/AppViewHandler.h>
#include <Widgets/2D/ControlTemplate/TCloneTraverser.h>
#include <Widgets/2D/Marker/MarkerWidget2D.h>
#include <Widgets/utils/WidgetTraverser.h>


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_FW_DRAG_DROP
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DefaultAdorner.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace adorner {

using ::Candera::Char;
CANDERA_RTTI_DEFINITION(DefaultAdorner)
CANDERA_RTTI_DEFINITION(DefaultAdornerAnimation)


/*****************************************************************************/
AdornerNode::~AdornerNode()
{
   if (!AdornerHandle.PointsToNull())
   {
      AdornerHandle->onNodeDeleted(this);
   }
}


/*****************************************************************************/
DefaultAdorner::DefaultAdorner(Candera::Widget2D* sourceWidget, Candera::Node2D* node)
{
   ETG_TRACE_USR1(("DefaultAdorner constructor this=%p, sourceWidget=%s",
                   this, _sourceWidget.isValid() ? _sourceWidget->GetLegacyName() : "null"));

   _sourceWidget = Widget2DAccessor(sourceWidget);
   _node = node;
}


/*****************************************************************************/
DefaultAdorner::~DefaultAdorner()
{
   ETG_TRACE_USR1(("DefaultAdorner destructor this=%p", this));
}


/*****************************************************************************/
void DefaultAdorner::show()
{
   Candera::Vector2 coord = (_node != NULL) ? _node->GetWorldPosition() : Candera::Vector2();

   ETG_TRACE_USR1(("DefaultAdorner::show this=%p, node=%p, coord=[%d,%d], sourceWidget=%s",
                   this, _node, static_cast<int>(coord.GetX()), static_cast<int>(coord.GetY()),
                   _sourceWidget.isValid() ? _sourceWidget->GetLegacyName() : "null"));

   if (_node != NULL)
   {
      _visible = true;
      _node->SetRenderingEnabled(true);
      invalidateView();
   }
}


/*****************************************************************************/
void DefaultAdorner::hide()
{
   ETG_TRACE_USR1(("DefaultAdorner::hide this=%p", this));

   _visible = false;
   if (_node != NULL)
   {
      _node->SetRenderingEnabled(false);
      invalidateView();
   }
}


/*****************************************************************************/
void DefaultAdorner::destroy()
{
   ETG_TRACE_USR1(("DefaultAdorner::destroy this=%p, node=%p", this, _node));

   Candera::Node2D* node = _node;
   _node = NULL;
   _sourceWidget = NULL;
   _visible = false;

   detachAnimation();

   if (node != NULL)
   {
      node->Unload(Candera::ScopeMask(), Candera::Node2D::Deep);
      node->SetRenderingEnabled(false);

      //only remove the content of the adorner node
      //don't dispose the adorner node because it may be still animated
      //the adorner node will be deleted when the scene is deleted because we leave it in the scene tree
      Candera::Node2D* temp = node->GetFirstChild();
      while (temp != NULL)
      {
         node->RemoveChild(temp);
         temp->Dispose();
         temp = node->GetFirstChild();
      }

      AdornerNode* adornerNode = Candera::Dynamic_Cast<AdornerNode*>(node);
      if (adornerNode != NULL)
      {
         adornerNode->AdornerHandle.Release();
      }
   }
}


/*****************************************************************************/
void DefaultAdorner::move(const Candera::Vector2& delta)
{
   Candera::Vector2 adjustedDelta(delta);

   AdornerMovementCapabilities* movementCapabilities = Data.get<AdornerMovementCapabilities>();
   if (movementCapabilities != NULL)
   {
      if (!movementCapabilities->IsHorizontalMovementEnabled)
      {
         adjustedDelta.SetX(0);
      }
      if (!movementCapabilities->IsVerticalMovementEnabled)
      {
         adjustedDelta.SetY(0);
      }
   }

   ETG_TRACE_USR1(("DefaultAdorner::move this=%p, node=%p, visible=%u, delta=[%d, %d], adjustedDelta=[%d,%d]",
                   this, _node, _visible,
                   static_cast<int>(delta.GetX()), static_cast<int>(delta.GetY()),
                   static_cast<int>(adjustedDelta.GetX()), static_cast<int>(adjustedDelta.GetY())));

   if ((_node != NULL) && ((adjustedDelta.GetX() != 0) || (adjustedDelta.GetY() != 0)))
   {
      _node->Translate(adjustedDelta);

      Candera::Vector2 nodePos(_node->GetWorldPosition());
      ETG_TRACE_USR1(("DefaultAdorner::move newPosition=[%d,%d]",
                      static_cast<int>(nodePos.GetX()), static_cast<int>(nodePos.GetY())));

      if (_visible)
      {
         invalidateView();
      }
   }
}


/*****************************************************************************/
void DefaultAdorner::onNodeDeleted(Candera::Node2D* node)
{
   ETG_TRACE_USR1(("DefaultAdorner::onNodeDeleted this=%p, node=%p, deletedNode=%p", this, _node, node));

   if (_node == node)
   {
      _node = NULL;
      _sourceWidget = NULL;
      _visible = false;

      if (_animation != NULL)
      {
         _animation->onAdornerNodeDeleted(node);
      }
   }
}


/*****************************************************************************/
void DefaultAdorner::invalidateView()
{
   if (_sourceWidget.isValid() && (_sourceWidget->GetParentView() != NULL))
   {
      _sourceWidget->GetParentView()->Invalidate();
   }
}


/*****************************************************************************/
void DefaultAdorner::detachAnimation()
{
   AdornerAnimation* animation = _animation;
   _animation = NULL;

   if (animation != NULL)
   {
      animation->detachFromAdorner();
   }
}


/*****************************************************************************/
DefaultAdornerAnimation::DefaultAdornerAnimation(AppViewHandler& appViewHandler) : _animationWrapper(appViewHandler), _listener(NULL)
{
   _animationWrapper.init();
   _animationWrapper.addListener(this);

   //2 keyframes (at 0ms and 1000ms), 2 channels (X,Y)
   _animatedPropertyWrapper.init(2, 2, Candera::Animation::LinearInterpolationStrategy::Create());
   _animatedPropertyWrapper.setSequenceTime(0, 0);
   _animatedPropertyWrapper.setSequenceTime(1, 1000);

   _animatedPropertyWrapper.addToPlayer(_animationWrapper.getPlayer());
}


/*****************************************************************************/
DefaultAdornerAnimation::~DefaultAdornerAnimation()
{
   _listener = NULL;
}


/*****************************************************************************/
void DefaultAdornerAnimation::addListener(AnimationListener& listener)
{
   _listener = &listener;
}


/*****************************************************************************/
void DefaultAdornerAnimation::removeListener(AnimationListener& listener)
{
   if (_listener == &listener)
   {
      _listener = NULL;
   }
}


/*****************************************************************************/
void DefaultAdornerAnimation::notifyAnimationEnded()
{
   if (_listener != NULL)
   {
      _listener->onAnimationEnded(this);
   }
}


/*****************************************************************************/
void DefaultAdornerAnimation::OnStartAnimation(Candera::Animation::AnimationPlayerBase* /*animationPlayer*/)
{
}


/*****************************************************************************/
void DefaultAdornerAnimation::OnStopAnimation(Candera::Animation::AnimationPlayerBase* animationPlayer)
{
   if (_animationWrapper.getPlayer().GetPointerToSharedInstance() == animationPlayer)
   {
      notifyAnimationEnded();
   }
}


/*****************************************************************************/
void DefaultAdornerAnimation::OnFinishAnimation(Candera::Animation::AnimationPlayerBase* animationPlayer)
{
   if (_animationWrapper.getPlayer().GetPointerToSharedInstance() == animationPlayer)
   {
      notifyAnimationEnded();
   }
}


/*****************************************************************************/
void DefaultAdornerAnimation::OnPastEnd(Candera::Animation::AnimationPlayerBase* animationPlayer, Candera::Int32 /*completedIterationsCount*/)
{
   if (_animationWrapper.getPlayer().GetPointerToSharedInstance() == animationPlayer)
   {
      notifyAnimationEnded();
   }
}


/*****************************************************************************/
bool DefaultAdornerAnimation::start()
{
   return !_animatedPropertyWrapper.getPropertySetter().PointsToNull() && _animationWrapper.start();
}


/*****************************************************************************/
void DefaultAdornerAnimation::stop()
{
   _animationWrapper.stop();
}


/*****************************************************************************/
void DefaultAdornerAnimation::reset()
{
   _animationWrapper.reset();
}


/*****************************************************************************/
bool DefaultAdornerAnimation::isRunning() const
{
   return _animationWrapper.isEnabled();
}


/*****************************************************************************/
bool DefaultAdornerAnimation::attachToAdorner(Adorner::SharedPointer adorner)
{
   ETG_TRACE_USR1(("DefaultAdornerAnimation::attachToAdorner this=%p, adorner=%p",
                   this, adorner.GetPointerToSharedInstance()));

   if (!adorner.PointsToNull() && (adorner->getNode() != NULL))
   {
      Candera::Animation::AnimationPropertySetter::SharedPointer propertySetter = ::hmibase::view::AnimationPropertySetterUtils::createNodePropertySetter<Candera::Animation::Transformable2DTranslatePropertySetter>(adorner->getNode());
      if (!propertySetter.PointsToNull())
      {
         _adorner = adorner;
         _adorner->setAnimation(this);

         if (_animationWrapper.getPlayer() != NULL)
         {
            AdornerDynamicProperties::SetAdorner(*_animationWrapper.getPlayer(), _adorner);
         }

         setBeginValues(_adorner->getNode()->GetWorldPosition());
         _animatedPropertyWrapper.setPropertySetter(propertySetter);

         return true;
      }
   }
   return false;
}


/*****************************************************************************/
void DefaultAdornerAnimation::detachFromAdorner()
{
   ETG_TRACE_USR1(("DefaultAdornerAnimation::detachFromAdorner this=%p, adorner=%p",
                   this, _adorner.GetPointerToSharedInstance()));

   stop();

   if (!_adorner.PointsToNull())
   {
      _adorner->setAnimation(NULL);
      _adorner.Release();
   }

   if (_animationWrapper.getPlayer() != NULL)
   {
      AdornerDynamicProperties::ClearAdorner(*_animationWrapper.getPlayer());
   }

   _animatedPropertyWrapper.setPropertySetter(Candera::Animation::AnimationPropertySetter::SharedPointer());
}


/*****************************************************************************/
void DefaultAdornerAnimation::onAdornerNodeDeleted(Candera::Node2D*)
{
   detachFromAdorner();
}


/*****************************************************************************/
Candera::Animation::SequenceTimeType DefaultAdornerAnimation::getDuration() const
{
   return _animatedPropertyWrapper.getSequenceTime(1);
}


/*****************************************************************************/
void DefaultAdornerAnimation::setDuration(Candera::Animation::SequenceTimeType duration)
{
   _animatedPropertyWrapper.setSequenceTime(1, duration);
}


/*****************************************************************************/
Candera::Vector2 DefaultAdornerAnimation::getBeginValues() const
{
   return Candera::Vector2(_animatedPropertyWrapper.getKeyframeValue(0, 0), _animatedPropertyWrapper.getKeyframeValue(0, 1));
}


/*****************************************************************************/
void DefaultAdornerAnimation::setBeginValues(const Candera::Vector2& values)
{
   ETG_TRACE_USR1(("DefaultAdorner::setBeginValues this=%p, values=[%d,%d]",
                   this, static_cast<int>(values.GetX()), static_cast<int>(values.GetY())));

   //first keyframe
   _animatedPropertyWrapper.setKeyframeValue(0, 0, values.GetX());
   _animatedPropertyWrapper.setKeyframeValue(0, 1, values.GetY());
}


/*****************************************************************************/
Candera::Vector2 DefaultAdornerAnimation::getEndValues() const
{
   return Candera::Vector2(_animatedPropertyWrapper.getKeyframeValue(1, 0), _animatedPropertyWrapper.getKeyframeValue(1, 1));
}


/*****************************************************************************/
void DefaultAdornerAnimation::setEndValues(const Candera::Vector2& values)
{
   ETG_TRACE_USR1(("DefaultAdorner::setEndValues this=%p, values=[%d,%d]",
                   this, static_cast<int>(values.GetX()), static_cast<int>(values.GetY())));

   //second keyframe
   _animatedPropertyWrapper.setKeyframeValue(1, 0, values.GetX());
   _animatedPropertyWrapper.setKeyframeValue(1, 1, values.GetY());
}


/*****************************************************************************/
class NodeCloneTraverser : public TCloneTraverser<Candera::Node2D>
{
      typedef TCloneTraverser<Candera::Node2D> Base;
   public:
      NodeCloneTraverser() : Base() {}

   protected:
      virtual TraverserBase::TraverserAction OnClone(const Candera::Node2D& node, Candera::Node2D& nodeClone)
      {
         Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(&nodeClone);
         if (renderNode != NULL)
         {
            Candera::Effect2D* effect = renderNode->GetEffect(0);
            if (effect != NULL)
            {
               Candera::TextBrush* textBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(effect->GetBrushEffect2D());
               if ((textBrush != NULL) && (textBrush->Text().Get() != NULL))
               {
                  FeatStd::TChar* buffer = FEATSTD_NEW_ARRAY(FeatStd::TChar, FeatStd::Internal::String::Length(textBrush->Text().Get()) + 1);
                  if (buffer != NULL)
                  {
                     FeatStd::Internal::String::Copy(buffer, textBrush->Text().Get());
                     textBrush->Text().Set(buffer, FeatStd::MemoryManagement::ArrayDisposer<const FeatStd::TChar*>::Dispose);
                  }
               }
            }
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
            renderNode->SetClippingRect();
#endif
         }
         return Base::OnClone(node, nodeClone);
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(NodeCloneTraverser);
};


/*****************************************************************************/
Candera::Node2D* DefaultAdornerManager::cloneNode(Candera::Node2D& node)
{
   NodeCloneTraverser cloneTraverser;
   cloneTraverser.Traverse(node);
   return cloneTraverser.GetRootClone();
}


/*****************************************************************************/
DefaultAdornerManager::DefaultAdornerManager()
{
}


/*****************************************************************************/
Adorner::SharedPointer DefaultAdornerManager::createAdorner(Candera::Widget2D& widget, Candera::Node2D* adornerNodeTemplate, Candera::Node2D* adornerContainer)
{
   Candera::Node2D* node = widget.GetNode();
   if ((node != NULL) && (node->GetScene() != NULL))
   {
      Candera::Node2D* adornerNodeContent = prepareAdornerNodeContent(widget, adornerNodeTemplate);
      if (adornerNodeContent != NULL)
      {
         AdornerNode* adornerNode = prepareAdornerNode(widget, adornerNodeContent, adornerContainer);
         if (adornerNode != NULL)
         {
            Adorner::SharedPointer adorner(prepareAdorner(widget, adornerNode));
            if (!adorner.PointsToNull())
            {
               return adorner;
            }

            //clean-up for adornerNode
            if (adornerNode != NULL)
            {
               adornerNode->Unload(Candera::ScopeMask(), Candera::Node2D::Deep);
               if (adornerNode->GetFirstChild() != NULL)
               {
                  adornerNode->RemoveChild(adornerNode->GetFirstChild());
               }
            }
         }

         //clean-up for adornerNodeContent
         if (adornerNodeContent != NULL)
         {
            adornerNodeContent->Dispose();
         }
      }
   }
   return Adorner::SharedPointer();
}


/*****************************************************************************/
Adorner* DefaultAdornerManager::newAdorner(Candera::Widget2D& widget, Candera::Node2D* adornerNode)
{
   return CANDERA_NEW(DefaultAdorner)(&widget, adornerNode);
}


/*****************************************************************************/
Candera::Node2D* DefaultAdornerManager::prepareAdornerNodeContent(Candera::Widget2D& widget, Candera::Node2D* adornerNodeTemplate)
{
   Candera::Node2D* adornerNodeContent = NULL;
   Candera::Node2D* node = widget.GetNode();
   if ((node != NULL) && (node->GetScene() != NULL))
   {
      /////////////////////////////////////////////////////////////////////////
      // obtain the adorner node content from the passed template or from widget node itself
      //if an adorner node is specified use it
      if (adornerNodeTemplate != NULL)
      {
         adornerNodeContent = cloneNode(*adornerNodeTemplate);
      }
      //else create a clone of the source node
      else
      {
         adornerNodeContent = cloneNode(*node);
         if (adornerNodeContent != NULL)
         {
            adornerNodeContent->SetRenderOrderRank(100);
         }
      }

      if (adornerNodeContent != NULL)
      {
         adornerNodeContent->SetRenderingEnabled(true);
         adornerNodeContent->SetPosition(Candera::Vector2());
      }
   }
   return adornerNodeContent;
}


/*****************************************************************************/
AdornerNode* DefaultAdornerManager::prepareAdornerNode(Candera::Widget2D& widget, Candera::Node2D* adornerNodeContent, Candera::Node2D* adornerContainer)
{
   AdornerNode* adornerNode = NULL;
   Candera::Node2D* node = widget.GetNode();
   if ((node != NULL) && (node->GetScene() != NULL))
   {
      /////////////////////////////////////////////////////////////////////////
      // get the adorner node group from the scene or create a new one
      static const char* const DEFAULT_ADORNER_GROUP_NAME = "#Adorner#";
      Candera::Node2D* root = (adornerContainer != NULL) ? adornerContainer : node->GetScene();
      adornerNode = Candera::Dynamic_Cast<AdornerNode*>(root->GetChild(DEFAULT_ADORNER_GROUP_NAME));
      if (adornerNode == NULL)
      {
         adornerNode = CANDERA_NEW(AdornerNode);
         if (adornerNode != NULL)
         {
            adornerNode->SetName(DEFAULT_ADORNER_GROUP_NAME);
            root->AddChild(adornerNode);
         }
      }
      if (adornerNode != NULL)
      {
         adornerNode->SetRenderingEnabled(false);

         Candera::Rectangle adornerNodeBounds(calculateAdornerNodeBounds(widget, adornerNode, adornerNodeContent, adornerContainer));
         adornerNode->SetPosition(adornerNodeBounds.GetPosition());
         adornerNode->SetBoundingRectangle(Candera::Rectangle(Candera::Vector2(), adornerNodeBounds.GetSize()));
         Candera::Layouter::SetSize(*adornerNode, adornerNodeBounds.GetSize());

         adornerNode->AddChild(adornerNodeContent);
         adornerNode->Upload(Candera::ScopeMask(), Candera::Node2D::Deep);
      }
   }
   return adornerNode;
}


/*****************************************************************************/
Candera::Rectangle DefaultAdornerManager::calculateAdornerNodeBounds(Candera::Widget2D& widget, AdornerNode* adornerNode, Candera::Node2D* /*adornerNodeContent*/, Candera::Node2D* /*adornerContainer*/)
{
   Candera::Rectangle adornerNodeBounds;

   Candera::Node2D* node = widget.GetNode();
   if ((node != NULL) && (node->GetScene() != NULL) && (adornerNode != NULL))
   {
      Candera::Rectangle nodeRect;
      node->GetWorldAxisAlignedBoundingRectangle(nodeRect);

      ETG_TRACE_USR1(("DefaultAdornerManager::calculateAdornerNodeBounds widgetNodeBounds=[%d,%d,%d,%d]",
                      static_cast<int>(nodeRect.GetLeft()), static_cast<int>(nodeRect.GetTop()),
                      static_cast<int>(nodeRect.GetWidth()), static_cast<int>(nodeRect.GetHeight())));

      if (adornerNode->GetParent() != NULL)
      {
         adornerNodeBounds.SetPosition(nodeRect.GetPosition() - adornerNode->GetParent()->GetWorldPosition());
      }

      adornerNodeBounds.SetSize(nodeRect.GetSize());
   }
   return adornerNodeBounds;
}


/*****************************************************************************/
Adorner::SharedPointer DefaultAdornerManager::prepareAdorner(Candera::Widget2D& widget, AdornerNode* adornerNode)
{
   if (adornerNode != NULL)
   {
      Adorner::SharedPointer adorner(newAdorner(widget, adornerNode));
      if (!adorner.PointsToNull())
      {
         adorner->Data.set(_defaultMovementCapabilities);
         adornerNode->AdornerHandle = adorner;
         return adorner;
      }
   }
   return Adorner::SharedPointer();
}


/*****************************************************************************/
void DefaultAdornerManager::destroyAdorner(Adorner::SharedPointer adorner)
{
   if (!adorner.PointsToNull())
   {
      adorner->destroy();
   }
}


/*****************************************************************************/
void DefaultAdornerManager::showAdorner(Adorner::SharedPointer adorner)
{
   if (!adorner.PointsToNull())
   {
      adorner->show();
   }
}


/*****************************************************************************/
void DefaultAdornerManager::hideAdorner(Adorner::SharedPointer adorner)
{
   if (!adorner.PointsToNull())
   {
      adorner->hide();
   }
}


/*****************************************************************************/
void DefaultAdornerManager::moveAdorner(Adorner::SharedPointer adorner, const Candera::Vector2& delta)
{
   if (!adorner.PointsToNull())
   {
      adorner->move(delta);
   }
}


/*****************************************************************************/
AdornerAnimation* DefaultAdornerManager::createAnimation(Candera::Widget2D& widget)
{
   AdornerAnimation* animation = newAnimation(widget);
   if ((animation != NULL) && (getDefaultAnimationDuration() > 0))
   {
      setAnimationDuration(*animation, getDefaultAnimationDuration());
   }
   return animation;
}


/*****************************************************************************/
AdornerAnimation* DefaultAdornerManager::newAnimation(Candera::Widget2D& widget)
{
   if (widget.GetParentView() != NULL)
   {
      AppViewHandler* appViewHandler = dynamic_cast<AppViewHandler*>(widget.GetParentView()->GetViewHandler());
      if (appViewHandler != NULL)
      {
         return CANDERA_NEW(DefaultAdornerAnimation)(*appViewHandler);
      }
   }

   return NULL;
}


/*****************************************************************************/
bool DefaultAdornerManager::prepareAnimation(AdornerAnimation* animation, Candera::Widget2D& /*widget*/)
{
   if ((animation != NULL) && (getDefaultAnimationDuration() > 0))
   {
      setAnimationDuration(*animation, getDefaultAnimationDuration());
   }
   return true;
}


/*****************************************************************************/
void DefaultAdornerManager::destroyAnimation(AdornerAnimation* animation)
{
   if (animation != NULL)
   {
      CANDERA_DELETE(animation);
   }
}


/*****************************************************************************/
bool DefaultAdornerManager::startAnimation(AdornerAnimation& animation)
{
   return animation.start();
}


/*****************************************************************************/
void DefaultAdornerManager::stopAnimation(AdornerAnimation& animation)
{
   animation.stop();
}


/*****************************************************************************/
void DefaultAdornerManager::resetAnimation(AdornerAnimation& animation)
{
   animation.reset();
}


/*****************************************************************************/
bool DefaultAdornerManager::attachAnimationToAdorner(AdornerAnimation& animation, Adorner::SharedPointer adorner)
{
   return animation.attachToAdorner(adorner);
}


/*****************************************************************************/
void DefaultAdornerManager::detachAnimationFromAdorner(AdornerAnimation& animation)
{
   animation.detachFromAdorner();
}


/*****************************************************************************/
void DefaultAdornerManager::setAnimationDuration(AdornerAnimation& animation, Candera::Animation::SequenceTimeType duration)
{
   animation.setDuration(duration);
}


/*****************************************************************************/
void DefaultAdornerManager::setAnimationBeginValues(AdornerAnimation& animation, const Candera::Vector2& values)
{
   animation.setBeginValues(values);
}


/*****************************************************************************/
void DefaultAdornerManager::setAnimationEndValues(AdornerAnimation& animation, const Candera::Vector2& values)
{
   animation.setEndValues(values);
}


}//adorner
}//widget
}//hmibase
