/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/videobuffer/ApxInstance.h"
#include "hmibase/gadget/videobuffer/ApxContext.h"
#include "hmibase/util/Error.h"
#include <cstring>
#include <stdint.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <drm.h>
#include <xf86drm.h>
#include <xf86drmMode.h>
#include <sys/mman.h>


#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/ApxInstance.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace hmibase::util;

namespace hmibase {
namespace gadget {
namespace videobuffer {


int         ApxInstance::smApxContextRefCount = 0;
ApxContext* ApxInstance::smApxContextPtr      = NULL;

int ApxInstance::getApxFd()
{
   if (smApxContextPtr != NULL)
   {
      return smApxContextPtr->getApxFd();
   }
   return -1;
}


ApxInstance::ApxInstance()
{
   smApxContextRefCount++;
   if (smApxContextPtr == NULL)
   {
      smApxContextPtr = new ApxContext();
   }
}


ApxInstance::~ApxInstance()
{
   smApxContextRefCount--;
   if (smApxContextRefCount <= 0 && smApxContextPtr != NULL)
   {
      delete smApxContextPtr;
      smApxContextPtr = NULL;
   }
}


int ApxInstance::getFd(uint32_t handle)
{
   int apxFd = getApxFd();

   drm_prime_handle drmPrimeHandle;
   memset(&drmPrimeHandle, 0, sizeof(drm_prime_handle));
   drmPrimeHandle.handle = handle;

   Error::reset();
   if (drmIoctl(apxFd, DRM_IOCTL_PRIME_HANDLE_TO_FD, &drmPrimeHandle) < 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Could not get FileDescriptor for handle apxFd=%d, handle=%d. %s", apxFd, handle, Error::getMessage().cPtr()));
      return -1;
   }

   return drmPrimeHandle.fd;
}


uint32_t ApxInstance::getHandle(int fd)
{
   int apxFd = getApxFd();

   drm_prime_handle drmPrimeHandle;
   memset(&drmPrimeHandle, 0, sizeof(drm_prime_handle));
   drmPrimeHandle.fd = fd;

   Error::reset();
   if (drmIoctl(apxFd, DRM_IOCTL_PRIME_FD_TO_HANDLE, &drmPrimeHandle) < 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Could not get Handle for FileDescriptor apxFd=%d, fd=%d. %s", apxFd, fd, Error::getMessage().cPtr()));
      return -1;
   }

   return drmPrimeHandle.handle;
}


uint64_t ApxInstance::getDumpOffset(uint32_t handle)
{
   int apxFd = getApxFd();

   drm_mode_map_dumb drmModeMapDump;
   memset(&drmModeMapDump, 0, sizeof(drm_mode_map_dumb));
   drmModeMapDump.handle = handle;

   Error::reset();
   if (drmIoctl(apxFd, DRM_IOCTL_MODE_MAP_DUMB, &drmModeMapDump) < 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Could not get DumpOffset for Handle apxFd=%d, fd=%d. %s", apxFd, handle, Error::getMessage().cPtr()));
      return 0;
   }

   return drmModeMapDump.offset;
}


uint32_t ApxInstance::createDump(uint16_t width, uint16_t height, uint16_t bpp, uint32_t& size, uint32_t& pitch)
{
   int apxFd = getApxFd();

   struct drm_mode_create_dumb cStruct;
   memset(&cStruct, 0, sizeof(drm_mode_create_dumb));
   cStruct.width  = width;
   cStruct.height = height;
   cStruct.bpp    = bpp;

   Error::reset();
   if (drmIoctl(apxFd, DRM_IOCTL_MODE_CREATE_DUMB, &cStruct) < 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Could not create dump buffer apxFd=%d, width=%d, height=%d, bpp=%d. %s", apxFd, width, height, bpp, Error::getMessage().cPtr()));
   }

   // restrict to 32 bit as there are no bigger values expected
   size = static_cast<uint32_t>(cStruct.size);
   pitch = cStruct.pitch;
   return cStruct.handle;
}


void ApxInstance::destroyDump(uint32_t handle)
{
   int apxFd = getApxFd();

   struct drm_mode_destroy_dumb dStruct;
   memset(&dStruct, 0, sizeof(drm_mode_destroy_dumb));
   dStruct.handle = handle;

   Error::reset();
   if (drmIoctl(apxFd, DRM_IOCTL_MODE_DESTROY_DUMB, &dStruct) < 0 || Error::is())
   {
      ETG_TRACE_ERR_THR(("Could not destroy dump buffer apxFd=%d, handle=%d. %s", apxFd, handle, Error::getMessage().cPtr()));
   }
}


uint8_t* ApxInstance::mapBuffer(uint64_t offset, uint32_t size)
{
   int apxFd = getApxFd();

   Error::reset();
   void* result = mmap(0, size, PROT_READ | PROT_WRITE, MAP_SHARED, apxFd, offset);
   if (result == NULL || Error::is())
   {
      ETG_TRACE_ERR_THR(("Could not map buffer apxFd=%d, offset=%d, size=%d. %s", apxFd, offset, size, Error::getMessage().cPtr()));
      if (result != NULL)
      {
         munmap(result, size);
         result = NULL;
      }
      return NULL;
   }

   return (uint8_t*) result;
}


void ApxInstance::unmapBuffer(void* data, uint32_t size)
{
   int apxFd = getApxFd();

   Error::reset();
   munmap(data, size);
   if (Error::is())
   {
      ETG_TRACE_ERR_THR(("Could not unmap buffer apxFd=%d, data=%p, size=%d. %s", apxFd, data, size, Error::getMessage().cPtr()));
   }
}


} // namespace
} // namespace
} // namespace
