/* ***************************************************************************************
* FILE:          Ticker.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Ticker.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hmibase/util/Ticker.h"

#ifdef WIN32
#include <windows.h>
#include <WinBase.h>
#else
#include <unistd.h>
#endif
#include <stdio.h>

namespace hmibase
{
namespace util
{


Ticker::Ticker()
{
   start();
}


void Ticker::start()
{
   // timer reset
   _tickstart = getTickCountUsec();
   _tickstop = 0;
}


void Ticker::stop()
{
   // stop timer in memory
   _tickstop = getTickCountUsec();
}


uint64_t Ticker::diffUsec() const
{
   // time elapsed since Start()
   if (_tickstop == 0)
   {
      return (getTickCountUsec() - _tickstart);
   }
   // time elapsed after Stop()
   return (_tickstop - _tickstart);
}


void Ticker::sleepUSec(unsigned long USec)
{
#ifdef WIN32
   if (USec > 1000)
   {
      Sleep(USec / 1000);
   }
#else
   ::usleep(static_cast<__useconds_t>(USec));
#endif
}


void Ticker::sleepMSec(unsigned long MSec)
{
#ifdef WIN32
   Sleep(MSec);
#else
   ::usleep(static_cast<__useconds_t>((MSec * 1000)));
#endif
}


uint64_t Ticker::getTickCountUsec()
{
#if defined(WIN32)
   static uint64_t gfrequency;
   if (gfrequency == 0)
   {
      ::QueryPerformanceFrequency((LARGE_INTEGER*)&gfrequency);
   }
   uint64_t tCount;
   ::QueryPerformanceCounter((LARGE_INTEGER*)&tCount);
   return (tCount * 1000000ULL / gfrequency);
#else
   // time since startup in micro-seconds
   struct timespec count;
   clock_gettime(CLOCK_MONOTONIC, &count);
   uint64_t tCount = 1000000ULL * static_cast<unsigned>(count.tv_sec) + (static_cast<unsigned>(count.tv_nsec) / 1000);

   return tCount;
#endif
}


uint32_t Ticker::getTickCountMsec()
{
   return static_cast<uint32_t>(getTickCountUsec() / 1000);
}


////////////////////////////////////////////////////////////////////////////
// file and folders
////////////////////////////////////////////////////////////////////////////


#ifdef WIN32
#include <direct.h>
#else
#define  _getcwd getcwd
#include <unistd.h>
#include <string.h>
#endif

// returns the folder where the execuatble proc was started, including '/' at the tail
std::string File::getBinFolder()
{
   char buffer[1000];
   buffer[0] = '\0';
#ifdef WIN32
   ::GetModuleFileName(NULL, buffer, sizeof(buffer));
#else
   char szTmp[50];
   SNPRINTF(szTmp, sizeof(szTmp), "/proc/%d/exe", getpid());
   size_t len = sizeof(buffer) - 1;
   ssize_t bytes = readlink(szTmp, buffer, len);
   if (bytes >= 0)
   {
      buffer[bytes] = '\0';
   }
#endif
   std::string::size_type pos = std::string(buffer).find_last_of("\\/");
   return std::string(buffer).substr(0, pos + 1);
}


// returns the super repository folder of ai_hmi_base, including '/' at the tail
std::string File::getSourceRootFolder()
{
   char dir[1000];

   _getcwd(dir, sizeof(dir));
   char* s = strstr(dir, "ai_hmi_");// old location for binaries
   if (s == NULL)
   {
      s = strstr(dir, "generated");// new location for binaries
   }
   if (s)
   {
      *s = '\0';
   }
   return std::string(dir);
}


// simple helper to create a full path of given relative file
std::string  File::getSourceRootFolderAddPath(const std::string& addPath)
{
   return getSourceRootFolder() + addPath;
}


}
}
