/* ***************************************************************************************
* FILE:          Thread.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Thread.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hmibase/util/Thread.h"
#include "hmibase/util/Condition.h"

#ifdef WIN32
#include <windows.h>
#include <WinBase.h>
#else
#include <unistd.h>
#include <cstring>
#endif

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Thread.cpp.trc.h"
#endif

namespace hmibase {
namespace util {

/////////////////////////////////////////////////////////////////////////////////////
///// Thread Implementation /////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

namespace internal {

static const EventRef wait_timeout = 0;
static const EventRef wait_error   = -1;

static const unsigned int PTHREAD_NAME_MAX_LEN = 16;
#ifndef WIN32  /* Linux, Posix */
static void* threadproc(void* classpointer);

class ThreadPosix : public ::hmibase::util::internal::ThreadImpl
{
   public:
      ThreadPosix() : _threadID(0), _joined(false), _fktThread(0)
      {
      }

      virtual ~ThreadPosix()
      {
         if (_threadID != 0)
         {
            ETG_TRACE_ERR_THR(("hmibase::util::Thread, !! ThreadProc not destroyed!!"));
         }
         _threadID = 0;
         _joined = false;
      }

      bool create(Thread::IThreadFunction* fktThread, uint32_t nStackSize)
      {
         int rc = -1;
         if (_threadID == 0)
         {
            _fktThread = fktThread;
            _fktThread->setThreadState(false);
            if (nStackSize)
            {
               pthread_attr_t  attr;
               pthread_attr_init(&attr);
               pthread_attr_setstacksize(&attr, nStackSize);
               rc = pthread_create(&_threadID, &attr, threadproc, this);
               pthread_attr_destroy(&attr);
            }
            else
            {
               rc = pthread_create(&_threadID, NULL, threadproc, (void*)this);
            }
            if (rc == 0)
            {
               // wait until thread is running
               _conStart.wait(0xFFFFFFFF);
            }
            else
            {
               ETG_TRACE_ERR_THR(("hmibase::util::Thread, !! Thread not created!!"));
            }
         }
         return rc == 0;
      }

      bool leave(uint32_t dwTimeoutMsec)
      {
         PARAM_UNUSED(dwTimeoutMsec);

         _fktThread->setThreadState(true);
         // Thread beenden und warten auf Abschluß
         bool rc = false;
         if (_threadID != 0 && !_joined)
         {
            rc = pthread_join(_threadID, 0) == 0;
            _threadID = 0;
            _joined = true;
         }
         else
         {
            // "Couldn't stop Thread in xxx sec. Aborting, kill thread.");
            terminate();
         }
         _threadID = 0;
         return rc;
      }

      void exit()
      {
         pthread_exit(NULL);
      }

      bool join()
      {
         // Main block now waits for threads terminate, before it exits */
         bool rc = false;
         if (_threadID != 0 && !_joined)
         {
            rc = pthread_join(_threadID, 0) == 0;
            if (rc != 0)
            {
               ETG_TRACE_ERR_THR(("hmibase::util::Thread, !! Thread join failed !!"));
            }
            _threadID = 0;
            _joined = true;
         }
         return rc;
      }

      void setName(const char* name)
      {
         if (NULL != name)
         {
            size_t srclen = strlen(name);
            size_t destlen = PTHREAD_NAME_MAX_LEN;
            if (srclen < destlen)
            {
               destlen = srclen;
            }
            _name.assign(name, destlen);
            if (_threadID)
            {
               pthread_setname_np(_threadID, _name.c_str());
            }
         }
      }

      void terminate() // kill
      {
         // kills the thread, do not use ist normaly, use exit
         if (_threadID)
         {
            pthread_cancel(_threadID);    // kill
            _threadID = 0;
         }
      }

      void enterThreadProc()
      {
         HMI_APP_ASSERT(_fktThread);
         _conStart.signal();
         _fktThread->beforeRunning();
         _fktThread->onStart();
         _fktThread->threadWorkProc();
         _fktThread->onTerminate();
         return;
      }

   private:
      Condition      _conStart;       // sync the start
      pthread_t      _threadID;       // Thread ID
      std::string    _name;
      bool           _joined;
      Thread::IThreadFunction* _fktThread;
};


static void* threadproc(void* classpointer)
{
   ThreadPosix* threadImpl = static_cast<ThreadPosix*>(classpointer);
   HMI_APP_ASSERT(threadImpl);
   if (threadImpl)
   {
      threadImpl->enterThreadProc();
   }
   return 0;
}


#else /* Windows */

static DWORD WINAPI threadproc(void* classpointer);

class ThreadWindows : public ::hmibase::util::internal::ThreadImpl
{
   public:
      ThreadWindows() : _threadID(0), _threadHdl(0), _joined(false), _fktThread(0)
      {
      }

      virtual ~ThreadWindows()
      {
         if (_threadHdl != 0)
         {
            _threadHdl = 0;
            _joined = false;
         }
      }

      bool create(Thread::IThreadFunction* fktThread, uint32_t nStackSize)
      {
         _fktThread = fktThread;
         _fktThread->setThreadState(false);
         _threadHdl = ::CreateThread(0, nStackSize, threadproc, this, 0/*or CREATE_SUSPENDED*/, &_threadID);
         if (_threadHdl != NULL)
         {
            // wait until thread is running
            _conStart.wait(0xFFFFFFFF);
         }
         return _threadHdl != NULL;
      }

      bool leave(uint32_t dwTimeoutMsec)
      {
         // Thread beenden und warten auf Abschluß
         _fktThread->setThreadState(true);
         if (_threadHdl != 0 && !_joined)
         {
            DWORD rc = WaitForSingleObject(_threadHdl, dwTimeoutMsec);
            bool bSuccees = (rc != WAIT_TIMEOUT && rc != WAIT_FAILED);
            if (bSuccees)
            {
               _joined = true;
            }
         }
         else
         {
            // "Couldn't stop Thread in xxx sec. Aborting, kill thread.");
            terminate();
         }
         if (_threadHdl)
         {
            CloseHandle(_threadHdl);
         }
         _threadHdl = 0;
         _threadID = 0;
         return true;
      }

      void exit()
      {
         ExitThread((DWORD)0);
      }

      bool join()
      {
         // check if the the call to WaitForSingleObject() was successfully perfomed before so don't call it again.
         if (_threadHdl != 0 && !_joined)
         {
            DWORD waitStatus = WaitForSingleObject(_threadHdl, INFINITE);
            switch (waitStatus)
            {
               case WAIT_OBJECT_0: // The state of the specified object is signaled.
                  _joined = true;
                  break;
               default:
                  return false;
            }
         }
         return true;
      }

      void setName(const char* name)
      {
         _name = name;  // todo
      }

      void terminate()
      {
         // kills the thread, do not use this normally, use exit
         if (_threadHdl != 0 && !_joined)
         {
            ETG_TRACE_ERR_THR(("hmibase::util::Thread - Forced termination of a thread, please analyze !!!"));
            ::TerminateThread(_threadHdl, 0xFFFFFFFF);
         }
         if (_threadHdl != 0)
         {
            CloseHandle(_threadHdl);
            _threadHdl = 0;
            _threadID = 0;
         }
      }

      void enterThreadProc()
      {
         HMI_APP_ASSERT(_fktThread);
         _fktThread->beforeRunning();
         //_semStart.post();
         _conStart.signal();
         _fktThread->onStart();
         _fktThread->threadWorkProc();
         _fktThread->onTerminate();
         ExitThread(0);
         return;
      }

   private:
      Condition   _conStart;
      DWORD       _threadID;       // Thread ID
      HANDLE      _threadHdl;       // Handle
      std::string _name;
      bool        _joined;
      Thread::IThreadFunction* _fktThread;
};


static DWORD WINAPI threadproc(void* classpointer)
{
   ThreadWindows* threadImpl = static_cast<ThreadWindows*>(classpointer);
   HMI_APP_ASSERT(threadImpl);
   if (threadImpl)
   {
      threadImpl->enterThreadProc();
   }
   return 0;
}


#endif // WIN32
} // namespace internal


/////////////////////////////////////////////////////////////////////////////////////
///// Thread Interface  /////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

Thread::Thread() : _impl(0)
{
   setImplementation();
}


Thread::Thread(Thread:: IThreadFunction* fktThread, uint32_t nStackSize, const char* name) : _impl(0)
{
   setImplementation();
   if (create(fktThread, nStackSize, name) == false)
   {
      ETG_TRACE_ERR_THR(("hmibase::util::Thread - create failed!"));
   }
}


Thread::~Thread()
{
   if (_impl)
   {
      terminate();
      delete _impl;
   }
   _impl = 0;
}


bool Thread::create(Thread:: IThreadFunction* fktThread, uint32_t nStackSize, const char* name)
{
   HMI_APP_ASSERT(_impl);
   if (fktThread == 0 || _impl == 0)
   {
      ETG_TRACE_ERR_THR(("hmibase::util::Thread - no thread function!"));
      return false;
   }
   if (_impl->create(fktThread, nStackSize) == false)
   {
      ETG_TRACE_ERR_THR(("hmibase::util::Thread - create failed!"));
      return false;
   }
   if (name != NULL)
   {
      _impl->setName(name);
   }
   return true;
}


bool Thread::leave(uint32_t dwTimeoutMsec)
{
   if (_impl == 0 || _impl->leave(dwTimeoutMsec) == false)
   {
      ETG_TRACE_ERR_THR(("hmibase::util::Thread - leave failed!"));
      return false;
   }
   return true;
}


/* static */
void Thread::sleep(unsigned long millisecs)
{
#ifdef WIN32
   ::Sleep(millisecs);
#else
   usleep(__useconds_t(1000) * __useconds_t(millisecs));
#endif
}


bool Thread::join()
{
   if (_impl == 0 || _impl->join() == false)
   {
      ETG_TRACE_ERR_THR(("hmibase::util::Thread - join failed!"));
      return false;
   }
   return true;
}


void Thread::terminate()
{
   if (_impl)
   {
      _impl->terminate();
   }
}


void Thread::setName(const char* name)
{
   if (_impl)
   {
      _impl->setName(name);
   }
}


void Thread::setImplementation()
{
#ifndef WIN32
   _impl = new ::hmibase::util::internal::ThreadPosix();
#else
   _impl = new ::hmibase::util::internal::ThreadWindows();
#endif
   HMI_APP_ASSERT(_impl);
}


}
}
