/* ***************************************************************************************
* FILE:          Socket.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Socket.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "hmibase/util/Socket.h"

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Socket.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


#ifdef WIN32

#define WIN32_LEAN_AND_MEAN
#ifndef _WIN32_WINNT
#define  _WIN32_WINNT   0x0600 //necessary for getaddrinfo/freeaddrinfo
#endif

#include <windows.h>
#include <winsock.h>

#define socklen_t int
typedef SSIZE_T ssize_t;

#else // linux

//#include <errno.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <netinet/in.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

#include <cstring>

#define SOCKET int
#define INVALID_SOCKET -1
#define closesocket(s) close(s)

#endif // WIN32

namespace hmibase {
namespace util {
namespace socket {
namespace linux_socket {

class LinuxSocket : public ::hmibase::util::socket::Socket
{
   public:
      LinuxSocket(int hSocket) : _hSocket(hSocket) {}
      virtual ~LinuxSocket()
      {
         ::closesocket(_hSocket);
      }

      virtual void close()
      {
         ::closesocket(_hSocket);
      }

      virtual int send(const void* buffer, size_t length)
      {
         ETG_TRACE_ERR_THR(("util::tcp Sending %u bytes...", (unsigned int)length));

         const char* charBuffer = reinterpret_cast<const char*>(buffer);
         ssize_t remainingBytesToSend = length;
         ssize_t totalBytesSent = 0;
         while (remainingBytesToSend > 0)
         {
            ssize_t bytesSent = ::send(_hSocket, charBuffer, static_cast<size_t>(remainingBytesToSend), 0);
            if (bytesSent == -1)
            {
               ETG_TRACE_ERR_THR(("util::tcp Error writing to socket"));
               return -1;
            }
            remainingBytesToSend -= bytesSent;
            totalBytesSent += bytesSent;
            charBuffer += bytesSent;
         }

         return (unsigned int)totalBytesSent;
      }

      virtual int receive(void* buffer, size_t length, bool waitUntilLengthBytesReceived)
      {
         ETG_TRACE_ERR_THR(("util::tcp Receiving %u bytes...", (unsigned int)length));

         ::memset(buffer, 0, length);

         char* charBuffer = reinterpret_cast<char*>(buffer);
         ssize_t remainingBytesToReceive = length;
         ssize_t totalBytesReceived = 0;
         while (remainingBytesToReceive > 0)
         {
            ssize_t bytesReceived = ::recv(_hSocket, charBuffer, static_cast<size_t>(remainingBytesToReceive), 0);
            if (bytesReceived == -1)
            {
               ETG_TRACE_ERR_THR(("util::tcp Error reading from socket"));
               return -1;
            }
            ETG_TRACE_ERR_THR(("util::tcp Received %u bytes", (unsigned int)bytesReceived));

            remainingBytesToReceive -= bytesReceived;
            totalBytesReceived += bytesReceived;
            charBuffer += bytesReceived;
            if (!waitUntilLengthBytesReceived)
            {
               break;
            }
         }

         return (unsigned int)totalBytesReceived;
      }

   private:
      SOCKET _hSocket;

      // Thou shalt not copy me
      LinuxSocket(const LinuxSocket&);
      LinuxSocket& operator=(const LinuxSocket&);
};


class LinuxAcceptor : public hmibase::util::socket::Acceptor
{
   public:
      LinuxAcceptor(int hSocket) : _hSocket(hSocket) {}

      virtual ~LinuxAcceptor()
      {
         ::closesocket(_hSocket);
      }

      virtual void close()
      {
         ::closesocket(_hSocket);
      }

      virtual hmibase::util::socket::Socket* accept()
      {
         struct sockaddr clientSocketAddress;
         ::memset(&clientSocketAddress, 0, sizeof(clientSocketAddress));
         socklen_t size = static_cast<socklen_t>(sizeof(clientSocketAddress));

         ETG_TRACE_ERR_THR(("util::tcp Listening..."));
         SOCKET hClientSocket = ::accept(_hSocket, (struct sockaddr*)&clientSocketAddress, &size);
         if (hClientSocket == INVALID_SOCKET)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error accepting socket"));
            return NULL;
         }

         return new LinuxSocket(hClientSocket);
      }

   private:
      SOCKET _hSocket;
};


class LinuxManager : public hmibase::util::socket::Manager
{
   public:
      virtual hmibase::util::socket::Acceptor* bind(unsigned short port)
      {
         ETG_TRACE_ERR_THR(("util::tcp Opening port %u...", port));

#ifdef WIN32
         WSADATA wsaData;
         if (WSAStartup(0x0202, &wsaData) != 0)
         {
            ETG_TRACE_ERR_THR(("util::tcp WSAStartup failed"));
            return NULL;
         }
#endif

         /* First call to socket() function */
         SOCKET hSocket = ::socket(AF_INET, static_cast<int>(SOCK_STREAM), static_cast<int>(IPPROTO_TCP));
         if (hSocket == INVALID_SOCKET)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error creating socket"));
            return NULL;
         }

#ifdef WIN32
         char reuseAddr = 1;
         if (::setsockopt(hSocket, SOL_SOCKET, SO_REUSEADDR, &reuseAddr, sizeof(char)) != 0)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error configuring socket (reuseaddr)"));
            ::closesocket(hSocket);
            return NULL;
         }
#endif
         struct timeval tv;
         tv.tv_sec = 5;
         tv.tv_usec = 0;
         if (::setsockopt(hSocket, SOL_SOCKET, SO_RCVTIMEO, (char*)&tv, sizeof(struct timeval)) != 0)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error configuring socket (timeval)"));
            ::closesocket(hSocket);
            return NULL;
         }

         struct sockaddr_in serverSocketAddress;
         ::memset(&serverSocketAddress, 0, sizeof(serverSocketAddress));
         serverSocketAddress.sin_family = AF_INET;
         serverSocketAddress.sin_addr.s_addr = INADDR_ANY;
         serverSocketAddress.sin_port = htons(port);

         if (::bind(hSocket, reinterpret_cast<struct sockaddr*>(&serverSocketAddress), sizeof(serverSocketAddress)) == -1)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error binding to port %u", port));
            ::closesocket(hSocket);
            return NULL;
         }

         if (::listen(hSocket, 1) == -1)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error listening on port %u", port));
            ::closesocket(hSocket);
            return NULL;
         }

         return new LinuxAcceptor(hSocket);
      }

      virtual hmibase::util::socket::Socket* connect(const char* address, unsigned short port)
      {
         ETG_TRACE_ERR_THR(("util::tcp Connecting to %15s:%u...", address, port));

         SOCKET hSocket = ::socket(AF_INET, static_cast<int>(SOCK_STREAM), static_cast<int>(IPPROTO_TCP));
         if (hSocket == INVALID_SOCKET)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error creating socket"));
            return NULL;
         }

         // Set non-blocking
         if (!setBlocking(hSocket, false))
         {
            ETG_TRACE_ERR_THR(("util::tcp Error configuring socket (blocking(false))"));
            ::closesocket(hSocket);
            return NULL;
         }

         struct sockaddr_in serverSocketAddress;
         ::memset(&serverSocketAddress, 0, sizeof(serverSocketAddress));
         serverSocketAddress.sin_addr.s_addr = ::inet_addr(address);
         serverSocketAddress.sin_family = AF_INET;
         serverSocketAddress.sin_port = htons(port);

         //Connect to remote server
         int res = ::connect(hSocket, reinterpret_cast<struct sockaddr*>(&serverSocketAddress), sizeof(serverSocketAddress));
#ifdef WIN32
         if (res  == -1)
         {
            ETG_TRACE_ERR_THR(("util::tcp Error connecting to %15s:%u", address, port));
            ::closesocket(hSocket);
            return NULL;
         }
#else
         if (res < 0)
         {
            if (errno != EINPROGRESS)
            {
               ETG_TRACE_ERR_THR(("Error connecting %d - %s", errno, ::strerror(errno)));
               ::closesocket(hSocket);
               return NULL;
            }
            else
            {
               fd_set myset;
               struct timeval tv;
               int valopt;

               ETG_TRACE_ERR_THR(("EINPROGRESS in connect() - selecting"));
               do
               {
                  tv.tv_sec = 3;
                  tv.tv_usec = 0;
                  FD_ZERO(&myset);
                  FD_SET(hSocket, &myset);
                  res = ::select(hSocket + 1, NULL, &myset, NULL, &tv);
                  if (res < 0 && errno != EINTR)
                  {
                     ETG_TRACE_ERR_THR(("Error connecting %d - %s", errno, ::strerror(errno)));
                     ::closesocket(hSocket);
                     return NULL;
                  }
                  else if (res > 0)
                  {
                     // Socket selected for write
                     socklen_t lon = sizeof(socklen_t);
                     if (::getsockopt(hSocket, SOL_SOCKET, SO_ERROR, (char*)(&valopt), &lon) < 0)
                     {
                        ETG_TRACE_ERR_THR(("Error in getsockopt() %d - %s", errno, ::strerror(errno)));
                        ::closesocket(hSocket);
                        return NULL;
                     }
                     // Check the value returned...
                     if (valopt)
                     {
                        ETG_TRACE_ERR_THR(("Error in delayed connection() %d - %s", valopt, ::strerror(valopt)));
                        ::closesocket(hSocket);
                        return NULL;
                     }
                     break;
                  }
                  else
                  {
                     ETG_TRACE_ERR_THR(("Timeout in select() - Cancelling!"));
                     ::closesocket(hSocket);
                     return NULL;
                  }
               }
               while (1);
            }
         }
#endif // WIN32

         if (!setBlocking(hSocket, true))
         {
            ETG_TRACE_ERR_THR(("util::tcp Error configuring socket (blocking(true))"));
            ::closesocket(hSocket);
            return NULL;
         }

         return new LinuxSocket(hSocket);
      }

   private:
      bool setBlocking(int hSocket, bool blocking)
      {
#ifdef WIN32
         (void)hSocket;
         (void)blocking;
#else
         long arg = ::fcntl(hSocket, F_GETFL, NULL);
         if (arg < 0)
         {
            ETG_TRACE_ERR_THR(("Error fcntl(..., F_GETFL) (%s)\n", ::strerror(errno)));
            return false;
         }
         if (blocking)
         {
            arg &= (~O_NONBLOCK);
         }
         else
         {
            arg |= O_NONBLOCK;
         }
         if (::fcntl(hSocket, F_SETFL, arg) < 0)
         {
            ETG_TRACE_ERR_THR(("Error fcntl(..., F_SETFL) (%s)\n", ::strerror(errno)));
            return false;
         }
#endif // WIN32
         return true;
      }
};


}//linux_socket

Manager& Manager::getInstance()
{
   static linux_socket::LinuxManager _instance;
   return _instance;
}


}//socket
}//util
}//hmibase
