/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#ifndef WIN32

#include "hmibase/util/SimpleThreadImplPosix.h"
#include "hmibase/util/Error.h"
#include <cstring>
#include <unistd.h>
#include <errno.h>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#include "trcGenProj/Header/SimpleThreadImplPosix.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace util {


SimpleThreadImplPosix::SimpleThreadImplPosix(Threadable& threadable, const SimpleString& name)
   : SimpleThreadImpl(threadable, name)
   , mThread()
{
   std::memset(&mThread, 0, sizeof(pthread_t));
}


SimpleThreadImplPosix::~SimpleThreadImplPosix()
{
   finish();
}


bool SimpleThreadImplPosix::createThread()
{
   ETG_TRACE_USR4_THR(("%s", createMessage("Starting").cPtr()));
   int errorCode = pthread_create(&mThread, NULL, threadRun, this);
   if (errorCode != 0)
   {
      ETG_TRACE_ERR_THR(("%s", createErrorMessage("Cannot create SimpleThreadImplPosix", errorCode).cPtr()));
      return false;
   }

   errorCode = pthread_setname_np(mThread, getName().cPtr());
   if (errorCode != 0)
   {
      ETG_TRACE_ERR_THR(("%s", createErrorMessage("Cannot name SimpleThreadImplPosix", errorCode).cPtr()));
   }

   ETG_TRACE_USR4_THR(("%s", createMessage("Started").cPtr()));
   return true;
}


bool SimpleThreadImplPosix::joinThread()
{
   ETG_TRACE_USR4_THR(("%s", createMessage("Joining").cPtr()));

   int errorCode = pthread_join(mThread, NULL);
   if (errorCode != 0 && errorCode != ESRCH) // ESRCH = "No such process"
   {
      ETG_TRACE_ERR_THR(("%s", createErrorMessage("Cannot join SimpleThreadImplPosix", errorCode).cPtr()));
      return false;
   }
   ETG_TRACE_USR4_THR(("%s", createMessage("Joined").cPtr()));
   return true;
}


void SimpleThreadImplPosix::sleep(unsigned int milliseconds)
{
   if (milliseconds >= 1000)
   {
      ::sleep(milliseconds / 1000);
   }
   else
   {
      usleep(milliseconds * 1000);
   }
}


void* SimpleThreadImplPosix::threadRun(void* selfPtr)
{
   SimpleThreadImplPosix* threadPtr = (SimpleThreadImplPosix*)selfPtr;
   if (threadPtr != NULL)
   {
      threadPtr->localThreadRun();
   }
   return NULL;
}


SimpleString SimpleThreadImplPosix::createMessage(const SimpleString& message)
{
   return SimpleString::format("SimpleThreadImplPosix[%s]: %s", getName().cPtr(), message.cPtr());
}


SimpleString SimpleThreadImplPosix::createErrorMessage(const SimpleString& message, int errorCode)
{
   return createMessage(message + " " + Error::getMessage(errorCode));
}


} // namespace
} // namespace

#endif // !WIN32
