/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/SimpleString.h"
#include <stddef.h>
#include <stdarg.h>
#include <string.h>
#include <stdio.h>
#include <cstdlib>

namespace hmibase {
namespace util {

SimpleString::SimpleString()
   : mStrLen(0), mCharPtr(NULL)
{}

SimpleString::SimpleString(const char* value)
   : mStrLen(0), mCharPtr(NULL)
{
   if (value != NULL)
   {
      size_t strLen = strlen(value);
      copy(value, strLen);
   }
}


SimpleString::SimpleString(const std::string& value)
   : mStrLen(0), mCharPtr(NULL)
{
   copy(value.c_str(), value.length());
}


SimpleString::SimpleString(const SimpleString& value)
   : mStrLen(0), mCharPtr(NULL)
{
   copy(value.mCharPtr, value.mStrLen);
}


SimpleString::~SimpleString()
{
   clear();
}


void SimpleString::clear()
{
   mStrLen = 0;
   if (mCharPtr != NULL)
   {
      free(mCharPtr);
      mCharPtr = NULL;
   }
}


const char* SimpleString::cPtr() const
{
   if (mCharPtr == NULL)
   {
      return "";
   }
   else
   {
      return mCharPtr;
   }
}


SimpleString SimpleString::format(const char* format, ...)
{
   char* charPtr = NULL;
   int   strLen  = 0;

   // Run sprintf with NULL, 0 to determine result length
   va_list argPtr;
   va_start(argPtr, format);
   strLen = vsnprintf(NULL, 0, format, argPtr);
   va_end(argPtr);

   // Create buffer big enough to hold result
   if (strLen > 0)
   {
      charPtr = (char*)calloc(strLen + 1, 1);
      if (charPtr == NULL)
      {
         strLen = 0;
      }
   }

   // Run snprintf again, this time with a real target
   if (strLen > 0)
   {
      va_start(argPtr, format);
      vsnprintf(charPtr, strLen + 1, format, argPtr);
      va_end(argPtr);
      charPtr[strLen] = 0;
   }

   SimpleString result;
   result.set(charPtr, strLen);
   return result;
}


SimpleString& SimpleString::operator=(const SimpleString& right)
{
   if (this != &right)
   {
      copy(right.mCharPtr, right.mStrLen);
   }
   return *this;
}


SimpleString& SimpleString::operator+=(const SimpleString& right)
{
   if (right.isEmpty())
   {
      return *this;
   }

   if (isEmpty())
   {
      *this = right;
      return *this;
   }

   size_t strLen  = mStrLen + right.mStrLen;
   char* charPtr = (char*)calloc(strLen + 1, 1);
   if (charPtr != NULL)
   {
      memcpy(charPtr, mCharPtr, mStrLen);
      memcpy(charPtr + mStrLen, right.mCharPtr, right.mStrLen);
      charPtr[strLen] = 0;

      set(charPtr, strLen);
   }
   return *this;
}


SimpleString& SimpleString::operator<<(const SimpleString& right)
{
   *this += right;
   return *this;
}


SimpleString SimpleString::operator+(const SimpleString& right) const
{
   SimpleString result(*this);
   result += right;
   return result;
}


bool SimpleString::operator==(const SimpleString& right) const
{
   return strcmp(cPtr(), right.cPtr()) == 0;
}


bool SimpleString::operator!=(const SimpleString& right) const
{
   return strcmp(cPtr(), right.cPtr()) != 0;
}


bool SimpleString::operator< (const SimpleString& right) const
{
   return strcmp(cPtr(), right.cPtr()) <  0;
}


bool SimpleString::operator> (const SimpleString& right) const
{
   return strcmp(cPtr(), right.cPtr()) >  0;
}


bool SimpleString::operator<=(const SimpleString& right) const
{
   return strcmp(cPtr(), right.cPtr()) <= 0;
}


bool SimpleString::operator>=(const SimpleString& right) const
{
   return strcmp(cPtr(), right.cPtr()) >= 0;
}


char SimpleString::smErrorChar      = 0;
char SimpleString::smErrorCharConst = 0;

char& SimpleString::operator[](size_t pos)
{
   if (pos < mStrLen)
   {
      return mCharPtr[pos];
   }
   smErrorChar = 0;
   return smErrorChar;
}


const char& SimpleString::operator[](size_t pos) const
{
   if (pos < mStrLen)
   {
      return mCharPtr[pos];
   }
   return smErrorCharConst;
}


void SimpleString::set(char* charPtr, size_t strLen)
{
   clear();
   if (charPtr != NULL && strLen > 0)
   {
      mCharPtr = charPtr;
      mStrLen  = strLen;
   }
}


void SimpleString::copy(const char* charPtr, size_t strLen)
{
   clear();

   if (charPtr != NULL && strLen > 0)
   {
      char* newCharPtr = (char*)calloc(strLen + 1, 1);
      if (newCharPtr != NULL)
      {
         memcpy(newCharPtr, charPtr, strLen);
         newCharPtr[strLen] = 0;

         mCharPtr = newCharPtr;
         mStrLen  = strLen;
      }
   }
}


bool SimpleString::startsWith(const SimpleString& str) const
{
   if (str.length() > length())
   {
      return false;
   }

   for (size_t i = 0; i < str.length(); i++)
   {
      if ((*this)[i] != str[i])
      {
         return false;
      }
   }

   return true;
}


bool SimpleString::endsWith(const SimpleString& str) const
{
   if (str.length() > length())
   {
      return false;
   }

   size_t offset = length() - str.length();
   for (size_t i = 0; i < str.length(); i++)
   {
      if ((*this)[offset + i] != str[i])
      {
         return false;
      }
   }

   return true;
}


bool SimpleString::isWhiteChar(const char c)
{
   switch (c)
   {
      case ' ':
      case '\n':
      case '\r':
      case '\t':
         return true;
      default:
         return false;
   }
}


SimpleString& SimpleString::trim()
{
   trimFront();
   trimBack();
   return *this;
}


SimpleString& SimpleString::trimFront()
{
   if (isEmpty())
   {
      return *this;
   }

   int numWhiteChars = 0;
   while (isWhiteChar(mCharPtr[numWhiteChars]))
   {
      numWhiteChars++;
   }

   if (numWhiteChars > 0)
   {
      SimpleString temp(mCharPtr + numWhiteChars);
      *this = temp;
   }

   return *this;
}


SimpleString& SimpleString::trimBack()
{
   if (isEmpty())
   {
      return *this;
   }

   int numWhiteChars = 0;
   for (size_t i = 0; i < mStrLen; i++)
   {
      if (isWhiteChar(mCharPtr[i]))
      {
         numWhiteChars++;
      }
      else
      {
         numWhiteChars = 0;
      }
   }

   for (int i = 0; i < numWhiteChars; i++)
   {
      mStrLen--;
      mCharPtr[mStrLen] = 0;
   }

   if (mStrLen == 0)
   {
      clear();
   }

   return *this;
}


} // namespace
} // namespace
