/* ***************************************************************************************
* FILE:          Mutex.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Mutex.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "hmibase/util/Mutex.h"

#ifdef WIN32
#include <windows.h>
#include <WinBase.h>
#endif

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Mutex.cpp.trc.h"
#endif


namespace hmibase {
namespace util {

/////////////////////////////////////////////////////////////////////////////////////
///// Mutex Implementation///////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

namespace internal {

#ifndef WIN32
class MutexPosix : public ::hmibase::util::internal::MutexImpl
{
   public:
      MutexPosix(bool processShared)
      {
         int rc = 0;
         pthread_mutexattr_t attr;
         rc  = pthread_mutexattr_init(&attr);
         rc += pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_ERRORCHECK);
         rc += pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);
         if (processShared)
         {
            rc += pthread_mutexattr_setpshared(&attr, PTHREAD_PROCESS_SHARED);
         }
         rc += pthread_mutex_init(&_cs, &attr);
         rc += pthread_mutexattr_destroy(&attr);
         if (rc != 0)
         {
            ETG_TRACE_FATAL_THR(("hmibase::util::Mutex init failed!"));
         }
      }
      virtual ~MutexPosix()
      {
         pthread_mutex_unlock(&_cs);
         pthread_mutex_destroy(&_cs);
      }
      bool lock()
      {
         if (pthread_mutex_lock(&_cs) != 0)
         {
            ETG_TRACE_FATAL_THR(("hmibase::util::Mutex lock failed!"));
            return false;
         }
         return true;
      }
      void unlock()
      {
         if (pthread_mutex_unlock(&_cs) != 0)
         {
            ETG_TRACE_FATAL_THR(("hmibase::util::Mutex unlock failed!"));
         }
      }
   private:
      /*mutable*/
      pthread_mutex_t  _cs;
};


#else

class MutexWindows : public ::hmibase::util::internal::MutexImpl
{
   public:
      MutexWindows(bool /*processShared*/)
      {
         ::InitializeCriticalSection(&_cs);
      }
      virtual ~MutexWindows()
      {
         ::DeleteCriticalSection(&_cs);
      }
      bool lock()
      {
         ::EnterCriticalSection(&_cs);
         return true;
      }
      void unlock()
      {
         ::LeaveCriticalSection(&_cs);
      }
   private:
      CRITICAL_SECTION _cs;
};


#endif // WIN32
} // namespace internal


/////////////////////////////////////////////////////////////////////////////////////
///// Mutex Interface  //////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

Mutex::Mutex(bool processShared) : _impl(0)
{
#ifndef WIN32
   _impl = new ::hmibase::util::internal::MutexPosix(processShared);
#else
   _impl = new ::hmibase::util::internal::MutexWindows(processShared);
#endif
}


Mutex::~Mutex()
{
   if (_impl)
   {
      delete _impl;
   }
   _impl = 0;
}


bool Mutex::lock() const
{
   return _impl ? _impl->lock() : false;
}


void Mutex::unlock() const
{
   if (_impl)
   {
      _impl->unlock();
   }
}


/////////////////////////////////////////////////////////////////////////////////////
///// ScopedMutex /////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////

ScopedMutex::ScopedMutex(Mutex& cs)
{
   _pcs = &cs;
   _pcs->lock();
}


void ScopedMutex::unlock()
{
   if (_pcs)
   {
      _pcs->unlock();
   }
   _pcs = NULL;
}


ScopedMutex::~ScopedMutex()
{
   unlock();
}


}  // namespace util
}  // namespace hmibase
