/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#ifndef WIN32

#include "hmibase/util/MemoryInfoImplPosix.h"
#include "hmibase/util/Error.h"
#include <cstring>
#include <cstdio>

#include "hmibase/util/Trace.h"
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW_UTIL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MemoryInfoImplPosix.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


namespace hmibase {
namespace util {


MemoryInfoImplPosix::MemoryInfoImplPosix()
   : mProcessUsedBytes(0)
{
   refresh();
}


MemoryInfoImplPosix::~MemoryInfoImplPosix()
{
}


void MemoryInfoImplPosix::refresh()
{
   readSysInfo();
   readProcSelf();
}


void MemoryInfoImplPosix::readSysInfo()
{
   std::memset(&mSysInfo, 0, sizeof(mSysInfo));

   if (sysinfo(&mSysInfo) != 0)
   {
      ETG_TRACE_FATAL_THR(("MemoryInfoImplPosix::readSysInfo(): sysinfo call failed (%s)", Error::getMessage().cPtr()));
   }
}


void MemoryInfoImplPosix::readProcSelf()
{
   mProcessUsedBytes = 0;

   FILE* file = fopen("/proc/self/status", "r");
   if (file == NULL)
   {
      ETG_TRACE_FATAL_THR(("MemoryInfoImplPosix::readProcSelf(): Unable to open file /proc/self/status for reading (%s)", Error::getMessage().cPtr()));
      return;
   }

   char line[128];
   int lineSize = (int)sizeof(line);

   std::memset(line, 0, lineSize);

   bool foundLine = false;
   while (fgets(line, lineSize, file) != NULL)
   {
      SimpleString ssLine = line;
      if (ssLine.startsWith("VmRSS:")
            && ssLine.endsWith(" kB\n"))
      {
         foundLine = true;

         for (int i = 0; i < lineSize; i++)
         {
            char c = line[i];

            if (c == 0)
            {
               break;
            }
            else if ('0' <= c && c <= '9')
            {
               mProcessUsedBytes *= 10;
               mProcessUsedBytes += (c - '0');
            }
         }
      }
   }

   fclose(file);

   if (!foundLine)
   {
      ETG_TRACE_FATAL_THR(("MemoryInfoImplPosix::readProcSelf(): Unable to find line \"VmRSS: ... kB\" in /proc/self/status "));
      return;
   }

   // We read kB and have to multiply by 1024 to get bytes
   mProcessUsedBytes <<= 10;
}


unsigned long long MemoryInfoImplPosix::getTotalMemory() const
{
   unsigned long long result = mSysInfo.totalram;
   result *= mSysInfo.mem_unit;
   return result;
}


unsigned long long MemoryInfoImplPosix::getTotalUsedMemory() const
{
   unsigned long long result = mSysInfo.totalram;
   result -= mSysInfo.freeram;
   result *= mSysInfo.mem_unit;
   return result;
}


unsigned long long MemoryInfoImplPosix::getProcessUsedMemory() const
{
   unsigned long long result = mProcessUsedBytes;
   return result;
}


} // namespace
} // namespace

#endif // !WIN32
