/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/util/DltLogger.h"
#include "hmibase/util/DltApplication.h"
#include "hmibase/util/Macros.h"
#include <string.h>
#include <iostream>
#include <unistd.h>

#ifdef VARIANT_S_FTR_ENABLE_ETG_TO_DLT
#include <dlt/dlt.h>
#endif


namespace hmibase {
namespace util {


#ifdef VARIANT_S_FTR_ENABLE_ETG_TO_DLT
DltLogLevelType getDltLogLevelType(DltLogger::LogLevel level)
{
   switch (level)
   {
      case DltLogger::LL_FATAL:
         return DLT_LOG_FATAL;

      case DltLogger::LL_ERROR:
         return DLT_LOG_ERROR;

      case DltLogger::LL_WARNING:
         return DLT_LOG_WARN;

      case DltLogger::LL_INFO:
         return DLT_LOG_INFO;

      case DltLogger::LL_DEBUG:
         return DLT_LOG_DEBUG;

      default:
         return DLT_LOG_FATAL;
   }
}


#endif


std::map<DltName, DltLogger*> DltLogger::smInstancePtrs;

DltLogger& DltLogger::getInstance(const DltName& name, const char* description)
{
//   std::cout << "Creating DltLogger instance " << name.cPtr() << std::endl;
//   std::cout.flush();

   if (smInstancePtrs.find(name) == smInstancePtrs.end())
   {
      smInstancePtrs[name] = new DltLogger(name, description);
   }

   return *(smInstancePtrs[name]);
}


void DltLogger::destroyAllInstances()
{
//   std::cout << "Destroying all DltLogger instances" << std::endl;
//   std::cout.flush();

   for (std::map<DltName, DltLogger*>::iterator iter = smInstancePtrs.begin(); iter != smInstancePtrs.end(); iter++)
   {
      if (iter->second != NULL)
      {
         delete iter->second;
         iter->second = NULL;
      }
   }
   smInstancePtrs.clear();
}


DltLogger::DltLogger(const DltName& name, const char* description)
   : mDltContextPtr(NULL)
   , mName(name)
   , mDescription(description)
{
#ifdef VARIANT_S_FTR_ENABLE_ETG_TO_DLT
   if (mDltContextPtr == NULL)
   {
      DltContext* contextPtr = new DltContext();
      memset(contextPtr, 0, sizeof(DltContext));

      DltReturnValue result = dlt_register_context(contextPtr, mName.cPtr(), mDescription.cPtr());
      if (result == DLT_RETURN_OK)
      {
         mDltContextPtr = contextPtr;
      }
      else
      {
         delete contextPtr;
         mDltContextPtr = NULL;
         std::cout << "dlt_register_context returned " << result << std::endl;
         std::cout.flush();
      }
   }
#endif
}


DltLogger::~DltLogger()
{
#ifdef VARIANT_S_FTR_ENABLE_ETG_TO_DLT
   if (mDltContextPtr != NULL)
   {
      DltContext* contextPtr = (DltContext*)mDltContextPtr;

      DltReturnValue result = dlt_unregister_context(contextPtr);
      delete contextPtr;
      mDltContextPtr = NULL;

      if (result != DLT_RETURN_OK)
      {
         mDltContextPtr = NULL;
         std::cout << "dlt_unregister_context returned " << result << std::endl;
         std::cout.flush();
      }
   }
#endif
}


void DltLogger::debug(const SimpleString& str)
{
   log(LL_DEBUG, str);
}


void DltLogger::info(const SimpleString& str)
{
   log(LL_INFO, str);
}


void DltLogger::warning(const SimpleString& str)
{
   log(LL_WARNING, str);
}


void DltLogger::error(const SimpleString& str)
{
   log(LL_ERROR, str);
}


void DltLogger::fatal(const SimpleString& str)
{
   log(LL_FATAL, str);
}


void DltLogger::log(const DltLogger::LogLevel logLevel, const SimpleString& str)
{
#ifdef VARIANT_S_FTR_ENABLE_ETG_TO_DLT
   //std::cout << "Logging: \"" << str.cPtr() << "\"" << std::endl;
   //std::cout.flush();
   if (mDltContextPtr != NULL)
   {
      DltContext* contextPtr = (DltContext*)mDltContextPtr;

      try
      {
         DltLogLevelType dltLevel = getDltLogLevelType(logLevel);

         if (dlt_user_is_logLevel_enabled(contextPtr, dltLevel) == DLT_RETURN_TRUE)
         {
            DltReturnValue result = dlt_log_string(contextPtr, dltLevel, str.cPtr());
            if (result != DLT_RETURN_OK)
            {
               std::cout << "dlt_log_string returned " << result << std::endl;
               std::cout.flush();
            }
         }
      }
      catch (...)
      {
         std::cout << "dlt_log_string threw an exception" << std::endl;
         std::cout.flush();
      }
   }
#else
   PARAM_UNUSED(logLevel);
   PARAM_UNUSED(str);
#endif
}


} // namespace
} // namespace
