/* ***************************************************************************************
* FILE:          WaylandBackend.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandBackend.cpp is part of HMI-Base testimagedaemon
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "WaylandBackend.h"

#include "EGL/egl.h"
#include "GLES2/gl2.h"
#include "GLES2/gl2ext.h"

#include <wayland-client.h>
#include "wayland-util.h"
#include "compositor-shim.h"

#include <ilm/ilm_client.h>
#include <ilm/ilm_control.h>

#include <string.h>

#include "ImageLoaderDevIL.h"

#include "TestImageDaemon_Trace.h"

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_TESTIMAGEDAEMON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/WaylandBackend.cpp.trc.h"
#endif

namespace hmibase {


WaylandBackend::WaylandBackend(unsigned int screenId, unsigned int layerId, unsigned int surfaceId, unsigned int width, unsigned int height) :
   _screenWidth(width),
   _screenHeight(height),
   _surfaceId(surfaceId),
   _layerId(layerId),
   _screenId(screenId),
   _errorCode(NO_ERROR),
   _imageStatus(IMAGE_UNKNOWN)
{
   // generate layer and surface id or get info from screenbroker
}


WaylandBackend::~WaylandBackend()
{
}


bool WaylandBackend::init()
{
   return egl_open();
}


void WaylandBackend::deinit()
{
   terminate_renderer();
   egl_close();
}


void WaylandBackend::setVisibility(bool activate)
{
   ilmErrorTypes err =  ilm_surfaceSetVisibility(_surfaceId, activate);

   if (err == ILM_SUCCESS)
   {
      err = ilm_commitChanges();
   }

   if (err != ILM_SUCCESS)
   {
      ETG_TRACE_ERR(("WaylandBackend::setVisibility failed"));
   }
}


bool WaylandBackend::makeCurrent()
{
   EGLBoolean ret = eglMakeCurrent(egl_object.egl_display, egl_object.egl_surface,
                                   egl_object.egl_surface, egl_object.egl_context);
   if (ret == EGL_FALSE)
   {
      handle_egl_error("eglMakeCurrent");
      _errorCode = EGL_ERROR;
      return false;
   }

   return true;
}


/**
 * @brief render
 * @param[out] void
 * It redraws a new frame and performs swap.
 */
bool WaylandBackend::render()
{
   _errorCode = NO_ERROR;

   if (_image.empty())
   {
      ETG_TRACE_USR1(("No image set"));
      _errorCode = NO_IMAGE;
      _imageStatus = IMAGE_UNKNOWN;
      //return NO_IMAGE;
   }

   ImageLoaderDevIL::ImageData data;
   ImageLoaderDevIL imageLoader(_image.c_str());
   imageLoader.getImageData(data);

   if (data.buffer)
   {
      if ((data.imageHeight != _screenHeight) && (data.imageWidth != _screenWidth))
      {
         _imageStatus = IMAGE_NOT_FULLSCREEN;
      }
      else
      {
         _imageStatus = IMAGE_OK;
      }
   }
   else
   {
      switch (data.errorCode)
      {
         case ImageLoaderDevIL::UNSUPPORTED_IMAGE_FORMAT:
            _imageStatus = IMAGE_FORMAT_ERROR;
            break;
         case ImageLoaderDevIL::CORRUPTED_FILE:
            _imageStatus = IMAGE_FILE_ERROR;
            break;
         case ImageLoaderDevIL::COULD_NOT_OPEN_FILE:
            _imageStatus = IMAGE_NOT_FOUND;
            break;
         default:
            _imageStatus = UNKNOWN_ERROR;
            break;
      }
      _errorCode = IL_ERROR;
   }

   if (makeCurrent())
   {
      init_renderer(data.buffer, data.imageWidth, data.imageHeight);
      swap_buffer();
   }

   return _errorCode == NO_ERROR;
}


void RegistryHandleGlobal(void* data, struct wl_registry* registry,
                          uint32_t name, const char* interface, uint32_t /*version*/)
{
   WaylandBackend::WaylandGles* sink = (WaylandBackend::WaylandGles*) data;
   if (!strcmp(interface, "wl_compositor"))
   {
      sink->wl_compositor = (struct wl_compositor*)(wl_registry_bind(registry,
                            name, &wl_compositor_interface, 1));
   }
}


static struct wl_registry_listener registryListener =
{
   RegistryHandleGlobal,
   NULL
};


EGLBoolean WaylandBackend::create_wl_context(WaylandGles* sink)
{
   sink->wl_display = wl_display_connect(NULL);
   if (sink->wl_display == NULL)
   {
      ETG_TRACE_ERR(("fail wl_display_connect"));
      return EGL_FALSE;
   }

   sink->wl_registry = wl_display_get_registry(sink->wl_display);

   wl_registry_add_listener(sink->wl_registry, &registryListener, sink);

   wl_display_dispatch(sink->wl_display);
   wl_display_roundtrip(sink->wl_display);

   if (sink->wl_compositor == NULL)
   {
      ETG_TRACE_ERR(("fail wl_compositor"));
      return EGL_FALSE;
   }

   sink->wlAdapterContext = compositor_shim_initialize(sink->wl_display);
   if (NULL == sink->wlAdapterContext)
   {
      ETG_TRACE_ERR(("fail wl compositor_shim_initialize"));
      return EGL_FALSE;
   }

   sink->wl_surface = wl_compositor_create_surface(sink->wl_compositor);
   if (sink->wl_surface == NULL)
   {
      ETG_TRACE_ERR(("fail wl_compositor_create_surface 4"));
      return EGL_FALSE;
   }

   sink->wl_native_window = wl_egl_window_create(sink->wl_surface, _screenWidth, _screenHeight);
   if (sink->wl_native_window == NULL)
   {
      ETG_TRACE_ERR(("fail wl_egl_window_create 5"));
      return EGL_FALSE;
   }

   return EGL_TRUE;
}


/**
 * @brief egl_open
 * @param[in]  void
 * @param[out] void
 * sets up an on-screen EGL rendering surface
 */
bool WaylandBackend::egl_open()
{
   static const EGLint gl_context_attribs[] =
   {
      EGL_CONTEXT_CLIENT_VERSION, 2,
      EGL_NONE
   };

   EGLint s_configAttribs[] =
   {
      EGL_RED_SIZE,        8,
      EGL_GREEN_SIZE,         8,
      EGL_BLUE_SIZE,       8,
      EGL_ALPHA_SIZE,         8,
      EGL_DEPTH_SIZE,         EGL_DONT_CARE,
      EGL_STENCIL_SIZE,    EGL_DONT_CARE,
      EGL_SURFACE_TYPE,    EGL_WINDOW_BIT,
      EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
      EGL_SAMPLES,         EGL_DONT_CARE,
      EGL_NONE
   };
   EGLint numConfigs;
   EGLint majorVersion;
   EGLint minorVersion;

   /* set wayland environment to run without shell script */
   if (NULL == getenv("XDG_RUNTIME_DIR"))
   {
      setenv("XDG_RUNTIME_DIR", "/tmp/", EGL_TRUE);
   }

   WaylandGles* sink = &aWayland;
   int adapter_error = 0;
   sink->surface_id = _surfaceId;
   sink->layer_id = _layerId;

   if (!create_wl_context(sink))
   {
      ETG_TRACE_ERR(("create_wl_context fail"));
      return false;
   }

   egl_object.egl_display = eglGetDisplay((EGLNativeDisplayType)sink->wl_display);

   adapter_error = compositor_shim_surface_init(&sink->wlAdapterSurfaceContext,
                   sink->wl_surface, sink->layer_id, sink->surface_id, _screenWidth, _screenHeight);
   if (adapter_error != 0)
   {
      ETG_TRACE_ERR(("compositor_shim_surface_init() failed"));
      return false;
   }

   adapter_error = compositor_shim_surface_configure(sink->wlAdapterContext,
                   &sink->wlAdapterSurfaceContext, ADAPTER_CONFIGURATION_ALL);
   if (adapter_error != 0)
   {
      ETG_TRACE_ERR(("compositor_shim_surface_configure() failed"));
      return false;
   }

   s_configAttribs[16] =  EGL_NONE;
   eglBindAPI(EGL_OPENGL_ES_API);

   eglInitialize(egl_object.egl_display, &majorVersion, &minorVersion);

   eglGetConfigs(egl_object.egl_display, NULL, 0, &numConfigs);

   eglChooseConfig(egl_object.egl_display, s_configAttribs,
                   & egl_object.eglConfig, 1, &numConfigs);

   egl_object.egl_surface = eglCreateWindowSurface(egl_object.egl_display,
                            egl_object.eglConfig, (EGLNativeWindowType)sink->wl_native_window, NULL);
   if (egl_object.egl_surface == EGL_NO_SURFACE)
   {
      handle_egl_error("eglCreateWindowSurface");
      return false;
   }

   egl_object.egl_context = eglCreateContext(egl_object.egl_display,
                            egl_object.eglConfig, EGL_NO_CONTEXT, gl_context_attribs);
   if (egl_object.egl_context == EGL_NO_CONTEXT)
   {
      handle_egl_error("eglCreateContext");
      return false;
   }
   eglSwapInterval(egl_object.egl_display, 1);

   /*************************************************************/

#ifndef VARIANT_S_FTR_ENABLE_IVI_SHELL
   ilmErrorTypes ilmError = ILM_SUCCESS;
#if NO_SCREENBROKER
   ilmError = ilm_layerAddSurface(sink->layer_id, sink->surface_id);
   handle_ilm_error(ilmError, "ilm_layerAddSurface");

   ilmError = ilm_surfaceSetVisibility(sink->surface_id, false);
   handle_ilm_error(ilmError, "ilm_surfaceSetVisibility");
#endif

   ilmError = ilm_surfaceSetOpacity(sink->surface_id, 1);
   handle_ilm_error(ilmError, "ilm_surfaceSetOpacity");

   ilmError = ilm_surfaceSetDestinationRectangle(sink->surface_id,
              0,
              0,
              _screenWidth,
              _screenHeight);
   handle_ilm_error(ilmError, "ilm_surfaceSetDestinationRectangle");

   ilmError = ilm_surfaceSetSourceRectangle(sink->surface_id,
              0,
              0,
              _screenWidth,
              _screenHeight);
   handle_ilm_error(ilmError, "ilm_surfaceSetSourceRectangle");

   // Do a central commit for all ILM changes applied in prior statements
   ilmError = ilm_commitChanges();
   handle_ilm_error(ilmError, "ilm_commitChanges");

#endif

   return _errorCode == NO_ERROR;
}


void WaylandBackend::handle_ilm_error(ilmErrorTypes err, const char* name)
{
   if (err != ILM_SUCCESS)
   {
      ETG_TRACE_FATAL(("WaylandBackEnd ILMError %d in %s", err, name));
      _errorCode = ILM_ERROR;
   }
}


void WaylandBackend::handle_egl_error(const char* name)
{
   static const char* const error_strings[] =
   {
      "EGL_SUCCESS",
      "EGL_NOT_INITIALIZED",
      "EGL_BAD_ACCESS",
      "EGL_BAD_ALLOC",
      "EGL_BAD_ATTRIBUTE",
      "EGL_BAD_CONFIG",
      "EGL_BAD_CONTEXT",
      "EGL_BAD_CURRENT_SURFACE",
      "EGL_BAD_DISPLAY",
      "EGL_BAD_MATCH",
      "EGL_BAD_NATIVE_PIXMAP",
      "EGL_BAD_NATIVE_WINDOW",
      "EGL_BAD_PARAMETER",
      "EGL_BAD_SURFACE"
   };
   EGLint error_code = eglGetError();

   ETG_TRACE_FATAL((" %25s: egl error (0x%x) \"%s\"", name, error_code, error_strings[error_code - EGL_SUCCESS]));
   ETG_TRACE_ERRMEM((" %25s: egl error (0x%x) \"%s\"", name, error_code, error_strings[error_code - EGL_SUCCESS]));
   //exit(EXIT_FAILURE);
}


/**
 * @brief egl_close
 * @param[in]  void
 * @param[out] void
 * destroys on-screen EGL rendering surface
 */
void WaylandBackend::egl_close(void)
{
   eglMakeCurrent(egl_object.egl_display, NULL, NULL, NULL);
   eglDestroyContext(egl_object.egl_display, egl_object.egl_context);
   eglDestroySurface(egl_object.egl_display, egl_object.egl_surface);
   eglTerminate(egl_object.egl_display);

   WaylandGles* sink = &aWayland;
   wl_display_flush(sink->wl_display);
   wl_display_roundtrip(sink->wl_display);
   compositor_shim_terminate(sink->wlAdapterContext);

   if (sink->wl_native_window)
   {
      wl_egl_window_destroy(sink->wl_native_window);
   }

   if (sink->wl_surface)
   {
      wl_surface_destroy(sink->wl_surface);
   }

   if (sink->wl_compositor)
   {
      wl_compositor_destroy(sink->wl_compositor);
   }
   if (sink->wl_display)
   {
      wl_display_disconnect(sink->wl_display);
   }
}


/**
 * @brief swap_buffer
 *
 *@param[in] void
 *@param[out] void
 * swap the buffers
 */
void WaylandBackend::swap_buffer()
{
   eglSwapBuffers(egl_object.egl_display, egl_object.egl_surface);
   //sleep(4);
}


typedef struct _es2shaderobject
{
   GLuint fragmentShaderId;
   GLuint vertexShaderId;
   GLuint shaderProgramId;
} ES2ShaderObject;

typedef struct _es2vertexbufferobject
{
   GLuint vbo;
} ES2VertexBuffer;

static ES2ShaderObject  gShaderObject;
static ES2VertexBuffer  gVertexBuffer;
static   GLuint uTexToRender;

// Vertex shader code
const char* pszVertShaTexMap = "\
      attribute highp vec4    vertPos;\
      uniform mediump mat4    MVMatrix;\
      attribute highp vec2    tex_coord;\
      varying highp vec2      tex_lookup;\
        void main(void)\
      {\
         tex_lookup = tex_coord;\
         gl_Position = vertPos * MVMatrix;\
      }";

// Fragment shader code
const char* pszFragShaTexMap = "\
      uniform mediump sampler2D    texobj;\
      varying highp vec2   tex_lookup;\
      precision mediump float;\
      void main(void)\
      {\
      gl_FragColor = texture2D(texobj, tex_lookup);\
        }";

bool WaylandBackend::init_shader()
{
   static bool initialized = false;
   GLint result = 0;
   char pszInfoLog[1024];
   int  nInfoLogLength;

   if (!initialized)
   {
      // Create the vertex shader object
      gShaderObject.vertexShaderId = glCreateShader(GL_VERTEX_SHADER);
      glShaderSource(gShaderObject.vertexShaderId, 1,
                     (const char**)&pszVertShaTexMap, NULL);
      glCompileShader(gShaderObject.vertexShaderId);

      glGetShaderiv(gShaderObject.vertexShaderId, GL_COMPILE_STATUS, &result);
      if (!result)
      {
         ETG_TRACE_ERR(("Error: Failed to compile GLSL shader"));
         glGetShaderInfoLog(gShaderObject.vertexShaderId, 1024, &nInfoLogLength, pszInfoLog);
         ETG_TRACE_ERR(("%s", pszInfoLog));
         return false;
      }
      // Create the fragment shader object
      gShaderObject.fragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
      glShaderSource(gShaderObject.fragmentShaderId, 1,
                     (const char**)&pszFragShaTexMap, NULL);
      glCompileShader(gShaderObject.fragmentShaderId);

      glGetShaderiv(gShaderObject.fragmentShaderId, GL_COMPILE_STATUS, &result);
      if (!result)
      {
         ETG_TRACE_ERR(("Error: Failed to compile GLSL shader"));
         glGetShaderInfoLog(gShaderObject.fragmentShaderId, 1024, &nInfoLogLength, pszInfoLog);
         ETG_TRACE_ERR(("%s", pszInfoLog));
         return false;
      }

      gShaderObject.shaderProgramId = glCreateProgram();
      glAttachShader(gShaderObject.shaderProgramId,
                     gShaderObject.fragmentShaderId);
      glAttachShader(gShaderObject.shaderProgramId,
                     gShaderObject.vertexShaderId);
      glLinkProgram(gShaderObject.shaderProgramId);

      glGetProgramiv(gShaderObject.shaderProgramId, GL_LINK_STATUS, &result);
      if (!result)
      {
         ETG_TRACE_ERR(("Error: Failed to validate GLSL program"));
         glGetProgramInfoLog(gShaderObject.shaderProgramId, 1024, &nInfoLogLength, pszInfoLog);
         ETG_TRACE_ERR(("%s", pszInfoLog));
         return false;
      }
      initialized = true;
   }

   glUseProgram(gShaderObject.shaderProgramId);

   return true;
}


void WaylandBackend::init_renderer(uint8_t* pTex, int width, int height)
{
   if (!init_shader())
   {
      ETG_TRACE_ERR(("Error: Failed InitShader"));
      return;
   }

   if (pTex)
   {
      glViewport(0, 0, width, height);
      GLint tmapvPos;
      GLint tmapTexPos;
      GLint MVmapPos;
      GLint texsmapler;
      GLfloat vertsForTexMap[] =
      {
         -1.0f, -1.0f, 0.0f,
         1.0f, -1.0f, 0.0f,
         -1.0f,  1.0f, 0.0f,
         1.0f,  1.0f, 0.0f,
         0.0f,  0.0f, 0.0f,
         1.0f,  0.0f, 0.0f,
         0.0f,  1.0f, 1.0f,
         1.0f,  1.0f, 0.0f
      };

      float pfIdentity[4][4] =
      {
         {1.0f, 0.0f, 0.0f, 0.0f},
         {0.0f, 1.0f, 0.0f, 0.0f},
         {0.0f, 0.0f, 1.0f, 0.0f},
         {0.0f, 0.0f, 0.0f, 1.0f}
      };

      //if (!init_shader())
      //{
      //   printf("Error: Failed InitShader\n");
      //}
      glUseProgram(gShaderObject.shaderProgramId);

      eglSwapInterval(egl_object.egl_display, 1);
      glGenTextures(1, &uTexToRender);
      glActiveTexture(GL_TEXTURE0);
      glBindTexture(GL_TEXTURE_2D, uTexToRender);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
      glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
      glTexImage2D(GL_TEXTURE_2D, 0, GL_BGRA_EXT, width, height, 0, GL_BGRA_EXT,
                   GL_UNSIGNED_BYTE, pTex);

      glGenBuffers(1, &gVertexBuffer.vbo);
      glBindBuffer(GL_ARRAY_BUFFER, gVertexBuffer.vbo);
      glBufferData(GL_ARRAY_BUFFER, sizeof(vertsForTexMap), vertsForTexMap,
                   GL_STATIC_DRAW);

      glClearColor(0.0, 1.0, 1.0, 1.0);
      glClear(GL_COLOR_BUFFER_BIT);

      tmapvPos = glGetAttribLocation(gShaderObject.shaderProgramId, "vertPos");
      glEnableVertexAttribArray(tmapvPos);
      glVertexAttribPointer(tmapvPos, 3, GL_FLOAT, GL_FALSE, 0, (GLvoid*)0);

      tmapTexPos = glGetAttribLocation(gShaderObject.shaderProgramId, "tex_coord");
      glEnableVertexAttribArray(tmapTexPos);
      glVertexAttribPointer(tmapTexPos, 3, GL_FLOAT, GL_FALSE, 0,
                            (GLvoid*)((sizeof(GLfloat)) * 12));

      MVmapPos = glGetUniformLocation(gShaderObject.shaderProgramId, "MVMatrix");
      glUniformMatrix4fv(MVmapPos, 1, GL_FALSE, (GLfloat*)pfIdentity);

      texsmapler = glGetUniformLocation(gShaderObject.shaderProgramId, "texobj");
      glUniform1i(texsmapler, 0);

      glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
      glFinish();
   }
   else
   {
      glViewport(0, 0, _screenWidth, _screenHeight);
      glClearColor(0.0, 1.0, 0.0, 1.0);
      glClear(GL_COLOR_BUFFER_BIT);
      glFinish();
   }
}


void WaylandBackend::terminate_renderer()
{
   glDeleteProgram(gShaderObject.shaderProgramId);
   glDeleteShader(gShaderObject.fragmentShaderId);
   glDeleteShader(gShaderObject.vertexShaderId);
   glDeleteBuffers(1, &gVertexBuffer.vbo);
   glDeleteTextures(1, &uTexToRender);
}


}
