/* ***************************************************************************************
* FILE:          ImageLoaderDevIL.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ImageLoaderDevIL.cpp is part of HMI-Base testimagedaemon
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "ImageLoaderDevIL.h"

#include <IL/il.h>
#include "IL/ilu.h"

#include "TestImageDaemon_Trace.h"

#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_TESTIMAGEDAEMON
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ImageLoaderDevIL.cpp.trc.h"
#endif

namespace hmibase {

bool ImageLoaderDevIL::_isILInitialized = false;

ImageLoaderDevIL::ImageLoaderDevIL(const std::string& fileName) : _handle(0)
{
   _imageData.fileName = fileName;
   // create an image
   if (!_isILInitialized)
   {
      _isILInitialized = true;
      ilInit();
      iluInit();
   }

   ilGenImages(1, &_handle);
   if (_handle != 0)
   {
      ilBindImage(_handle);
      loadImage();
   }
}


ImageLoaderDevIL::~ImageLoaderDevIL()
{
   if (_imageData.buffer != 0)
   {
      delete[] _imageData.buffer;
      _imageData.buffer = 0;
   }

   if (_handle != 0)
   {
      ilDeleteImages(1, &_handle);
   }
}


void ImageLoaderDevIL::loadImage()
{
   _imageData.errorCode = NO_ERROR;

   if (ilLoadImage(_imageData.fileName.c_str()) == IL_TRUE)
   {
      if (ilEnable(IL_ORIGIN_SET) == IL_TRUE)
      {
         if (ilOriginFunc(IL_ORIGIN_LOWER_LEFT) == IL_TRUE)
         {
            if (ilConvertImage(IL_RGBA, IL_UNSIGNED_BYTE) == IL_TRUE)
            {
               _imageData.imageWidth = static_cast<ILuint>(ilGetInteger(IL_IMAGE_WIDTH));
               _imageData.imageHeight = static_cast<ILuint>(ilGetInteger(IL_IMAGE_HEIGHT));
               _imageData.depth = static_cast<ILuint>(ilGetInteger(IL_IMAGE_DEPTH));
               _imageData.bpp = static_cast<ILuint>(ilGetInteger(IL_IMAGE_BPP));
               _imageData.imageSize = static_cast<ILuint>(ilGetInteger(IL_IMAGE_SIZE_OF_DATA));

               ILuint format = static_cast<ILuint>(ilGetInteger(IL_IMAGE_FORMAT));
               ILuint type = static_cast<ILuint>(ilGetInteger(IL_IMAGE_TYPE));

               if ((format == IL_RGBA) && (type == IL_UNSIGNED_BYTE))
               {
                  if (_imageData.buffer != 0)
                  {
                     delete[] _imageData.buffer;
                     _imageData.buffer = 0;
                  }
                  //size_t bufferSize = static_cast<size_t>(_imageData.imageWidth * _imageData.imageHeight * _imageData.bpp);
                  _imageData.buffer = new uint8_t[_imageData.imageSize];
                  if (_imageData.buffer != NULL)
                  {
                     if (ilCopyPixels(0, 0, 0, _imageData.imageWidth, _imageData.imageHeight, _imageData.depth, format, type, (void*)_imageData.buffer) == 0)
                     {
                        handleILError("ilCopyPixels");
                     }
                  }
               }
               else
               {
                  ETG_TRACE_ERR(("Image format or type error, format %u != IL_RGBA (%u), type %u != IL_UNSIGNED_BYTE (%u)", format, IL_RGBA, type, IL_UNSIGNED_BYTE));
                  _imageData.errorCode = UNSUPPORTED_IMAGE_FORMAT;
               }
            }
            else
            {
               handleILError("ilConvertImage");
            }
         }
         else
         {
            handleILError("ilOriginFunc");
         }
      }
      else
      {
         handleILError("ilEnable");
      }
   }
   else
   {
      handleILError("ilLoadImage");
   }
}


void ImageLoaderDevIL::getImageData(ImageData& d) const
{
   d = _imageData;
}


void ImageLoaderDevIL::handleILError(const char* name)
{
   ILenum errorCode = ilGetError();
   if (errorCode != IL_NO_ERROR)
   {
      ETG_TRACE_ERR((" %25s: IL error 0x%x, file %s", name, errorCode, _imageData.fileName.c_str()));

      switch (errorCode)
      {
         case IL_FORMAT_NOT_SUPPORTED:
         case IL_INVALID_EXTENSION:
            _imageData.errorCode = UNSUPPORTED_IMAGE_FORMAT;
            break;
         case IL_COULD_NOT_OPEN_FILE:
            _imageData.errorCode = COULD_NOT_OPEN_FILE;
            break;
         case IL_INVALID_VALUE:
            _imageData.errorCode = CORRUPTED_FILE;
            break;
         default:
            _imageData.errorCode = INTERNAL_ERROR;
            break;
      }
   }
}


}
