/******************************************************************
*COPYRIGHT: (C) 2017 Robert Bosch GmbH
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/
#include "hmibase/gadget/syncblock2/socks/Message.h"
#include <sys/socket.h>
#include <cstring>

#include "hmibase/util/Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_HMI_FW
#include "trcGenProj/Header/Message.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

namespace hmibase {
namespace gadget {
namespace syncblock2 {
namespace socks {

Message::Message(unsigned int dataSize, unsigned int numFds)
   : mNumFds(0)
   , mDataSize(0)
   , mMsgPtr(NULL)
   , mIovecPtr(NULL)
   , mPayloadPtr(NULL)
   , mRawFdsPtr(NULL)
   , mAncillaryPtr(NULL)

{
   // Create Payload Buffer
   if (dataSize > 0)
   {
      mDataSize   = dataSize;
      mPayloadPtr = new char[dataSize];
      memset(mPayloadPtr, 0, dataSize);
   }

   // Create buffer for untranslated fds
   if (numFds > 0)
   {
      mRawFdsPtr = new int[numFds];
      memset(mRawFdsPtr, 0, sizeof(int) * numFds);
   }

   // Create Containers for payload and raw fds
   mIovecPtr = new iovec[2];
   mIovecPtr[0].iov_base = mPayloadPtr;
   mIovecPtr[0].iov_len  = mDataSize;
   mIovecPtr[1].iov_base = mRawFdsPtr;
   mIovecPtr[1].iov_len  = sizeof(int) * numFds;

   // Create message struct
   mMsgPtr = new msghdr();
   mMsgPtr->msg_name       = NULL;          // optional address
   mMsgPtr->msg_namelen    = 0;             // size of address
   mMsgPtr->msg_iov        = mIovecPtr;     // scatter/gather array
   mMsgPtr->msg_iovlen     = 2;             // # elements in msg_iov
   mMsgPtr->msg_control    = NULL;          // ancillary data, see below
   mMsgPtr->msg_controllen = 0;             // ancillary data buffer len
   mMsgPtr->msg_flags      = 0;             // flags on received message

   mNumFds = numFds;

#if defined (GEN3ARM) || defined (GEN4ARM)
   if (numFds > 0)
   {
      // Create Ancillary Buffer for file descriptors
      int ancillarySize = CMSG_SPACE(sizeof(int) * numFds);
      mAncillaryPtr     = new char[ancillarySize];
      memset(mAncillaryPtr, 0, ancillarySize);

      mMsgPtr->msg_control    = mAncillaryPtr;   // ancillary data, see below
      mMsgPtr->msg_controllen = ancillarySize;   // ancillary data buffer len

      // Do some Macro Magic to initialize the contents of ancillary.
      cmsghdr* cmsgPtr    = CMSG_FIRSTHDR(mMsgPtr);
      cmsgPtr->cmsg_level = SOL_SOCKET;
      cmsgPtr->cmsg_type  = SCM_RIGHTS;
      cmsgPtr->cmsg_len   = CMSG_LEN(sizeof(int) * numFds);

      // Set controllen again, but this time including alignment
      mMsgPtr->msg_controllen = cmsgPtr->cmsg_len;  // ancillary data buffer len
   }
#endif

   // Initialize file descriptors
   for (unsigned int i = 0; i < numFds; i++)
   {
      setFd(i, -1);
   }
}


Message::~Message()
{
   mNumFds   = 0;
   mDataSize = 0;

   if (mMsgPtr != NULL)
   {
      delete mMsgPtr;
      mMsgPtr = NULL;
   }

   if (mIovecPtr != NULL)
   {
      delete[] mIovecPtr;
      mIovecPtr = NULL;
   }

   if (mPayloadPtr != NULL)
   {
      delete[] mPayloadPtr;
      mPayloadPtr = NULL;
   }

   if (mRawFdsPtr != NULL)
   {
      delete[] mRawFdsPtr;
      mRawFdsPtr = NULL;
   }

   if (mAncillaryPtr != NULL)
   {
      delete[] mAncillaryPtr;
      mAncillaryPtr = NULL;
   }
}


msghdr* Message::getMsgPtr()
{
   return mMsgPtr;
}


int Message::getDataSize() const
{
   return mDataSize;
}


void* Message::getDataPtr() const
{
   if (mPayloadPtr != NULL)
   {
      return mPayloadPtr;
   }

   return NULL;
}


int Message::getNumFds() const
{
   return mNumFds;
}


int Message::getFd(int index) const
{
   int* fdPtr = getFdPtr(index);
   if (fdPtr != NULL)
   {
      return *fdPtr;
   }
   return -1;
}


void Message::setFd(int index, int fd)
{
   int* fdPtr = getFdPtr(index);
   if (fdPtr != NULL)
   {
      *fdPtr = fd;
   }
}


int* Message::getFdPtr(int index) const
{
   if (index < 0 || mNumFds <= index)
   {
      return NULL;
   }

#if defined (GEN3ARM) || defined (GEN4ARM)
   // Some Macro Magic to get FileDescriptor from ancillary message data
   cmsghdr* cmsgPtr = CMSG_FIRSTHDR(mMsgPtr);
   if (cmsgPtr == NULL)
   {
      return NULL;
   }

   if (cmsgPtr->cmsg_len < CMSG_LEN(sizeof(int) * index))
   {
      return NULL;
   }

   return ((int*)CMSG_DATA(cmsgPtr)) + index;
#else
   return mRawFdsPtr + index;
#endif
}


} // namespace
} // namespace
} // namespace
} // namespace
