/* ***************************************************************************************
* FILE:          WaylandContext.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WaylandContext.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBrokerPlugins_KeyHandler_WaylandContext_h)
#define ScreenBrokerPlugins_KeyHandler_WaylandContext_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <Shared/WaylandInc.h>
#include <Shared/WaylandEventHook.h>
#include <Shared/WaylandServerinfoClientProtocol.h>

namespace ScreenBroker {
/** Holds the current status of the wayland communication */
class WaylandContext
{
   public:
      ///
      WaylandContext();

      ///
      virtual ~WaylandContext();

      /** Initializes the input handling specific wayland context.
          @param displayId Wayland display ID (shall be NULL with layer manager).
          @param data User specific data, channelled to user specific eventhandler.
          @param eventHook User specific event handling.
          @return <em>true</em>, if initialization was successful,
                  <em>false</em>, otherwise.*/
      bool Init(const char* displayId,
                void* data,
                WaylandEventHook* eventHook = 0);

      ///
      void Destroy();

      ///
      void Listen();

      ///
      struct wl_display* GetDisplay();

      ///
      struct wl_registry* GetRegistry()
      {
         return mRegistry;
      }

      ///
      struct serverinfo* GetServerInfo()
      {
         return mServerInfo;
      }

      /**
       * @brief Returns the wayland compositor, which is used to create wayland surfaces.
       * @return The wayland compositor for wayland display of the context object.
       */
      struct wl_compositor* GetCompositor()
      {
         return mCompositor;
      }

      /**
       * @brief Returns the connection ID to the wayland server, which is used to calculate
       *        the native handle, needed to create layer manager surfaces.
       * @return The connection Id to the wayland server.
       */
      uint32_t GetConnectionId() const
      {
         return mConnectionId;
      }

      ///
      struct wl_seat* GetSeat()
      {
         return mSeat;
      }

      ///
      struct wl_keyboard* GetKeyboard()
      {
         return mKeyboard;
      }

      ///
      wl_surface* GetSurface()
      {
         return mSurface;
      }

      ///
      void* GetData()
      {
         return mData;
      }

      ///
      WaylandEventHook* GetEventHook()
      {
         return mEventHook;
      }

   private:
      friend class WaylandEventHandler;

      struct wl_display* mDisplay;
      struct wl_registry* mRegistry;
      struct wl_event_queue* mEventQueue;
      struct wl_compositor* mCompositor;
      struct serverinfo* mServerInfo;
      uint32_t mConnectionId;
      struct wl_seat* mSeat;
      struct wl_keyboard* mKeyboard;
      struct wl_surface* mSurface;
      void* mData;
      WaylandEventHook* mEventHook;
      CriticalSection mCs;

      ///
      void SetCompositor(struct wl_compositor* compositor)
      {
         mCompositor = compositor;
      }

      ///
      void SetServerInfo(struct serverinfo* serverInfo)
      {
         mServerInfo = serverInfo;
      }

      ///
      void SetConnectionId(uint32_t connectionId)
      {
         mConnectionId = connectionId;
      }

      ///
      void SetSeat(struct wl_seat* seat)
      {
         mSeat = seat;
      }

      ///
      void SetKeyboard(struct wl_keyboard* keyboard)
      {
         mKeyboard = keyboard;
      }
};


}

#endif
