/* ***************************************************************************************
* FILE:          PopupFilter.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  PopupFilter.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBrokerPlugins_Shared_PopupFilter_h)
#define ScreenBrokerPlugins_Shared_PopupFilter_h

#include <ScreenBroker/PopupPresentationArg.h>
#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>
#include <list>
#include <map>

#ifndef TESTER_DECLARE_FRIEND
#define TESTER_DECLARE_FRIEND
#endif

#ifndef TESTER_DEFINE_FRIEND
#define TESTER_DEFINE_FRIEND
#endif

TESTER_DECLARE_FRIEND
namespace ScreenBroker {
///
class PopupFilter
{
   public:
      ///
      static PopupFilter& GetInstance();

      ///
      PopupFilter();

      ///
      void Insert(UInt32 displayId,
                  UInt32 appId,
                  const PopupPresentationArg& filterArg,
                  bool disableAll = false);

      ///
      void Delete(UInt32 displayId,
                  UInt32 appId);

      /**
       * @brief On successful return the out parameter
       * <em>popupPresentationArg</em> is filled with the application's
       * filter data.
       * @param displayId Display for which the filter data is requested.
       * @param appId Application ID for which the filter data is requested.
       * @param popupPresentationArg Filter information stored within
       * the popup presentation structure. This out parameter is only valid,
       * if the method returns <em>true</em>.
       * @return True, if application has set filter given with the out
       * parameter popupPresentationArg.
       */
      bool Get(UInt32 displayId,
               UInt32 appId,
               PopupPresentationArg& popupPresentationArg) const;

      /**
       * @brief Informs if the "disable all" filter flag is set for given application.
       * @param displayId Display for which the "disable all" filter flag is requested.
       * @param appId Application ID for which the "disable all" filter flag is requested.
       * @return True, if application has set the "disable all" filter flag.
       */
      bool IsDisableAll(UInt32 displayId,
                        UInt32 appId) const;

      ///
      bool Exists(UInt32 displayId,
                  UInt32 appId) const;

      ///
      void Dump();

      /**
       * @brief Completely reset the internal states.
       */
      void Reset();

   private:
      TESTER_DEFINE_FRIEND
      ///
      class Properties
      {
         public:
            Properties() : mDisableAll(false) {}
            Properties(const PopupPresentationArg& popupPresentationArg,
                       bool disableAll)
               : mPopupPresentationArg(popupPresentationArg),
                 mDisableAll(disableAll) {}
            const PopupPresentationArg& GetPopupPresentationArg() const
            {
               return mPopupPresentationArg;
            }
            bool GetDisableAll() const
            {
               return mDisableAll;
            }
         private:
            PopupPresentationArg mPopupPresentationArg;
            bool mDisableAll;
      };
      typedef std::map<UInt32, Properties> AppFilterMap;

      typedef std::map<UInt32, AppFilterMap> DisplayAppFilterMap;
      DisplayAppFilterMap mDisplayAppFilterMap;
};


}

#endif
