/* ***************************************************************************************
* FILE:          FocusHandler.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FocusHandler.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBrokerPlugins_Shared_FocusHandler_h)
#define ScreenBrokerPlugins_Shared_FocusHandler_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>
#include <Shared/FocusList.h>
#include <list>
#include <map>
#include <vector>

namespace ScreenBroker {
class FocusHandler
{
   public:
      ///
      static FocusHandler& GetInstance();

      ///
      FocusHandler();

      ///
      UInt32 GetKeyboardFocus() const;

      ///
      void SetKeyboardFocus(UInt32 surfaceId);

      ///
      UInt32 GetPointerFocus() const;

      ///
      void SetPointerFocus(UInt32 surfaceId);

      ///
      Int32 GetKeyboardPriority(UInt32 surfaceId);

      ///
      void SetKeyboardPriority(UInt32 surfaceId, Int32 priority);

      ///
      Int32 GetPointerPriority(UInt32 surfaceId);

      ///
      void SetPointerPriority(UInt32 surfaceId, Int32 priority);

      ///
      void ForceKeyboardFocus(UInt32 surfaceId, Int32 priority);

      ///
      void ForcePointerFocus(UInt32 surfaceId, Int32 priority);

      ///
      UInt32 GetKeyboardFocusSize() const;

      ///
      UInt32 GetPointerFocusSize() const;

      ///
      void RemoveSurface(UInt32 surfaceId);

      ///
      void PushBackSurface(UInt32 surfaceId);

      /** Sets the keyboard focus to the given surface, if the surface is capable of receiving keyboard input events.
       *  @param surfaceId The surface for which the keyboard focus shall set.
       *  @return True if successful, false otherwise. */
      static bool UpdateKeyboardFocus(UInt32 surfaceId);

      /** Sets the pointer focus to the given surface, if the surface is capable of receiving pointer input events.
       *  @param surfaceId The surface for which the pointer focus shall set.
       *  @return True if successful, false otherwise.
       */
      static bool UpdatePointerFocus(UInt32 surfaceId);

      /** Enables/disables input events according ProjectSettings.
       *  @param surfaceId The surface for which the input event reception shall be manipulated.
       *  @return True if successful, false otherwise.
       */
      static bool EnableInputEvents(UInt32 surfaceId, bool enable);

      /** Transfers the various input focus according to the current surface and popup states.
       */
      bool TransferInputFocus();

      /** @brief Update the input acceptance of the involved surfaces according to the feature
       *  for disabling input acceptances for either surfaces resp. popup surfaces.
       */
      static void UpdateInputAcceptance(UInt32 surfaceId, bool enable);

      /**
       * @brief Updates the input foci (keyboard, pointer) according ProjectSettings with respect to the
       * input acceptance per surface and the feature for disabling input acceptance
       * for underlying elements (like popups or surfaces).
       * @param surfaceId Newly requested surface ID, for which the input acceptance
       * is enabled according the ProjectSettings.
       */
      bool UpdateInputFocus(UInt32 surfaceId);

      /**
       * @brief RemoveSurfaces Removes surfaces from focus lists
       * @param popup If popup surfaces or other surfaces should be removed
       */
      void RemoveSurfaces(bool popup = false);

      ///
      void Dump() const;

      ///
      void FilterKeyboardFocus(UInt32 displayId);

      ///
      void FilterPointerFocus(UInt32 displayId);
   private:
      struct ForcedFocus
      {
         UInt32 mSurfaceId;
         Int32 mPriority;

         ForcedFocus() :
            mSurfaceId(0),
            mPriority(0) { }
      };

      typedef std::map<UInt32, Int32> PriorityMap;

      ///
      UInt32 TransferInputKeyboardFocus(UInt32 displayId);

      ///
      UInt32 TransferInputPointerFocus(UInt32 displayId);

      ///
      void UpdateSeatAcceptance(UInt32 surfaceId) const;

      /// Focus history list of keyboard focus
      FocusList mKeyboardFocusList;

      ///
      PriorityMap mKeyboardFocusPriority;

      ///
      ForcedFocus mForcedKeyboardFocusPriority;

      /// Focus history list of pointer focus
      FocusList mPointerFocusList;

      ///
      PriorityMap mPointerFocusPriority;

      ///
      ForcedFocus mForcedPointerFocusPriority;

      ///
      std::vector<UInt32> mFilteredKeyboardFocus;

      ///
      std::vector<UInt32> mFilteredPointerFocus;
};


}

#endif
