/* ***************************************************************************************
* FILE:          FocusHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FocusHandler.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "FocusHandler.h"
#include "IlmAccessor.h"

#include <ScreenBrokerSettings/Settings.h>
#include <algorithm>
#include <ScreenBroker/Service/ServiceApi.h>
#include <ScreenLayouter/ScreenLayouter.h>
#include <Shared/ActivePopups.h>
#include <Shared/ActiveSurfaces.h>
#include <Shared/PluginActions.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/FocusHandler.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Focus);

// ------------------------------------------------------------------------
// (Plugin) global handled FocusHandler which has to be instantiated once for all plugins
// This is normally done in ScreenBrokerActivator.cpp.
extern FocusHandler gFocusHandlerInstance;

// ------------------------------------------------------------------------
FocusHandler& FocusHandler::GetInstance()
{
   return gFocusHandlerInstance;
}


// ------------------------------------------------------------------------
FocusHandler::FocusHandler()
{
}


// ------------------------------------------------------------------------
UInt32 FocusHandler::GetKeyboardFocus() const
{
   return mKeyboardFocusList.GetFocus();
}


// ------------------------------------------------------------------------
void FocusHandler::SetKeyboardFocus(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("Set keyboard focus to surface %u", surfaceId));
   mKeyboardFocusList.SetFocus(surfaceId);
}


// ------------------------------------------------------------------------
UInt32 FocusHandler::GetPointerFocus() const
{
   return mPointerFocusList.GetFocus();
}


// ------------------------------------------------------------------------
void FocusHandler::SetPointerFocus(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("Set pointer focus to surface %u", surfaceId));
   mPointerFocusList.SetFocus(surfaceId);
}


// ------------------------------------------------------------------------
Int32 FocusHandler::GetKeyboardPriority(UInt32 surfaceId)
{
   return mKeyboardFocusPriority[surfaceId];
}


// ------------------------------------------------------------------------
void FocusHandler::SetKeyboardPriority(UInt32 surfaceId, Int32 priority)
{
   mKeyboardFocusPriority[surfaceId] = priority;
}


// ------------------------------------------------------------------------
Int32 FocusHandler::GetPointerPriority(UInt32 surfaceId)
{
   return mPointerFocusPriority[surfaceId];
}


// ------------------------------------------------------------------------
void FocusHandler::SetPointerPriority(UInt32 surfaceId, Int32 priority)
{
   mPointerFocusPriority[surfaceId] = priority;
}


// ------------------------------------------------------------------------
void FocusHandler::ForceKeyboardFocus(UInt32 surfaceId, Int32 priority)
{
   mForcedKeyboardFocusPriority.mSurfaceId = surfaceId;
   mForcedKeyboardFocusPriority.mPriority = priority;
}


// ------------------------------------------------------------------------
void FocusHandler::ForcePointerFocus(UInt32 surfaceId, Int32 priority)
{
   mForcedPointerFocusPriority.mSurfaceId = surfaceId;
   mForcedPointerFocusPriority.mPriority = priority;
}


// ------------------------------------------------------------------------
UInt32 FocusHandler::GetKeyboardFocusSize() const
{
   return mKeyboardFocusList.Size();
}


// ------------------------------------------------------------------------
UInt32 FocusHandler::GetPointerFocusSize() const
{
   return mPointerFocusList.Size();
}


// ------------------------------------------------------------------------
void FocusHandler::RemoveSurface(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("Remove surface %u from focus lists", surfaceId));

   mKeyboardFocusList.Remove(surfaceId);
   mPointerFocusList.Remove(surfaceId);

   if (surfaceId == mForcedKeyboardFocusPriority.mSurfaceId)
   {
      ForceKeyboardFocus(0, 0);
   }

   if (surfaceId == mForcedPointerFocusPriority.mSurfaceId)
   {
      ForcePointerFocus(0, 0);
   }
}


// ------------------------------------------------------------------------
void FocusHandler::PushBackSurface(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("PushBack surface %u to focus lists", surfaceId));

   mKeyboardFocusList.PushBack(surfaceId);
   mPointerFocusList.PushBack(surfaceId);
}


// ------------------------------------------------------------------------
bool FocusHandler::UpdateKeyboardFocus(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("Update keyboard focus with surfaceId %u", surfaceId));

   ilmErrorTypes lIlmError = ILM_SUCCESS;

   // First check if surface is configured to receive keyboard input
   bool lRc = (0 != (IlmAccessor::GetInputAcceptance(surfaceId) & ILM_INPUT_DEVICE_KEYBOARD));
   if (lRc)
   {
      if (!IlmAccessor::HasKeyboardFocus(surfaceId))
      {
         // Set keyboard focus to given surface and update focus list of active surfaces
         lRc = IlmAccessor::SetKeyboardFocus(surfaceId, false, lIlmError);
         if (lRc)
         {
            ETG_TRACE_USR4(("Keyboard focus set to surface %u.",
                            surfaceId));
         }
         else
         {
            ETG_TRACE_ERR(("Keyboard focus set to surface %u failed!",
                           surfaceId));
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("Keyboard focus transfer currently not applicable for surface %u! Focus transfer request ignored...",
                      surfaceId));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool FocusHandler::UpdatePointerFocus(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("Update pointer focus with surfaceId %u", surfaceId));

   ilmErrorTypes lIlmError = ILM_SUCCESS;

   // First check if surface is configured to receive pointer input
   bool lRc = (0 != (IlmAccessor::GetInputAcceptance(surfaceId) & ILM_INPUT_DEVICE_POINTER));
   if (lRc)
   {
      if (!IlmAccessor::HasPointerFocus(surfaceId))
      {
         // Set pointer focus to given surface and update focus list of active surfaces
         lRc = IlmAccessor::SetPointerFocus(surfaceId, false, lIlmError);
         if (lRc)
         {
            ETG_TRACE_USR4(("Pointer focus set to surface %u.",
                            surfaceId));
         }
         else
         {
            ETG_TRACE_ERR(("Pointer focus set to surface %u failed!",
                           surfaceId));
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("Pointer focus transfer currently not applicable for surface %u! Focus transfer request ignored...",
                      surfaceId));
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool FocusHandler::EnableInputEvents(UInt32 surfaceId, bool enable)
{
   ETG_TRACE_USR1(("%40s input events of surface %u",
                   enable ? "Enable" : "Disable",
                   surfaceId));

   bool lRc = (0 != surfaceId);
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   if (lRc)
   {
      // Disable first all input events
      lRc = IlmAccessor::AcceptInputEvents(surfaceId,
                                           ILM_INPUT_DEVICE_ALL,
                                           ILM_FALSE,
                                           false,
                                           lIlmError);

      // Enable all input events defined by the project settings
      if (enable)
      {
         ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
         if (0 != lScreenLayouter)
         {
            lRc = lRc && IlmAccessor::AcceptInputEvents(surfaceId,
                  lScreenLayouter->GetInputSources(surfaceId),
                  ILM_TRUE,
                  false,
                  lIlmError);
         }
         else
         {
            lRc = false;
            ETG_TRACE_SYS(("%40sabling input event acceptance failed for surface %u, ScreenLayouter instance not accessible!",
                           enable ? "En" : "Dis",
                           surfaceId));
         }
      }

      if (!lRc)
      {
         ETG_TRACE_SYS(("%40sabling input event acceptance failed for surface %u!",
                        enable ? "En" : "Dis",
                        surfaceId));
      }
   }
   else
   {
      ETG_TRACE_USR4(("%40s: Surface ID %d is not applicable! Ignoring request...",
                      __FUNCTION__,
                      surfaceId));
   }
   return lRc;
}


void FocusHandler::FilterKeyboardFocus(UInt32 displayId)
{
   UInt32 lDisplayId = 0;
   UInt32 lSurfaceId = mKeyboardFocusList.GetFirst();
   mFilteredKeyboardFocus.clear();

   while (0 != lSurfaceId)
   {
      lDisplayId = PluginActions::GetDisplayIdOfSurface(lSurfaceId);
      if (lDisplayId == displayId)
      {
         mFilteredKeyboardFocus.push_back(lSurfaceId);
      }
      lSurfaceId = mKeyboardFocusList.GetNext();
   }
}


void FocusHandler::FilterPointerFocus(UInt32 displayId)
{
   UInt32 lSurfaceId = mPointerFocusList.GetFirst();
   mFilteredPointerFocus.clear();
   while (0 != lSurfaceId)
   {
      UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(lSurfaceId);
      if (lDisplayId == displayId)
      {
         mFilteredPointerFocus.push_back(lSurfaceId);
      }
      lSurfaceId = mPointerFocusList.GetNext();
   }
}


// ------------------------------------------------------------------------
bool FocusHandler::TransferInputFocus()
{
   ETG_TRACE_USR1(("Transfer input focus"));

   ilmErrorTypes lIlmError = ILM_SUCCESS;
   bool lFocusChanged = false;

   //Extract display id list from ScreenLayouter
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (lScreenLayouter != NULL)
   {
      ScreenLayouter::DisplayIdList lDisplayIdList = lScreenLayouter->GetDisplayIdList();

      //Itrate through multiple diplsplay id's list
      for (ScreenLayouter::DisplayIdList::iterator lIt = lDisplayIdList.begin(); lIt != lDisplayIdList.end(); ++lIt)
      {
         // Transfer keyboard focus
         UInt32 lKeyboardFocusedSurfaceId = TransferInputKeyboardFocus(*lIt);

         if ((0 != lKeyboardFocusedSurfaceId) && (!IlmAccessor::HasKeyboardFocus(lKeyboardFocusedSurfaceId)))
         {
            lFocusChanged = IlmAccessor::SetKeyboardFocus(lKeyboardFocusedSurfaceId, false, lIlmError);
         }

         // Transfer pointer focus
         UInt32 lPointerFocusedSurfaceId = TransferInputPointerFocus(*lIt);

         if ((0 != lPointerFocusedSurfaceId) && (!IlmAccessor::HasPointerFocus(lPointerFocusedSurfaceId)))
         {
            lFocusChanged =  IlmAccessor::SetPointerFocus(lPointerFocusedSurfaceId, false, lIlmError);
         }
      }
   }
   return lFocusChanged;
}


// ------------------------------------------------------------------------
void FocusHandler::UpdateInputAcceptance(UInt32 surfaceId, bool enable)
{
   if (enable)
   {
      (void)EnableInputEvents(surfaceId, enable);
   }

   // Retrieve instances of status container
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
   ActivePopups& lActivePopups = ActivePopups::GetInstance();

   UInt32 lTopSurfaceId = lActiveSurfaces.GetTop(PluginActions::GetDisplayIdOfSurface(surfaceId));
   PopupState* lTopVisiblePopupState = PluginActions::GetTopVisiblePopup();

   ETG_TRACE_USR1(("%40s: Active applications top surface: %u)",
                   __FUNCTION__,
                   lTopSurfaceId));

   // Enable exclusive input acceptance for top popup
   if (0 != lTopVisiblePopupState)
   {
      // First re-activate input event reception of top popup
      if (surfaceId != lTopVisiblePopupState->GetSurfaceId())
      {
         if (FocusHandler::EnableInputEvents(lTopVisiblePopupState->GetSurfaceId(), true))
         {
            ETG_TRACE_USR1(("Enable input event reception for top popup surface %u",
                            lTopVisiblePopupState->GetSurfaceId()));
         }
      }

      if (ProjectSettings::DisableInputReceptionOfUnderlyingPopup())
      {
         // Then traverse through active popups list for disabling input acceptance
         PopupState* lPopupState = lActivePopups.GetOverallFirst();
         while (0 != lPopupState)
         {
            if (lPopupState->GetSurfaceId() != lTopVisiblePopupState->GetSurfaceId())
            {
               if (FocusHandler::EnableInputEvents(lPopupState->GetSurfaceId(), false))
               {
                  ETG_TRACE_USR1(("Disable input event reception for popup surface %u",
                                  lPopupState->GetSurfaceId()));
               }
            }
            lPopupState = lActivePopups.GetOverallNext();
         }
      }
      if (ProjectSettings::DisableInputReceptionOfUnderlyingSurface())
      {
         // Then traverse through active surfaces list for disabling input acceptance
         UInt32 displayId = PluginActions::GetDisplayIdOfSurface(lTopVisiblePopupState->GetSurfaceId());
         UInt32 lSurfaceId = lActiveSurfaces.GetFirst(displayId, false);
         while (0 != lSurfaceId)
         {
            if (FocusHandler::EnableInputEvents(lSurfaceId, false))
            {
               ETG_TRACE_USR1(("Disable input event reception for surface %u",
                               lSurfaceId));
            }
            lSurfaceId = lActiveSurfaces.GetNext(displayId, false);
         }
      }
   }
   // Enable exclusive input acceptance for top surface
   else if (0 != lTopSurfaceId)
   {
      // First re-activate input event reception of top surface
      if (surfaceId != lTopSurfaceId)
      {
         if (FocusHandler::EnableInputEvents(lTopSurfaceId, true))
         {
            ETG_TRACE_USR1(("Enable input event reception for top surface %u",
                            lTopSurfaceId));
         }
      }

      if (ProjectSettings::DisableInputReceptionOfUnderlyingSurface())
      {
         // Then traverse through active surfaces list for disabling input acceptance
         UInt32 displayId = PluginActions::GetDisplayIdOfSurface(lTopVisiblePopupState->GetSurfaceId());
         UInt32 lSurfaceId = lActiveSurfaces.GetFirst(displayId , false);
         while (0 != lSurfaceId)
         {
            if (lSurfaceId != lTopSurfaceId)
            {
               if (FocusHandler::EnableInputEvents(lSurfaceId, false))
               {
                  ETG_TRACE_USR1(("Disable input event reception for surface %u",
                                  lSurfaceId));
               }
            }
            lSurfaceId = lActiveSurfaces.GetNext(false);
         }
      }
   }
}


// ------------------------------------------------------------------------
bool FocusHandler::UpdateInputFocus(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("Update input focus of surface %u", surfaceId));

   UpdateSeatAcceptance(surfaceId);
   UpdateInputAcceptance(surfaceId, true);
   return TransferInputFocus();
}


// ------------------------------------------------------------------------
void FocusHandler::RemoveSurfaces(bool popup)
{
   UInt32 lKeyboardFocus = mKeyboardFocusList.GetFirst();
   while (0 != lKeyboardFocus)
   {
      bool lIsPopup = ProjectSettings::IsPopup(lKeyboardFocus);

      if ((popup && lIsPopup) || (!popup && !lIsPopup))
      {
         lKeyboardFocus = mKeyboardFocusList.Remove(lKeyboardFocus);
      }
      else
      {
         lKeyboardFocus = mKeyboardFocusList.GetNext();
      }
   }

   UInt32 lPointerFocus = mPointerFocusList.GetFirst();
   while (0 != lPointerFocus)
   {
      bool lIsPopup = ProjectSettings::IsPopup(lPointerFocus);

      if ((popup && lIsPopup) || (!popup && !lIsPopup))
      {
         lPointerFocus = mPointerFocusList.Remove(lPointerFocus);
      }
      else
      {
         lPointerFocus = mPointerFocusList.GetNext();
      }
   }
}


// ------------------------------------------------------------------------
void FocusHandler::Dump() const
{
   ETG_TRACE_USR1(("FocusHandler: Dump"));

   ETG_TRACE_USR1(("  Keyboard Focus List:"));
   mKeyboardFocusList.Dump();

   ETG_TRACE_USR1(("  Pointer Focus List:"));
   mPointerFocusList.Dump();
}


// ------------------------------------------------------------------------
UInt32 FocusHandler::TransferInputKeyboardFocus(UInt32 displayId)
{
   ETG_TRACE_USR1(("Transfer input keyboard focus"));

   UInt32 lKeyboardFocusedItem = 0;
   UInt32 lFocusListItem = 0;

   FilterKeyboardFocus(displayId);

   //
   for (UInt16 index = 0; index < mFilteredKeyboardFocus.size(); index++)
   {
      lFocusListItem = mFilteredKeyboardFocus[index];
      if ((0 != (IlmAccessor::GetInputAcceptance(lFocusListItem) & ILM_INPUT_DEVICE_KEYBOARD)) &&
            ((!ProjectSettings::IsPopup(lKeyboardFocusedItem)) || (ProjectSettings::IsPopup(lFocusListItem))))
      {
         Int32 lKeyboardFocusedItemPrio = mForcedKeyboardFocusPriority.mSurfaceId == lKeyboardFocusedItem ?
                                          mForcedKeyboardFocusPriority.mPriority :
                                          mKeyboardFocusPriority[lKeyboardFocusedItem];

         Int32 lFocusListItemPrio = mForcedKeyboardFocusPriority.mSurfaceId == lFocusListItem ?
                                    mForcedKeyboardFocusPriority.mPriority :
                                    mKeyboardFocusPriority[lFocusListItem];

         if ((0 == lKeyboardFocusedItem) || (lFocusListItemPrio <= lKeyboardFocusedItemPrio))
         {
            lKeyboardFocusedItem = lFocusListItem;
         }
      }
   }
   return lKeyboardFocusedItem;
}


// ------------------------------------------------------------------------
UInt32 FocusHandler::TransferInputPointerFocus(UInt32 displayId)
{
   ETG_TRACE_USR1(("Transfer input pointer focus"));

   UInt32 lPointerFocusedItem = 0;
   UInt32 lFocusListItem = 0;

   FilterPointerFocus(displayId);

   //
   for (UInt16 index = 0; index < mFilteredPointerFocus.size(); index++)
   {
      lFocusListItem = mFilteredPointerFocus[index];
      if ((0 != (IlmAccessor::GetInputAcceptance(lFocusListItem) & ILM_INPUT_DEVICE_POINTER)) &&
            ((!ProjectSettings::IsPopup(lPointerFocusedItem)) || (ProjectSettings::IsPopup(lFocusListItem))))
      {
         Int32 lPointerFocusedItemPrio = mForcedPointerFocusPriority.mSurfaceId == lPointerFocusedItem ?
                                         mForcedPointerFocusPriority.mPriority :
                                         mPointerFocusPriority[lPointerFocusedItem];

         Int32 lFocusListItemPrio = mForcedPointerFocusPriority.mSurfaceId == lFocusListItem ?
                                    mForcedPointerFocusPriority.mPriority :
                                    mPointerFocusPriority[lFocusListItem];

         if ((0 == lPointerFocusedItem) || (lFocusListItemPrio <= lPointerFocusedItemPrio))
         {
            lPointerFocusedItem = lFocusListItem;
         }
      }
   }
   return lPointerFocusedItem;
}


// ------------------------------------------------------------------------
void FocusHandler::UpdateSeatAcceptance(UInt32 surfaceId) const
{
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      ScreenLayouter::SeatList* lSeatList = lScreenLayouter->GetSeatList(surfaceId);
      if ((0 != lSeatList) && (0 != lSeatList->size()))
      {
         std::vector<char*> lSeats(lSeatList->size());
         ScreenLayouter::SeatList::iterator it;
         UInt32 i = 0;
         for (it = lSeatList->begin(); lSeatList->end() != it; ++it)
         {
            lSeats[i] = const_cast<char*>(lSeatList->at(i).c_str());
            ++i;
         }
         IlmAccessor::SetIntputAcceptanceOn(surfaceId, (t_ilm_uint)lSeatList->size(), &lSeats[0]);
      }
   }
}


}
