/* ***************************************************************************************
* FILE:          AnimationHandler.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationHandler.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include <Base.h>
#include "AnimationHandler.h"

#include <ScreenBroker/Service/ServiceApi.h>
#include <Shared/PluginActions.h>
#include <Shared/Animation/Animation.h>
#include <Shared/Animation/AnimationFactory.h>
#include <Shared/Animation/AnimationUserData.h>
#include <ScreenLayouter/ScreenLayouter.h>
#include <Shared/IlmAccessor.h>
#include <Shared/Animation/Animator.h>
#include <Shared/Animation/SimpleAnimation.h>

#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/AnimationHandler.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);

// ------------------------------------------------------------------------
AnimationHandler::AnimatorList AnimationHandler::mAnimatorList;
AnimationHandler::AnimatorList AnimationHandler::mAnimationQueue;
AnimationBase* AnimationHandler::mCurrentAnimation = 0;
bool AnimationHandler::mCancel = false;
AnimationMethod::Enum AnimationHandler::_CurrentAnimationMethod = AnimationMethod::Parallel;
bool AnimationHandler::_isAnimationMethodRead = false;


// ------------------------------------------------------------------------

bool AnimationHandler::isInAnimatorList(UInt32 surfaceId)
{
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   bool ret = false;
   for (AnimatorList::iterator it = mAnimatorList.begin(); it != mAnimatorList.end(); ++it)
   {
      if (surfaceId == (*it)->GetSurfaceId())
      {
         ret = true;
         break;
      }
   }
   ETG_TRACE_USR1(("isInAnimatorList: surfaceId [%u], ret [%u]", surfaceId, ret));
   return ret;
}


void AnimationHandler::cancelAnimation(UInt32 surfaceId)
{
   //If Surface id is 0 cancel All Animations
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   AnimatorList lCalcelANimationList;
   for (AnimatorList::iterator it = mAnimatorList.begin(); it != mAnimatorList.end(); ++it)
   {
      if (surfaceId == (*it)->GetSurfaceId())
      {
         lCalcelANimationList.push_back(*it);
         break;
      }
      else if (0 == surfaceId)
      {
         //Cancel All animations on Abort All Animations
         lCalcelANimationList.push_back(*it);
      }
   }
   while (!lCalcelANimationList.empty())
   {
      AnimationBase* lAnimation = lCalcelANimationList.front();
      lCalcelANimationList.pop_front();
      ETG_TRACE_USR1(("Canceling Animation for SurfaceId [%u]", lAnimation->GetSurfaceId()));
      lAnimation->Cancel();
   }
}


void AnimationHandler::removeAnimationData(UInt32 surfaceId)
{
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   AnimatorList::iterator iter = mAnimatorList.begin();
   while (iter != mAnimatorList.end())
   {
      if (surfaceId == (*iter)->GetSurfaceId())
      {
         if ((*iter)->GetAnimationStatus())
         {
            ETG_TRACE_USR1(("removeAnimationData: SurfaceId [%u] is Removed", (*iter)->GetSurfaceId()));
            iter = mAnimatorList.erase(iter);
         }
         break;
      }
      else
      {
         ++iter;
      }
   }
}


void AnimationHandler::popOutAnimationQueue(UInt32 surfaceId)
{
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   AnimatorList::iterator iter = mAnimationQueue.begin();
   while (iter != mAnimationQueue.end())
   {
      if (surfaceId == 0)
      {
         ETG_TRACE_USR1(("popAnimationQueue: SurfaceId [%u] is Removed", (*iter)->GetSurfaceId()));
         iter = mAnimationQueue.erase(iter);
      }
      else if (surfaceId == (*iter)->GetSurfaceId())
      {
         ETG_TRACE_USR1(("popAnimationQueue: SurfaceId [%u] is Removed", (*iter)->GetSurfaceId()));
         iter = mAnimationQueue.erase(iter);
         break;
      }
      else
      {
         ++iter;
      }
   }
}


void AnimationHandler::OnAnimationFinished(const AnimationSurfacePropertiesBase& surfaceProperties,
      bool triggerNext)
{
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   // Send AnimationStopped only for current animations
   ETG_TRACE_USR1(("AnimationFinished requested for surface[%u]", surfaceProperties.surfaceId));

   if (isInAnimatorList(surfaceProperties.surfaceId))
   {
      PluginActions::SendAnimationState(surfaceProperties.animationUserData.userData,
                                        surfaceProperties.surfaceId,
                                        ScreenBroker::SurfaceState::AnimationStopped);
      removeAnimationData(surfaceProperties.surfaceId);
   }
   else
   {
      // Set surface visible and notify application about surface state change in case of
      // canceled show animation without animation start
      if (PluginActions::CalculateVisibility(surfaceProperties.animationUserData.surfaceState))
      {
         ilmErrorTypes lIlmError = ILM_SUCCESS;
         (void) IlmAccessor::SetSurfaceVisibility(surfaceProperties.surfaceId, ILM_TRUE, true, lIlmError);

         PluginActions::SendNotification(surfaceProperties.surfaceId,
                                         surfaceProperties.animationUserData);
      }
   }

   ETG_TRACE_USR1(("Animation of surface %u finished", surfaceProperties.surfaceId));

   if (AnimationDirection::Hide == surfaceProperties.animationDirection)
   {
      // Make sure the surface is really invisible after a hide animation
      ETG_TRACE_USR4(("Make sure the Surface is really invisible"));
      ilmErrorTypes lIlmError = ILM_SUCCESS;
      (void) IlmAccessor::SetSurfaceVisibility(surfaceProperties.surfaceId, ILM_FALSE, true, lIlmError);
   }

   // Notify application about successful change of surface state in case surface gets invisible after animation
   if (!PluginActions::CalculateVisibility(surfaceProperties.animationUserData.surfaceState))
   {
      ETG_TRACE_USR4(("Notify invisible after animation"));
      PluginActions::SendNotification(surfaceProperties.surfaceId,
                                      surfaceProperties.animationUserData);
   }
   if (triggerNext && !mCancel)
   {
      // Start next queued animation, if available
      TriggerAnimations();
   }
}


void AnimationHandler::OnLayerAnimationFinished(const AnimationSurfacePropertiesBase& surfaceProperties,
      bool triggerNext)
{
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   // Send AnimationStopped only for current animations
   ETG_TRACE_USR1(("AnimationFinished requested for Layer[%u]", surfaceProperties.layerId));

   if (isInAnimatorList(surfaceProperties.layerId))
   {
      PluginActions::SendLayerAnimationState(surfaceProperties.animationUserData.userData,
                                             surfaceProperties.layerId,
                                             ScreenBroker::LayerState::AnimationStopped);
      removeAnimationData(surfaceProperties.layerId);
   }

   if (AnimationDirection::Hide == surfaceProperties.animationDirection)
   {
      // Make sure the surface is really invisible after a hide animation
      ETG_TRACE_USR4(("Make sure the Layer is really invisible"));
      ilmErrorTypes lIlmError = ILM_SUCCESS;
      (void) IlmAccessor::SetLayerVisibility(surfaceProperties.layerId, ILM_FALSE, true, lIlmError);
   }
   if (triggerNext && !mCancel)
   {
      // Start next queued animation, if available
      TriggerAnimations();
   }
}


// ------------------------------------------------------------------------
bool AnimationHandler::EstablishAnimation(AnimationDirection::Enum animationDirection,
      UInt32 surfaceId,
      const AnimationUserData& animationUserData,
      AnimationType::Enum animationType)
{
   AnimationBase* lAnimation = 0;
   // Retrieve instance of screen layouter
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);

   if (0 != lScreenLayouter)
   {
      const AnimationProperties* lAnimationProperties = lScreenLayouter->GetAnimationProperties(surfaceId,
            animationDirection, animationType);
      if ((0 != lAnimationProperties) && (GetAnimationMethod() != AnimationMethod::Disabled))
      {
         /* animationDirection - It will provides either Show/Hide/Entry/Exit animation details */
         /* This parameter will be used to set FadeIn/FadeOut */
         lAnimation = AnimationFactory::CreateAnimation((AnimationType::Enum)lAnimationProperties->type,
                      surfaceId,
                      animationUserData,
                      animationDirection);

         // Update additional animation parameter if animation was found
         if (0 != lAnimation)
         {
            lAnimation->SetAnimationStatus(false);
            ComplexAnimation* lComplexAnimation = dynamic_cast<ComplexAnimation*>(lAnimation);

            if (0 != lComplexAnimation)
            {
               lComplexAnimation->SetDuration(lAnimationProperties->duration);
               lComplexAnimation->SetUpdateCount(lAnimationProperties->steps);
               lComplexAnimation->SetAfterAnimationCallback(OnAnimationFinished);
               lComplexAnimation->SetAcceleration(lAnimationProperties->acceleration);
               lComplexAnimation->SetAmplitude(lAnimationProperties->amplitude);
               lComplexAnimation->SetDelayDuration(lAnimationProperties->delayduration);
            }
            lAnimation->UpdateAnimationCategory(((GetAnimationMethod() == AnimationMethod::Sequential) ? true : false));
            CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
            popOutAnimationQueue(surfaceId);
            mAnimationQueue.push_back(lAnimation);
         }
         else
         {
            ETG_TRACE_ERR(("Animation creation for surface %u failed", surfaceId));
         }
      }
      else
      {
         if (AnimationDirection::Show == animationDirection)
         {
            lAnimation = AnimationFactory::CreateAnimation(AnimationType::ShowImmediately,
                         surfaceId,
                         animationUserData,
                         animationDirection);
         }
         //else
         else if (AnimationDirection::Hide == animationDirection)
         {
            lAnimation = AnimationFactory::CreateAnimation(AnimationType::HideImmediately,
                         surfaceId,
                         animationUserData,
                         animationDirection);
         }
         else
         {
            ETG_TRACE_ERR(("Relavent Animation creation is not found for surface [%u]!!!", surfaceId));
         }
         if (0 != lAnimation)
         {
            lAnimation->SetAnimationStatus(false);
            lAnimation->UpdateAnimationCategory(((GetAnimationMethod() == AnimationMethod::Sequential) ? true : false));
            CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
            popOutAnimationQueue(surfaceId);
            mAnimationQueue.push_back(lAnimation);
         }
      }
   }
   //To Stop On going animation of the surface if new animation is established
   if (0 != lAnimation)
   {
      AbortAnimations(surfaceId);
      return true;
   }
   return false;
}


// ------------------------------------------------------------------------
bool AnimationHandler::EstablishLayerAnimation(AnimationDirection::Enum animationDirection,
      UInt32 layerId,
      const AnimationUserData& animationUserData,
      AnimationType::Enum animationType)
{
   AnimationBase* lAnimation = 0;
   // Retrieve instance of screen layouter
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);

   if (0 != lScreenLayouter)
   {
      const AnimationProperties* lAnimationProperties = lScreenLayouter->GetLayerAnimationProperties(layerId,
            animationDirection, animationType);
      if ((0 != lAnimationProperties) && (GetAnimationMethod() != AnimationMethod::Disabled))
      {
         /* animationDirection - It will provides either Show/Hide/Entry/Exit animation details */
         /* This parameter will be used to set FadeIn/FadeOut */
         lAnimation = AnimationFactory::CreateAnimation((AnimationType::Enum)lAnimationProperties->type,
                      layerId,
                      animationUserData,
                      animationDirection,
                      true);

         // Update additional animation parameter if animation was found
         if (0 != lAnimation)
         {
            lAnimation->SetAnimationStatus(false);
            ComplexAnimation* lComplexAnimation = dynamic_cast<ComplexAnimation*>(lAnimation);

            if (0 != lComplexAnimation)
            {
               lComplexAnimation->SetDuration(lAnimationProperties->duration);
               lComplexAnimation->SetUpdateCount(lAnimationProperties->steps);
               lComplexAnimation->SetAfterAnimationCallback(OnLayerAnimationFinished);
               lComplexAnimation->SetAcceleration(lAnimationProperties->acceleration);
               lComplexAnimation->SetAmplitude(lAnimationProperties->amplitude);
               lComplexAnimation->SetDelayDuration(lAnimationProperties->delayduration);
            }
            lAnimation->UpdateAnimationCategory(((GetAnimationMethod() == AnimationMethod::Sequential) ? true : false));
            CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
            popOutAnimationQueue(layerId);
            mAnimationQueue.push_back(lAnimation);
         }
         else
         {
            ETG_TRACE_ERR(("Animation creation for surface %u failed", layerId));
         }
      }
      else
      {
         if (AnimationDirection::Show == animationDirection)
         {
            lAnimation = AnimationFactory::CreateAnimation(AnimationType::ShowImmediately,
                         layerId,
                         animationUserData,
                         animationDirection, true);
         }
         //else
         else if (AnimationDirection::Hide == animationDirection)
         {
            lAnimation = AnimationFactory::CreateAnimation(AnimationType::HideImmediately,
                         layerId,
                         animationUserData,
                         animationDirection, true);
         }
         else
         {
            ETG_TRACE_ERR(("Relavent Animation creation is not found for layer [%u]!!!", layerId));
         }
         if (0 != lAnimation)
         {
            lAnimation->SetAnimationStatus(false);
            lAnimation->UpdateAnimationCategory(((GetAnimationMethod() == AnimationMethod::Sequential) ? true : false));
            CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
            popOutAnimationQueue(layerId);
            mAnimationQueue.push_back(lAnimation);
         }
      }
   }
   return (0 != lAnimation);
}


// ------------------------------------------------------------------------
void AnimationHandler::TriggerAnimations(bool commit)
{
   static bool triggerSynchronousAnimation = false;
   // Implement sequential processing of queued animations
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   while (!mAnimationQueue.empty())
   {
      ETG_TRACE_USR1(("Size- %d", mAnimatorList.size()));
      mAnimatorList.push_back(mAnimationQueue.front());
      mAnimationQueue.pop_front();
      triggerSynchronousAnimation = true;
      if (GetAnimationMethod() == AnimationMethod::Sequential)
      {
         ETG_TRACE_USR1(("Sequential animation configured"));
         break;
      }
   }

   if (triggerSynchronousAnimation)
   {
      std::list<ComplexAnimation*> lComplexAnimationBatch;
      lComplexAnimationBatch.clear();
      std::list<SimpleAnimation*> lSimpleAnimationQueue;
      lSimpleAnimationQueue.clear();
      for (AnimatorList::iterator it = mAnimatorList.begin(); it != mAnimatorList.end();)
      {
         if (false == (*it)->GetAnimationStatus())
         {
            (*it)->SetAnimationStatus(true);

            //For simple animation
            ComplexAnimation* lComplexAnimation = dynamic_cast<ComplexAnimation*>((*it));
            if (lComplexAnimation == 0)
            {
               ETG_TRACE_USR1(("Adding Simple Animation to queue: SurfaceId[%u]", (*it)->GetSurfaceId()));
               SimpleAnimation* lSimpleAnimation = dynamic_cast<SimpleAnimation*>((*it));
               if (0 != lSimpleAnimation)
               {
                  lSimpleAnimationQueue.push_back(lSimpleAnimation);
               }
               else
               {
                  ETG_TRACE_USR1(("Invalid Animation Entry Removing from the queue"));
               }
               (*it) = 0;
               it = mAnimatorList.erase(it);
            }
            else
            {
               ETG_TRACE_USR1(("Adding Animation to queue:  %40s ID [%u]",
                               ((*it)->GetContentType() ? "Layer" : "Surface"),
                               (*it)->GetSurfaceId()));
               lComplexAnimationBatch.push_back(lComplexAnimation);
               ++it;
            }
         }
         else
         {
            //If in between some new requests came and for the current one Animation is not yet finished
            ++it;
         }
      }
      if (!lComplexAnimationBatch.empty())
      {
         ETG_TRACE_USR1(("Starting Complex Animation Batch"));
         Animator::getInstance()->InitiateGroupAnimations(lComplexAnimationBatch);
      }
      while (!lSimpleAnimationQueue.empty())
      {
         SimpleAnimation* lSimpleAnimation = lSimpleAnimationQueue.front();
         lSimpleAnimationQueue.pop_front();
         lSimpleAnimation->StartSimpleAnimation(commit);
      }
   }
}


// ------------------------------------------------------------------------
void AnimationHandler::AbortAnimations(UInt32 surfaceId)
{
   // If surfaceId is 0 (default value) abort all animations, otherwise only animations of that surface
   if (0 == surfaceId)
   {
      ETG_TRACE_SYS(("Cancel currently running and queued animations.Queued:[%d] Running:[%d]", mAnimationQueue.size(), mAnimatorList.size()));
      popOutAnimationQueue(surfaceId);
   }
   else
   {
      ETG_TRACE_SYS(("Cancel currently running animation of surface %d", surfaceId));
   }

   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   mCancel = true;
   ETG_TRACE_USR1(("Running Animation cancellation is triggered = %d", mCancel));

   cancelAnimation(surfaceId);

   mCancel = false;
   ETG_TRACE_USR1(("Running Animation cancellation is triggered_1 = %d", mCancel));
}


// ------------------------------------------------------------------------
void AnimationHandler::ObtainCriticalSection()
{
   AnimatorTimer::_criticalSection.Obtain();
}


// ------------------------------------------------------------------------
void AnimationHandler::ReleaseCriticalSection()
{
   AnimatorTimer::_criticalSection.Release();
}


// ------------------------------------------------------------------------
bool AnimationHandler::IsCancelled()
{
   CriticalSectionLocker lLock(&AnimatorTimer::_criticalSection);
   return mCancel;
}


AnimationMethod::Enum AnimationHandler::GetAnimationMethod()
{
   if (false == _isAnimationMethodRead)
   {
      ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
      if (0 != lScreenLayouter)
      {
         UInt32 u32AnimationMethod;
         _isAnimationMethodRead = lScreenLayouter->getAnimationMethod(u32AnimationMethod);
         if (true == _isAnimationMethodRead)
         {
            _CurrentAnimationMethod = (static_cast<AnimationMethod::Enum>(u32AnimationMethod));
         }
      }
   }
   return _CurrentAnimationMethod;
}


}
