/* ***************************************************************************************
* FILE:          SimpleAnimation.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SimpleAnimation.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "SimpleAnimation.h"

#include <Shared/AnimationHandler.h>
#include <Shared/IlmAccessor.h>
#include <Shared/PluginActions.h>

#include <Shared/Animation/AnimationUserData.h>
#include <Shared/Animation/AnimationPoolBase.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/SimpleAnimation.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);

SimpleAnimation::SimpleAnimation() :
   mSurfaceId(0),
   mLayerId(0),
   mAnimationDirection(AnimationDirection::Undefined),
   mAnimationUserData(),
   mInitialSurfaceBounds(),
   mStarted(false)
{
}


SimpleAnimation::~SimpleAnimation()
{
   mPool = 0;
}


// ------------------------------------------------------------------------
void SimpleAnimation::StartSimpleAnimation(bool commit)
{
   if (Start() && commit)
   {
      if (isLayerContent)
      {
         ilmErrorTypes lIlmError = ILM_SUCCESS;
         (void)IlmAccessor::Commit(lIlmError);
      }
      else
      {
         PluginActions::CheckExplicitCommit(mSurfaceId);
      }
   }
   if (GetAnimationCategory())
   {
      AnimationHandler::TriggerAnimations(commit);
   }
}


// ------------------------------------------------------------------------
bool SimpleAnimation::Start()
{
   ETG_TRACE_USR1(("Start %40s SimpleAnimation for %40s  %u",
                   (isLayerContent ? "Layer" : "Surface"),
                   (mAnimationUserData.visibility ? "Show" : "Hide"),
                   (isLayerContent ? mLayerId : mSurfaceId)));
   mStarted = true;
   bool lRc = false;
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   if (isLayerContent)
   {
      lRc = IlmAccessor::SetLayerOpacity(mLayerId, 1.0, false, lIlmError);
      lRc = lRc && IlmAccessor::SetLayerRectangle(mLayerId, mInitialSurfaceBounds.position.x, mInitialSurfaceBounds.position.y,
            mInitialSurfaceBounds.width, mInitialSurfaceBounds.height, false, lIlmError);
      lRc = lRc && IlmAccessor::SetLayerVisibility(mLayerId,
            (mAnimationUserData.visibility ? ILM_TRUE : ILM_FALSE),
            false,
            lIlmError);
   }
   else
   {
      lRc = IlmAccessor::SetSurfaceOpacity(mSurfaceId, 1.0, false, lIlmError);
      lRc = lRc && IlmAccessor::SetSurfaceRectangle(mSurfaceId, mInitialSurfaceBounds.position.x, mInitialSurfaceBounds.position.y,
            mInitialSurfaceBounds.width, mInitialSurfaceBounds.height, false, lIlmError);
      lRc = lRc && IlmAccessor::SetSurfaceVisibility(mSurfaceId,
            (mAnimationUserData.visibility ? ILM_TRUE : ILM_FALSE),
            false,
            lIlmError);

      PluginActions::SendNotification(mSurfaceId, mAnimationUserData);
   }

   if (mPool != 0)
   {
      mPool->ReturnAnimation(this);
   }

   //AnimationHandler::ClearCurrentAnimation(mSurfaceId);
   mStarted = false;
   ETG_TRACE_USR4(("Start(): %40s[%u]",
                   (isLayerContent ? "Layer" : "Surface"),
                   mSurfaceId));
   return lRc;
}


// ------------------------------------------------------------------------
void SimpleAnimation::Cancel()
{
   if (!mStarted)
   {
      ETG_TRACE_USR1(("Cancel %40s SimpleAnimation for %40s %u",
                      (isLayerContent ? "Layer" : "Surface"),
                      (mAnimationUserData.visibility ? "Show" : "Hide"),
                      mSurfaceId));

      ilmErrorTypes lIlmError = ILM_SUCCESS;
      if (isLayerContent)
      {
         IlmAccessor::SetLayerOpacity(mLayerId, 1.0, false, lIlmError);
         IlmAccessor::SetLayerRectangle(mLayerId, mInitialSurfaceBounds.position.x,
                                        mInitialSurfaceBounds.position.y, mInitialSurfaceBounds.width,
                                        mInitialSurfaceBounds.height, false, lIlmError);
         IlmAccessor::SetLayerVisibility(mLayerId,
                                         (mAnimationUserData.visibility ? ILM_TRUE : ILM_FALSE),
                                         true,
                                         lIlmError);
      }
      else
      {
         IlmAccessor::SetSurfaceOpacity(mSurfaceId, 1.0, false, lIlmError);
         IlmAccessor::SetSurfaceRectangle(mSurfaceId, mInitialSurfaceBounds.position.x,
                                          mInitialSurfaceBounds.position.y, mInitialSurfaceBounds.width,
                                          mInitialSurfaceBounds.height, false, lIlmError);
         IlmAccessor::SetSurfaceVisibility(mSurfaceId,
                                           (mAnimationUserData.visibility ? ILM_TRUE : ILM_FALSE),
                                           true,
                                           lIlmError);

         PluginActions::SendNotification(mSurfaceId, mAnimationUserData);
      }
      if (mPool != 0)
      {
         mPool->ReturnAnimation(this);
      }
   }
}


// ------------------------------------------------------------------------
void SimpleAnimation::SetSurfaceId(UInt32 surfaceId)
{
   mSurfaceId = surfaceId;
}


// ------------------------------------------------------------------------
UInt32 SimpleAnimation::GetSurfaceId()
{
   return mSurfaceId;
}


// ------------------------------------------------------------------------
void SimpleAnimation::SetLayerId(UInt32 layerId)
{
   mLayerId = layerId;
}


// ------------------------------------------------------------------------
UInt32 SimpleAnimation::GetLayerId()
{
   return mLayerId;
}


// ------------------------------------------------------------------------
void SimpleAnimation::SetAnimationDirection(AnimationDirection::Enum animationDirection)
{
   mAnimationDirection = animationDirection;
}


// ------------------------------------------------------------------------
void SimpleAnimation::SetAnimationUserData(const AnimationUserData& animationUserData)
{
   mAnimationUserData = animationUserData;
}


// ------------------------------------------------------------------------
void SimpleAnimation::SetInitialSurfaceBounds(const Rectangle& initialSurfaceBounds)
{
   mInitialSurfaceBounds = initialSurfaceBounds;
}


// ------------------------------------------------------------------------
void SimpleAnimation::Reset()
{
   mSurfaceId = 0;
   mAnimationDirection = AnimationDirection::Undefined;
}


}
