/* ***************************************************************************************
* FILE:          ScaleAndFadeAnimation.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScaleAndFadeAnimation.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "ScaleAndFadeAnimation.h"
#include "FadingAnimation.h"

#include <Shared/IlmAccessor.h>
#include <Shared/Animation/InterpolationFactory.h>
#include <stdio.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/ScaleAndFadeAnimation.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);
//float initialOpatcity = 0.0f;
//float finalOpacity = 1.0f;
bool bModifySource = false;
// ------------------------------------------------------------------------
bool ScaleAndFadeAnimation::InitializeSurfaceState(ScaleAndFadeAnimationSurfaceProperties& surfaceProperties, bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   //float opacity = 0.0f;

   AccelerationType::Enum lAcceleration = GetAcceleration();
   bool lRc = false;
   InterpolationStrategy* lInterpolation = InterpolationFactory::Create(lAcceleration);
   if (0 != lInterpolation)
   {
      Float lAmplitude = GetAmplitude();
      lInterpolation->SetAmplitude(lAmplitude);
      SetInterpolation(lInterpolation);

      Rectangle& lInitialBounds = surfaceProperties.originalPropertyValue;
      Rectangle& lFinalBounds = surfaceProperties.finalPropertyValue;

      {
         char str[230];
         sprintf(str, "Scale/Translate: Established for %40s ID:%u from (%d,%d)/%dx%d to (%d,%d)/%dx%d in %ums with %u updates",
                 (isLayerContent ? "Layer" : "Surface"),
                 (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                 lInitialBounds.position.x, lInitialBounds.position.y,
                 lInitialBounds.width, lInitialBounds.height,
                 lFinalBounds.position.x, lFinalBounds.position.y,
                 lFinalBounds.width, lFinalBounds.height,
                 GetDuration(),
                 GetUpdateCount());
         ETG_TRACE_USR4(("%230s", str));
      }

      ETG_TRACE_USR1(("Scale/Translate: Initialize %40s ID:%u to x=%d y=%d width=%d height=%d",
                      (isLayerContent ? "Layer" : "Surface"),
                      (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                      lInitialBounds.position.x, lInitialBounds.position.y,
                      lInitialBounds.width, lInitialBounds.height));
      if (lInitialBounds.position.x < 0 || lInitialBounds.position.y < 0)
      {
         bModifySource = true;
         lInitialBounds.position.x *= -1;
         lInitialBounds.position.y *= -1;
         //opacity = finalOpacity;
      }
      else if (lFinalBounds.position.x < 0 || lFinalBounds.position.y < 0)
      {
         bModifySource = true;
         lFinalBounds.position.x *= -1;
         lFinalBounds.position.y *= -1;
         //opacity = finalOpacity;
      }
      else
      {
         bModifySource  = false;
         //opacity = initialOpatcity;
      }
      float startOpacity, endOpacity;
      GetFadeAnimationDetails(startOpacity, endOpacity);
      //Fade implementation reference
      //For Fade
      if (isLayerContent)
      {
         IlmAccessor::SetLayerOpacity(surfaceProperties.layerId,
                                      static_cast<t_ilm_float>(startOpacity),
                                      false,
                                      lIlmError);
      }
      else
      {
         IlmAccessor::SetSurfaceOpacity(surfaceProperties.surfaceId,
                                        static_cast<t_ilm_float>(startOpacity),
                                        false,
                                        lIlmError);
      }
      ETG_TRACE_USR1(("Scale/Translate: startOpacity: %f, endOpacity: %f", startOpacity, endOpacity));

      if (!bModifySource)
      {
         if (isLayerContent)
         {
            lRc = IlmAccessor::SetLayerDestinationRectangle(
                     surfaceProperties.layerId,
                     lInitialBounds.position.x,
                     lInitialBounds.position.y,
                     lInitialBounds.width,
                     lInitialBounds.height,
                     commit,
                     lIlmError);
         }
         else
         {
            lRc = IlmAccessor::SetSurfaceDestinationRectangle(
                     surfaceProperties.surfaceId,
                     lInitialBounds.position.x,
                     lInitialBounds.position.y,
                     lInitialBounds.width,
                     lInitialBounds.height,
                     commit,
                     lIlmError);
         }
      }
      else
      {
         if (isLayerContent)
         {
            lRc = IlmAccessor::SetLayerSourceRectangle(
                     surfaceProperties.layerId,
                     lInitialBounds.position.x,
                     lInitialBounds.position.y,
                     lInitialBounds.width,
                     lInitialBounds.height,
                     commit,
                     lIlmError);
         }
         else
         {
            lRc = IlmAccessor::SetSurfaceSourceRectangle(
                     surfaceProperties.surfaceId,
                     lInitialBounds.position.x,
                     lInitialBounds.position.y,
                     lInitialBounds.width,
                     lInitialBounds.height,
                     commit,
                     lIlmError);
         }
      }
   }
   return lRc;
}


// ------------------------------------------------------------------------
bool ScaleAndFadeAnimation::UpdateSurfaceState(const ScaleAndFadeAnimationSurfaceProperties& surfaceProperties,
      UInt32 count,
      bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   bool lRc = false;
   //float opacity = 0.0f;

   InterpolationStrategy* lInterpolation = GetInterpolation();
   if (0 != lInterpolation)
   {
      const Rectangle& lOriginalBounds = surfaceProperties.originalPropertyValue;
      const Position& lOriginalPosition = lOriginalBounds.position;
      const Rectangle& lFinalBounds = surfaceProperties.finalPropertyValue;
      const Position& lFinalPosition = lFinalBounds.position;

      t_ilm_int lCurrentPosX = lOriginalPosition.x + static_cast<t_ilm_int>(lInterpolation->Interpolate(GetUpdateCount(),
                               static_cast<Float>(lOriginalPosition.x),
                               static_cast<Float>(lFinalPosition.x),
                               count));
      t_ilm_int lCurrentPosY = lOriginalPosition.y + static_cast<t_ilm_int>(lInterpolation->Interpolate(GetUpdateCount(),
                               static_cast<Float>(lOriginalPosition.y),
                               static_cast<Float>(lFinalPosition.y),
                               count));
      t_ilm_int lCurrentWidth = lOriginalBounds.width + static_cast<t_ilm_int>(lInterpolation->Interpolate(GetUpdateCount(),
                                static_cast<Float>(lOriginalBounds.width),
                                static_cast<Float>(lFinalBounds.width),
                                count));
      t_ilm_int lCurrentHeight = lOriginalBounds.height + static_cast<t_ilm_int>(lInterpolation->Interpolate(GetUpdateCount(),
                                 static_cast<Float>(lOriginalBounds.height),
                                 static_cast<Float>(lFinalBounds.height),
                                 count));
      ETG_TRACE_USR4(("Scale/Translate: Update %40s ID:%u to (%d,%d)/%dx%d (count: %u)",
                      (isLayerContent ? "Layer" : "Surface"),
                      (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                      lCurrentPosX, lCurrentPosY,
                      lCurrentWidth, lCurrentHeight,
                      count));
      if (!bModifySource)
      {
         if (isLayerContent)
         {
            lRc = IlmAccessor::SetLayerDestinationRectangle(surfaceProperties.layerId,
                  lCurrentPosX, lCurrentPosY,
                  lCurrentWidth, lCurrentHeight,
                  false, ///check
                  lIlmError);
         }
         else
         {
            lRc = IlmAccessor::SetSurfaceDestinationRectangle(surfaceProperties.surfaceId,
                  lCurrentPosX, lCurrentPosY,
                  lCurrentWidth, lCurrentHeight,
                  false, ///check
                  lIlmError);
            //opacity = initialOpatcity;
         }
      }
      else
      {
         if (isLayerContent)
         {
            lRc = IlmAccessor::SetLayerSourceRectangle(surfaceProperties.layerId,
                  lCurrentPosX, lCurrentPosY,
                  lCurrentWidth, lCurrentHeight,
                  false, ///check
                  lIlmError);
         }
         else
         {
            lRc = IlmAccessor::SetSurfaceSourceRectangle(surfaceProperties.surfaceId,
                  lCurrentPosX, lCurrentPosY,
                  lCurrentWidth, lCurrentHeight,
                  false, ///check
                  lIlmError);
            //opacity = finalOpacity;
         }
      }
      float startOpacity, endOpacity;
      GetFadeAnimationDetails(startOpacity, endOpacity);
      //Fade implementation reference
      //For Fade
      float lDeltaOpacity = lInterpolation->Interpolate(GetUpdateCount(),
                            startOpacity,
                            endOpacity,
                            count);
      //Add delta opacity values to original Opacity values to correct the animation direction
      float lCurrentOpacity = startOpacity + lDeltaOpacity;
      if (isLayerContent)
      {
         IlmAccessor::SetLayerOpacity(surfaceProperties.layerId,
                                      lCurrentOpacity,
                                      commit,
                                      lIlmError);
      }
      else
      {
         IlmAccessor::SetSurfaceOpacity(surfaceProperties.surfaceId,
                                        lCurrentOpacity,
                                        commit,
                                        lIlmError);
      }
      ETG_TRACE_USR1(("Scale/Translate: Opacity: %f, Dealt Opacity: %f", lCurrentOpacity, lDeltaOpacity));
   }

   return lRc;
}


// ------------------------------------------------------------------------
bool ScaleAndFadeAnimation::SetFinalSurfaceState(const ScaleAndFadeAnimationSurfaceProperties& surfaceProperties, bool commit)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   const Rectangle& lFinalBounds = surfaceProperties.finalPropertyValue;
   //const Rectangle& lOriginalBounds = surfaceProperties.originalPropertyValue;
   const Position& lFinalPosition = lFinalBounds.position;

   ETG_TRACE_USR1(("Scale/Translate: Finalize %40s ID:%u to (%d,%d)/%dx%d",
                   (isLayerContent ? "Layer" : "Surface"),
                   (isLayerContent ? surfaceProperties.layerId : surfaceProperties.surfaceId),
                   lFinalPosition.x, lFinalPosition.y,
                   lFinalBounds.width, lFinalBounds.height));
   bool lRc = false;
   if (!bModifySource)
   {
      if (isLayerContent)
      {
         lRc = IlmAccessor::SetLayerDestinationRectangle(surfaceProperties.layerId,
               lFinalPosition.x, lFinalPosition.y,
               lFinalBounds.width, lFinalBounds.height,
               false,
               lIlmError);
      }
      else
      {
         lRc = IlmAccessor::SetSurfaceDestinationRectangle(surfaceProperties.surfaceId,
               lFinalPosition.x, lFinalPosition.y,
               lFinalBounds.width, lFinalBounds.height,
               false,
               lIlmError);
      }
   }
   else
   {
      if (isLayerContent)
      {
         lRc = IlmAccessor::SetLayerSourceRectangle(surfaceProperties.layerId,
               lFinalPosition.x, lFinalPosition.y,
               lFinalBounds.width, lFinalBounds.height,
               false,
               lIlmError);
      }
      else
      {
         lRc = IlmAccessor::SetSurfaceSourceRectangle(surfaceProperties.surfaceId,
               lFinalPosition.x, lFinalPosition.y,
               lFinalBounds.width, lFinalBounds.height,
               false,
               lIlmError);
      }
   }

   InterpolationStrategy* lInterpolation = GetInterpolation();
   if (0 != lInterpolation)
   {
      lInterpolation->Release();
      SetInterpolation(0);
   }
   float startOpacity, endOpacity;
   GetFadeAnimationDetails(startOpacity, endOpacity);

   //Fade implementation reference
   if (isLayerContent)
   {
      IlmAccessor::SetLayerOpacity(surfaceProperties.layerId,
                                   static_cast<t_ilm_float>(endOpacity),
                                   commit,
                                   lIlmError);
   }
   else
   {
      IlmAccessor::SetSurfaceOpacity(surfaceProperties.surfaceId,
                                     static_cast<t_ilm_float>(endOpacity),
                                     commit,
                                     lIlmError);
   }
   ETG_TRACE_USR1(("ScaleAndFade: Opacity: %f", endOpacity));
   return lRc;
}


void ScaleAndFadeAnimation::GetFadeAnimationDetails(float& start, float& end)
{
   start = initialOpacity;
   end = finalOpacity;
}


void ScaleAndFadeAnimation::SetOpacityDetails(float aStart, float aEnd)
{
   initialOpacity = aStart;
   finalOpacity = aEnd;
}


}
