/* ***************************************************************************************
* FILE:          ComplexAnimation.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ComplexAnimation.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBrokerPlugins_Shared_Animation_ComplexAnimation_h)
#define ScreenBrokerPlugins_Shared_Animation_ComplexAnimation_h

#include <Shared/Animation/AccelerationType.h>
#include <Shared/Animation/AnimationBase.h>
#include <Shared/Animation/Animator.h>
#include <Shared/Animation/AnimationProperties.h>
#include <Shared/Animation/AnimationUserData.h>
#include <Shared/Animation/InterpolationStrategy.h>

namespace ScreenBroker {
class AnimationSurfacePropertiesBase;

///
class ComplexAnimation : public AnimationBase
{
   public:
      typedef void (*BeforeAnimationCallback)(const AnimationSurfacePropertiesBase&);
      typedef void (*AfterAnimationCallback)(const AnimationSurfacePropertiesBase&, bool triggerNext);

      ComplexAnimation() :
         mDuration(AnimationProperties::cDefaultDuration),
         mDelayDuration(AnimationProperties::cDefaultDelayDuration),
         mStepCount(0),
         mTotalStepCount(AnimationProperties::cDefaultUpdateCount),
         mUpdateCount(AnimationProperties::cDefaultUpdateCount),
         mAcceleration(AccelerationType::Linear),
         mAmplitude(1.0F),
         mDirection(AnimationDirection::Undefined),
         mBeforeAnimationCallback(0),
         mAfterAnimationCallback(0),
         mInterpolation(0)
      { }

      virtual ~ComplexAnimation();

      void SetDuration(UInt32 duration)
      {
         mDuration = duration;
      }
      AnimationDirection::Enum GetDirection()
      {
         return mDirection;
      }
      UInt32 GetDuration() const
      {
         return mDuration;
      }

      void SetUpdateCount(UInt32 updateCount)
      {
         mUpdateCount = updateCount;
      }
      UInt32 GetUpdateCount() const
      {
         return mUpdateCount;
      }

      void SetAcceleration(AccelerationType::Enum acceleration)
      {
         mAcceleration = acceleration;
      }
      AccelerationType::Enum GetAcceleration() const
      {
         return mAcceleration;
      }

      void SetAmplitude(Float amplitude)
      {
         mAmplitude = amplitude;
      }
      Float GetAmplitude() const
      {
         return mAmplitude;
      }

      void SetBeforeAnimationCallback(BeforeAnimationCallback callback)
      {
         mBeforeAnimationCallback = callback;
      }
      BeforeAnimationCallback GetBeforeAnimationCallback() const
      {
         return mBeforeAnimationCallback;
      }

      void SetAfterAnimationCallback(AfterAnimationCallback callback)
      {
         mAfterAnimationCallback = callback;
      }
      AfterAnimationCallback GetAfterAnimationCallback() const
      {
         return mAfterAnimationCallback;
      }

      UInt32 GetExpirationPeriod()
      {
         return static_cast<UInt32>(GetDuration() / GetUpdateCount());
      }

      UInt32 GetDelayDuration()
      {
         return mDelayDuration;
      }
      void SetDelayDuration(UInt32 delay)
      {
         mDelayDuration = delay;
      }

      bool Start();
      bool IsActive();
      void Cancel();
      virtual void Reset();
      void AnimationFunction();
      virtual UInt32 GetSurfaceId() = 0;
      //Update From Animator Class
      void AnimateFunction(UInt32 missedSteps, bool commit = false);

   protected:
      virtual bool InitializeSurfaceStates(bool commit = true) = 0;
      virtual bool UpdateSurfaceStates(UInt32 count, bool commit = true) = 0;
      virtual void NotifyAnimationsStart() = 0;
      virtual bool SetFinalSurfaceStates(bool commit = true) = 0;
      virtual void NotifyAnimationsFinished(bool triggerNext) = 0;

      void SetInterpolation(InterpolationStrategy* interpolation)
      {
         mInterpolation = interpolation;
      }
      InterpolationStrategy* GetInterpolation()
      {
         return mInterpolation;
      }

   private:
      UInt32 mDuration;
      UInt32 mDelayDuration;
      UInt32 mStepCount;
      UInt32 mTotalStepCount;
      UInt32 mUpdateCount;
      AccelerationType::Enum mAcceleration;
      Float mAmplitude;
      AnimationDirection::Enum mDirection;
      BeforeAnimationCallback mBeforeAnimationCallback;
      AfterAnimationCallback mAfterAnimationCallback;
      InterpolationStrategy* mInterpolation;
      void FinalizeAnimation(bool commit = true);
};


///
class AnimationSurfacePropertiesBase
{
   public:
      UInt32 surfaceId;
      UInt32 layerId;
      AnimationDirection::Enum animationDirection;
      AnimationUserData animationUserData;

      AnimationSurfacePropertiesBase() :
         surfaceId(0),
         layerId(0),
         animationDirection(AnimationDirection::Undefined)
      {}

      AnimationSurfacePropertiesBase(UInt32 aSurfaceId, UInt32 aLayerId,
                                     AnimationDirection::Enum aAnimationDirection,
                                     const AnimationUserData& aAnimationUserData):
         surfaceId(aSurfaceId),
         layerId(aLayerId),
         animationDirection(aAnimationDirection),
         animationUserData(aAnimationUserData)
      { }
      AnimationDirection::Enum GetAnimationDirection()
      {
         return animationDirection;
      }
};


}

#endif
