/* ***************************************************************************************
* FILE:          ComplexAnimation.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ComplexAnimation.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// This header file is out of scope of sbplugins.
//lint -esym(451, errno.h) "Header file '/usr/include/errno.h' repeatedly included but does not have a standard include guard"
// =============================================================================

#include "ComplexAnimation.h"
#include <Shared/AnimationHandler.h>
#include <Shared/Animation/AnimationPoolBase.h>
#include <Shared/Animation/Animator.h>

#include <errno.h>
#include <sys/time.h>
#include <sys/timerfd.h>
#include <unistd.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/ComplexAnimation.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);

ComplexAnimation::~ComplexAnimation()
{
   mPool = 0;
   mStepCount = 0;
   mTotalStepCount = 0;
}


// ------------------------------------------------------------------------
bool ComplexAnimation::Start()
{
   return Animator::getInstance()->startAnimation(this);
}


// ------------------------------------------------------------------------
bool ComplexAnimation::IsActive()
{
   return ((mStepCount > 0) && (mStepCount < GetUpdateCount()));
}


// ------------------------------------------------------------------------
void ComplexAnimation::Cancel()
{
   // Finalize and cleanup animation
   Animator::getInstance()->stopAnimation(this);
   FinalizeAnimation(true);
}


// ------------------------------------------------------------------------
void ComplexAnimation::Reset()
{
   mDuration = AnimationProperties::cDefaultDuration;
   mUpdateCount = AnimationProperties::cDefaultUpdateCount;
   mAcceleration = AccelerationType::Linear;
   mAmplitude = 1.0F;
   mBeforeAnimationCallback = 0;
   mAfterAnimationCallback = 0;
   mPool = 0;
   mInterpolation = 0;
}


// ------------------------------------------------------------------------
void ComplexAnimation::AnimateFunction(UInt32 missedSteps, bool commit)
{
   ETG_TRACE_USR4_THR(("UpdateSurfaceState with Count: %d and Commit: %d", missedSteps, commit));
   bool lRc = false;
   if (0 == mStepCount)
   {
      NotifyAnimationsStart();
      lRc = InitializeSurfaceStates(true);
      mTotalStepCount = GetUpdateCount();
      mStepCount = 1;
      mTotalStepCount -= mStepCount;
   }
   else
   {
      lRc = true;
   }
   UInt32 lUpdateCountDecrement = (mTotalStepCount > missedSteps) ? missedSteps : mTotalStepCount;
   mStepCount += lUpdateCountDecrement;
   mTotalStepCount = mTotalStepCount - lUpdateCountDecrement;
   ETG_TRACE_USR4_THR(("StepCount: %d ,UpdateCount:%d ,lUpdateCountDecrement: %d", mStepCount, mTotalStepCount, lUpdateCountDecrement));
   lRc = lRc && UpdateSurfaceStates(mStepCount, commit);

   if ((mTotalStepCount == 0) || (!lRc))
   {
      // Finalize and cleanup animation
      Animator::getInstance()->stopAnimation(this);
      FinalizeAnimation(true);
   }
}


// ------------------------------------------------------------------------
void ComplexAnimation::FinalizeAnimation(bool commit)
{
   ETG_TRACE_USR4_THR(("Finalizing Animation with commit:%d", commit));
   SetFinalSurfaceStates(commit);
   NotifyAnimationsFinished(GetAnimationCategory());
   mStepCount = 0;
   mTotalStepCount = GetUpdateCount();
   // Return animation resource back to animation pool
   if (mPool != 0)
   {
      mPool->ReturnAnimation(this);
   }
}


}
