/* ***************************************************************************************
* FILE:          Animator.cpp
* SW-COMPONENT:  HMI-BASE
* DESCRIPTION:  Animator.cpp is part of HMI-Base ScreenBrokerPlugins
* COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include <Shared/Animation/AnimatorTimer.h>
#include "ComplexAnimation.h"
#include "Animator.h"
#include "ScreenBroker/ScreenBroker_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/Animator.cpp.trc.h"
#endif

namespace ScreenBroker {
Animator* Animator::_instanceSelf = 0;
// ------------------------------------------------------------------------
Animator::Animator()
{
   _AnimationDataMap.clear();
   initializeTimer();
}


// ------------------------------------------------------------------------
Animator::~Animator()
{
   _AnimationDataMap.clear();
   AnimatorTimer::destroyInstance();
}


// ------------------------------------------------------------------------
void Animator::initializeTimer()
{
   AnimatorTimer::getInstance()->setCallBackInstance(this);
}


// ------------------------------------------------------------------------
bool Animator::startAnimation(ComplexAnimation* lCompexAnimation)
{
   CriticalSectionLocker lockData(&AnimatorTimer::_criticalSection);
   if (0 != lCompexAnimation)
   {
      if (0 < lCompexAnimation->GetDelayDuration())
      {
         Int32 lTimerFD = AnimatorTimer::getInstance()->createTimer(lCompexAnimation->GetDelayDuration());
         if (AnimatorTimer::getInstance()->startTimer(lTimerFD, lCompexAnimation->GetDelayDuration()), true)
         {
            if (AnimatorTimer::getInstance()->initializeThread())
            {
               std::list<ComplexAnimation*> lCompexAnimationlist;
               lCompexAnimationlist.push_back(lCompexAnimation);
               _EventDelayMap[lTimerFD] = lCompexAnimationlist;
               ETG_TRACE_USR4_THR(("Adding Animation to Delay Queue"));
            }
            else
            {
               lTimerFD = -1;
            }
         }
         else
         {
            lTimerFD = -1;
         }
         if (-1 == lTimerFD)
         {
            ETG_TRACE_ERR_THR(("Initializing & Starting Delay timer Failed"));
         }
         else
         {
            return true;
         }
      }

      if (0 < lCompexAnimation->GetExpirationPeriod())
      {
         Int32 lTimerFD = -1;
         if (((_AnimationDataMap.empty()) || (_AnimationDataMap.find(lCompexAnimation) == _AnimationDataMap.end())))
         {
            lTimerFD = AnimatorTimer::getInstance()->createTimer(lCompexAnimation->GetExpirationPeriod());
            if (-1 != lTimerFD)
            {
               _AnimationDataMap[lCompexAnimation] = lTimerFD;
            }
         }
         if (((_AnimationDataMap.find(lCompexAnimation) != _AnimationDataMap.end())) &&
               (AnimatorTimer::getInstance()->startTimer(_AnimationDataMap.find(lCompexAnimation)->second, lCompexAnimation->GetExpirationPeriod())))
         {
            if (AnimatorTimer::getInstance()->initializeThread())
            {
               return true;
            }
            else
            {
               ETG_TRACE_ERR_THR(("Animation Failed :Finalizing Surface"));
               lCompexAnimation->Cancel();
            }
         }
         else
         {
            ETG_TRACE_ERR_THR(("Starting Delay Timer Failed Canceling Animation"));
            lCompexAnimation->Cancel();
         }
      }
      else
      {
         ETG_TRACE_ERR_THR(("Invalid Expiration Period Ignoring Request"));
      }
   }
   return false;
}


// ------------------------------------------------------------------------
void Animator::InitiateGroupAnimations(std::list<ComplexAnimation*> lCompexAnimationlist)
{
   if (!lCompexAnimationlist.empty())
   {
      CriticalSectionLocker lockData(&AnimatorTimer::_criticalSection);
      std::map<UInt32, std::list<ComplexAnimation*> > lEvenDelayGroupMap;
      std::list<ComplexAnimation*>::iterator iter = lCompexAnimationlist.begin();
      while (iter != lCompexAnimationlist.end())
      {
         if (0 < (*iter)->GetDelayDuration())
         {
            if ((!lEvenDelayGroupMap.empty()) && (lEvenDelayGroupMap.find((*iter)->GetDelayDuration()) != lEvenDelayGroupMap.end()))
            {
               _EventDelayMap.find((*iter)->GetDelayDuration())->second.push_back(*iter);
            }
            else
            {
               std::list<ComplexAnimation*> lEvenDelayGroup;
               lEvenDelayGroup.push_back(*iter);
               lEvenDelayGroupMap[((*iter)->GetDelayDuration())] = lEvenDelayGroup;
            }
            iter = lCompexAnimationlist.erase(iter);
         }
         else
         {
            ++iter;
         }
      }
      for (std::map<UInt32, std::list<ComplexAnimation*> >::iterator iter = lEvenDelayGroupMap.begin(); iter != lEvenDelayGroupMap.end(); ++iter)
      {
         Int32 lTimerFD = AnimatorTimer::getInstance()->createTimer(iter->first);
         if (AnimatorTimer::getInstance()->startTimer(lTimerFD, iter->first), true)
         {
            if (AnimatorTimer::getInstance()->initializeThread())
            {
               _EventDelayMap[lTimerFD] = iter->second;
               ETG_TRACE_USR4_THR(("Start Delay Timer of : %u ms with Timer ID: %d", iter->first, lTimerFD));
            }
            else
            {
               lTimerFD = -1;
            }
         }
         else
         {
            lTimerFD = -1;
         }
         if (-1 == lTimerFD)
         {
            ETG_TRACE_ERR_THR((" Delay Timer failed to start for Duration: %d ,Starting instantly", iter->first));
            lCompexAnimationlist.merge(iter->second);
         }
      }
      startBatchAnimation(lCompexAnimationlist);
   }
}


// ------------------------------------------------------------------------
void Animator::startBatchAnimation(std::list<ComplexAnimation*> lCompexAnimationlist)
{
   if (!lCompexAnimationlist.empty())
   {
      CriticalSectionLocker lockData(&AnimatorTimer::_criticalSection);
      std::map<UInt32, std::list<ComplexAnimation*> > lEvenExpirationMap;
      for (std::list<ComplexAnimation*>::iterator it = lCompexAnimationlist.begin(); it != lCompexAnimationlist.end(); ++it)
      {
         if ((!lEvenExpirationMap.empty()) && (lEvenExpirationMap.find((*it)->GetExpirationPeriod()) != lEvenExpirationMap.end()))
         {
            lEvenExpirationMap.find((*it)->GetExpirationPeriod())->second.push_back(*it);
         }
         else
         {
            std::list<ComplexAnimation*> lEvenExpirationGroup;
            lEvenExpirationGroup.push_back(*it);
            lEvenExpirationMap[((*it)->GetExpirationPeriod())] = lEvenExpirationGroup;
         }
      }
      for (std::map<UInt32, std::list<ComplexAnimation*> >::iterator iter = lEvenExpirationMap.begin(); iter != lEvenExpirationMap.end(); ++iter)
      {
         Int32 lTimerFD = -1;
         if (0 < iter->first)
         {
            lTimerFD = AnimatorTimer::getInstance()->createTimer(iter->first);
            for (std::list<ComplexAnimation*>::iterator it = iter->second.begin(); it != iter->second.end(); ++it)
            {
               if (_AnimationDataMap.find(*it) != _AnimationDataMap.end())
               {
                  stopAnimation(*it);
               }
               _AnimationDataMap[*it] = lTimerFD;
            }
            if (AnimatorTimer::getInstance()->startTimer(lTimerFD, iter->first))
            {
               if (!AnimatorTimer::getInstance()->initializeThread())
               {
                  lTimerFD = -1;
               }
               else
               {
                  ETG_TRACE_USR4_THR(("Timer Started For Batch with Expiration Time: %d ", iter->first));
               }
            }
            else
            {
               ETG_TRACE_ERR_THR(("Timer Start Failed Finalizing Animations"));
               lTimerFD = -1;
            }
         }
         else
         {
            ETG_TRACE_ERR_THR(("Invalid Expiration Period Finalizing the Batch"));
         }
         if (-1 == lTimerFD)
         {
            for (std::list<ComplexAnimation*>::iterator it = iter->second.begin(); it != iter->second.end(); ++it)
            {
               (*it)->Cancel();
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("Empty Batch for Animation"));
   }
}


// ------------------------------------------------------------------------
void Animator::stopAnimation(ComplexAnimation* lCompexAnimation)
{
   CriticalSectionLocker lockData(&AnimatorTimer::_criticalSection);
   if (!_AnimationDataMap.empty())
   {
      Int32 lTimerFD = -1;
      if ((0 != lCompexAnimation) && (_AnimationDataMap.find(lCompexAnimation) != _AnimationDataMap.end()))
      {
         lTimerFD = _AnimationDataMap.find(lCompexAnimation)->second;
         _AnimationDataMap.erase(lCompexAnimation);
         if (!_AnimationDataMap.empty())
         {
            AnimationDataMap::iterator it = _AnimationDataMap.begin();
            for (; it != _AnimationDataMap.end(); ++it)
            {
               if (it->second == lTimerFD)
               {
                  ETG_TRACE_USR4_THR(("Animator :%d is Still Valid", lTimerFD));
                  break;
               }
            }
            if (it == _AnimationDataMap.end())
            {
               ETG_TRACE_USR4_THR(("Animator :%d is expired", lTimerFD));
               AnimatorTimer::getInstance()->destroyTimer(lTimerFD);
            }
         }
      }
      else
      {
         ETG_TRACE_ERR_THR(("Invalid Entry in AnimationDataMap"));
      }
   }

   if (!(_EventDelayMap.empty()))
   {
      Int32 lTimerFD = -1;
      EventDelayMap::iterator iter = _EventDelayMap.begin();
      while (iter != _EventDelayMap.end())
      {
         std::list<ComplexAnimation*> lCompexAnimationlist = iter->second;
         std::list<ComplexAnimation*>::iterator it = lCompexAnimationlist.begin();
         while (it != lCompexAnimationlist.end())
         {
            if ((*it) == lCompexAnimation)
            {
               lTimerFD = iter->first;
               iter->second.remove(lCompexAnimation);
               break;
            }
            else
            {
               ++it;
            }
         }
         if ((-1 != lTimerFD) && (iter->second.empty()))
         {
            ETG_TRACE_USR4_THR(("Animator :%d is expired", lTimerFD));
            _EventDelayMap.erase(iter++);
            AnimatorTimer::getInstance()->destroyTimer(lTimerFD);
            break;
         }
         else
         {
            iter++;
         }
      }
   }

   if ((_EventDelayMap.empty()) && (_AnimationDataMap.empty()))
   {
      ETG_TRACE_ERR_THR(("Terminating Thread due to Empty EventDelayMap && AnimationDataMap"));
      AnimatorTimer::getInstance()->terminateThread();
   }
}


// ------------------------------------------------------------------------
void Animator::abortAllAnimations()
{
   ETG_TRACE_USR4_THR(("Abort All Running Animations"));
   CriticalSectionLocker lockData(&AnimatorTimer::_criticalSection);
   _AnimationDataMap.clear();
   _EventDelayMap.clear();
   AnimatorTimer::getInstance()->destroyTimer(-1);
   AnimatorTimer::getInstance()->terminateThread();
}


// ------------------------------------------------------------------------
void Animator::expiredTimerUpdate(Int32 timerFD, UInt32 missed)
{
   CriticalSectionLocker lockData(&AnimatorTimer::_criticalSection);
   ETG_TRACE_USR4_THR(("Expired Timer :%d Update with Missed:%d", timerFD , missed));
   std::list<ComplexAnimation*> lCompexAnimationlist;
   for (AnimationDataMap::iterator it = _AnimationDataMap.begin(); it != _AnimationDataMap.end(); ++it)
   {
      if (((it)->first != 0) && ((it)->second == timerFD))
      {
         lCompexAnimationlist.push_back((it->first));
      }
   }
   size_t lListSize = lCompexAnimationlist.size();
   for (std::list<ComplexAnimation*>::iterator it = lCompexAnimationlist.begin(); it != lCompexAnimationlist.end(); ++it)
   {
      (*it)->AnimateFunction(missed, ((1 < lListSize) ? false : true));
      lListSize --;
   }
   EventDelayMap::iterator iter = _EventDelayMap.begin();
   while (iter != _EventDelayMap.end())
   {
      if ((iter)->first == timerFD)
      {
         ETG_TRACE_USR4_THR(("One Shot Timer Expired TimerID: %d", timerFD));
         AnimatorTimer::getInstance()->destroyTimer(timerFD);
         if (!((iter)->second).empty())
         {
            startBatchAnimation(((iter)->second));
         }
         _EventDelayMap.erase(iter++);
         break;
      }
      else
      {
         ++iter;
      }
   }
}


}
