/* ***************************************************************************************
* FILE:          AnimationFactory.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AnimationFactory.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#include "AnimationFactory.h"

#include "AnimationPool.h"
#include "AnimationUserData.h"
#include "FadingAnimation.h"
#include "ScaleAndTranslationAnimation.h"
#include "SimpleAnimation.h"
#include "SlidingAnimation.h"
#include "ScaleAndFadeAnimation.h"
#include <Shared/IlmAccessor.h>
#include <ScreenBroker/Service/ServiceApi.h>
#include <ScreenLayouter/ScreenLayouter.h>

#include <Shared/PluginActions.h>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/AnimationFactory.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::Animation);

// ------------------------------------------------------------------------
typedef AnimationPool<FadingAnimation, AnimationFactory::cMaxParallelFadeAnimations> FadingAnimationPool;
typedef AnimationPool<ScaleAndTranslationAnimation, AnimationFactory::cMaxParallelZoomAnimations> ScaleAndTranslationAnimationPool;
typedef AnimationPool<ScaleAndFadeAnimation, AnimationFactory::cMaxParallelZoomAndFadeAnimations> ScaleAndFadeAnimationPool;
typedef AnimationPool<SlidingAnimation, AnimationFactory::cMaxParallelSlidingAnimations> SlidingAnimationPool;
typedef AnimationPool<SimpleAnimation, AnimationFactory::cMaxParallelSimpleAnimations> SimpleAnimationPool;

// ------------------------------------------------------------------------
AnimationFactory::SurfaceBoundsMap AnimationFactory::mInitialSurfaceBounds;
AnimationFactory::SurfaceBoundsMap AnimationFactory::mInitialLayerBounds;
// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateAnimation(AnimationType::Enum animationType,
      UInt32 surfaceId,
      const AnimationUserData& animationUserData,
      AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   AnimationBase* lAnimation = 0;
   switch (animationType)
   {
      case AnimationType::FadeIn:
         lAnimation = CreateFadeInAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::FadeOut:
         lAnimation = CreateFadeOutAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomInFromTopLeft:
         lAnimation = CreateZoomInFromTopLeftAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomOutToTopLeft:
         lAnimation = CreateZoomOutToTopLeftAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideInFromLeft:
         lAnimation = CreateSlideInFromLeftAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideOutToLeft:
         lAnimation = CreateSlideOutToLeftAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideInFromTop:
         lAnimation = CreateSlideInFromTopAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideOutToTop:
         lAnimation = CreateSlideOutToTopAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideInFromRight:
         lAnimation = CreateSlideInFromRightAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideOutToRight:
         lAnimation = CreateSlideOutToRightAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideInFromBottom:
         lAnimation = CreateSlideInFromBottomAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::SlideOutToBottom:
         lAnimation = CreateSlideOutToBottomAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomInFromTopRight:
         lAnimation = CreateZoomInFromTopRightAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomOutToTopRight:
         lAnimation = CreateZoomOutToTopRightAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomInFromBottomLeft:
         lAnimation = CreateZoomInFromBottomLeftAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomOutToBottomLeft:
         lAnimation = CreateZoomOutToBottomLeftAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomInFromBottomRight:
         lAnimation = CreateZoomInFromBottomRightAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomOutToBottomRight:
         lAnimation = CreateZoomOutToBottomRightAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomInFromCenter:
         lAnimation = CreateZoomInFromCenterAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomOutToCenter:
         lAnimation = CreateZoomOutToCenterAnimation(surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ZoomInFadeIn:
      case AnimationType::ZoomInFadeOut:
      case AnimationType::ZoomOutFadeIn:
      case AnimationType::ZoomOutFadeOut:
         lAnimation = CreateZoomPlusFadeAnimation(animationType, surfaceId, animationUserData, animationDirection, isLayerContent);
         break;
      case AnimationType::ShowImmediately:
         lAnimation = CreateShowImmediatelyAnimation(surfaceId, animationUserData, isLayerContent);
         break;
      case AnimationType::HideImmediately:
         lAnimation = CreateHideImmediatelyAnimation(surfaceId, animationUserData, isLayerContent);
         break;
      default:
         break;
   }
   if (0 != lAnimation)
   {
      ETG_TRACE_USR1(("Created animation (%40s) for %40s %u",
                      (isLayerContent ? "Layer" : "Surface"),
                      PluginActions::AnimationTypeString(animationType).c_str(),
                      surfaceId));
   }
   else
   {
      ETG_TRACE_SYS(("Failed to create animation (%40s) for %40s %u!",
                     (isLayerContent ? "Layer" : "Surface"),
                     PluginActions::AnimationTypeString(animationType).c_str(),
                     surfaceId));
   }

   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateFadeInAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   FadingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,   // Hide/show surface after finished animation
      animationUserData,          // user data for animation post processing
      0.0f,                       // initial opacity
      1.0f                        // final opacity
   );
   FadingAnimation* lAnimation =
      dynamic_cast<FadingAnimation*>(FadingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateFadeOutAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   FadingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      1.0f,
      0.0f
   );
   FadingAnimation* lAnimation =
      dynamic_cast<FadingAnimation*>(FadingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomInFromTopLeftAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(0, 0, 0, 0),
      lBounds);

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomInFromCenterAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;

   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(lBounds.width / 2, lBounds.height / 2, 0, 0),
      lBounds);

   ScaleAndFadeAnimation* lAnimation =
      dynamic_cast<ScaleAndFadeAnimation*>(ScaleAndFadeAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
      lAnimation->SetOpacityDetails(0.0, 1.0);
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomOutToCenterAnimation(UInt32 surfaceId,
      const AnimationUserData& animationUserData,
      AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(lBounds.width / 2, lBounds.height / 2, 0, 0));

   ScaleAndFadeAnimation* lAnimation =
      dynamic_cast<ScaleAndFadeAnimation*>(ScaleAndFadeAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
      lAnimation->SetOpacityDetails(1.0, 0.0);
   }
   return lAnimation;
}


void AnimationFactory::DeterminePosXAndWidth(UInt32 surfaceId, Int32& posX, UInt32& width, UInt32 scale, Int32 pivotX)
{
   Rectangle lBounds;
   //Determine X or Y position
   bool lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   if (!lRc)
   {
      return;
   }
   posX = (pivotX - ((pivotX * scale) / 100));

   if (posX < 0)
   {
      width = lBounds.width + 2 * posX;
   }
   else
   {
      width = lBounds.width - 2 * posX;
   }
}


void AnimationFactory::DeterminePosYAndHeight(UInt32 surfaceId, Int32& posY, UInt32& height, UInt32 scale, Int32 pivotY)
{
   Rectangle lBounds;
   //Determine X or Y position
   bool lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   if (!lRc)
   {
      return;
   }
   posY = (pivotY - ((pivotY * scale) / 100));

   if (posY < 0)
   {
      height = lBounds.height + 2 * posY;
   }
   else
   {
      height = lBounds.height - 2 * posY;
   }
}


AnimationBase* AnimationFactory::CreateZoomPlusFadeAnimation(AnimationType::Enum animationType,
      UInt32 surfaceId,
      const AnimationUserData& animationUserData,
      AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   Rectangle lBounds;
   Rectangle sourceRect;
   Rectangle destRect;
   float initialOpacity = 0.0f;
   float finalOpacity = 0.0f;
   Int32 pivotX = 0;
   Int32 pivotY = 0;

   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }
   UInt32 scaleFrom = 100;
   UInt32 scaleTo = 100;
   PivotPoint::Enum pivotPoint = PivotPoint::Undefined;

   switch (animationType)
   {
      case AnimationType::ZoomInFadeIn:
      case AnimationType::ZoomOutFadeIn:
         initialOpacity = 0.0f;
         finalOpacity = 1.0f;
         break;
      case AnimationType::ZoomInFadeOut:
      case AnimationType::ZoomOutFadeOut:
         initialOpacity = 1.0f;
         finalOpacity = 0.0f;
         break;
      default:
         break;
   }

   // Retrieve instance of screen layouter
   ScreenLayouter* lScreenLayouter = PLUGIN(ScreenLayouter);
   if (0 != lScreenLayouter)
   {
      const AnimationProperties* lAnimationProperties = lScreenLayouter->GetAnimationProperties(surfaceId, animationDirection, animationType);
      if (lAnimationProperties != NULL)
      {
         scaleFrom = lAnimationProperties->scaleFrom;
         scaleTo = lAnimationProperties->scaleTo;
         pivotPoint = lAnimationProperties->pivotPoint;
      }
   }

   Int32 posSourceX = 0;
   Int32 posSourceY = 0;
   UInt32 sourceWidth = 0;
   UInt32 sourceHeight = 0;

   switch (pivotPoint)
   {
      case PivotPoint::Center:
         pivotX = lBounds.width;
         pivotY = lBounds.height;
         break;
         /* Other pivot points are yet to implement */
         /*case PivotPoint::TopLeft:
                break;*/
      default:
         break;
   }

   DeterminePosXAndWidth(surfaceId, posSourceX, sourceWidth, scaleFrom, pivotX);
   DeterminePosYAndHeight(surfaceId, posSourceY, sourceHeight, scaleFrom, pivotY);

   ETG_TRACE_USR1(("CreateZoom + Fade Animation: Source -(%d,%d,%u,%u), Scale(%d, %d)",
                   posSourceX,
                   posSourceY,
                   sourceWidth,
                   sourceHeight,
                   scaleFrom,
                   scaleTo));
   Int32 posDestX = 0;
   Int32 posDestY = 0;
   UInt32 destWidth = 0;
   UInt32 destHeight = 0;

   DeterminePosXAndWidth(surfaceId, posDestX, destWidth, scaleTo, pivotX);
   DeterminePosYAndHeight(surfaceId, posDestY, destHeight, scaleTo, pivotY);

   ETG_TRACE_USR1(("CreateZoom + Fade Animation: Destination -(%d,%d,%u,%u)",
                   posDestX,
                   posDestY,
                   destWidth,
                   destHeight));

   destRect = Rectangle(posDestX, posDestY, destWidth, destHeight);
   sourceRect = Rectangle(posSourceX, posSourceY, sourceWidth, sourceHeight);
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   ScaleAndFadeAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : animationDirection), //Set show if animation type is custom
      animationUserData,
      sourceRect,
      destRect);

   ScaleAndFadeAnimation* lAnimation =
      dynamic_cast<ScaleAndFadeAnimation*>(ScaleAndFadeAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->SetOpacityDetails(initialOpacity, finalOpacity);

      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      //Set OnBeforeCustomAnimation if animation type is custom else OnBeforeAnimation
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomOutToTopLeftAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(0, 0, 0, 0));

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideInFromLeftAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(-lBounds.width, lBounds.position.y, lBounds.width, lBounds.height),
      lBounds);

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideOutToLeftAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(-lBounds.width, lBounds.position.y, lBounds.width, lBounds.height));

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideInFromTopAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(lBounds.position.x, -lBounds.height, lBounds.width, lBounds.height),
      lBounds);

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideOutToTopAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(lBounds.position.x, -lBounds.height, lBounds.width, lBounds.height));

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideInFromRightAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }
   //PosX need not be double of width
   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(lBounds.width, lBounds.position.y, lBounds.width, lBounds.height),
      lBounds);

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideOutToRightAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }
   //PosX need not be double of width
   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(lBounds.width, lBounds.position.y, lBounds.width, lBounds.height));

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideInFromBottomAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }
   //Posy need not be double of height
   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(lBounds.position.x, lBounds.height, lBounds.width, lBounds.height),
      lBounds);

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateSlideOutToBottomAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }
   //Posy need not be double of height
   SlidingAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(lBounds.position.x, lBounds.height, lBounds.width, lBounds.height));

   SlidingAnimation* lAnimation =
      dynamic_cast<SlidingAnimation*>(SlidingAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomInFromTopRightAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(lBounds.width, 0, 0, 0),
      lBounds);

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());
   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomOutToTopRightAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }
   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(lBounds.width, 0, 0, 0));

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());

   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomInFromBottomLeftAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(0, lBounds.height, 0, 0),
      lBounds);

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());

   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomOutToBottomLeftAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(0, lBounds.height, 0, 0));

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());

   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomInFromBottomRightAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }
   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      AnimationDirection::Show,
      animationUserData,
      Rectangle(lBounds.width, lBounds.height, 0, 0),
      lBounds);

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());

   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateZoomOutToBottomRightAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, AnimationDirection::Enum animationDirection, bool isLayerContent)
{
   bool isCustomAnimation = (AnimationDirection::Custom == animationDirection);
   Rectangle lBounds;
   bool lRc = false;

   if (isLayerContent)
   {
      lRc = GetInitialLayerBounds(surfaceId, lBounds);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   }
   if (!lRc)
   {
      return 0;
   }

   ScaleAndTranslationAnimationSurfaceProperties lSurfaceProperties(
      (isLayerContent ? 0 : surfaceId), //Surface ID
      (isLayerContent ? surfaceId : 0), //Layer ID
      (isCustomAnimation ? AnimationDirection::Show : AnimationDirection::Hide), //Set show if animation type is custom
      animationUserData,
      lBounds,
      Rectangle(lBounds.width, lBounds.height, 0, 0));

   ScaleAndTranslationAnimation* lAnimation =
      dynamic_cast<ScaleAndTranslationAnimation*>(ScaleAndTranslationAnimationPool::GetInstance().GetAnimation());

   if (lAnimation != 0)
   {
      lAnimation->AddSurfaceProperties(lSurfaceProperties);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetBeforeAnimationCallback((isCustomAnimation ? OnBeforeCustomAnimation : OnBeforeAnimation));
   }
   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateShowImmediatelyAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, bool isLayerContent)
{
   SimpleAnimation* lAnimation =
      dynamic_cast<SimpleAnimation*>(SimpleAnimationPool::GetInstance().GetAnimation());

   if (0 != lAnimation)
   {
      lAnimation->SetAnimationDirection(AnimationDirection::Show);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetSurfaceId(isLayerContent ? 0 : surfaceId);
      lAnimation->SetLayerId(isLayerContent ? surfaceId : 0);
      lAnimation->SetAnimationUserData(animationUserData);

      // set initial surface bounds to restore surface bounds on lRcanimation start
      Rectangle lBounds;
      bool lrc = GetInitialSurfaceBounds(surfaceId, lBounds);
      if (!lrc)
      {
         return 0;
      }
      lAnimation->SetInitialSurfaceBounds(lBounds);
   }

   return lAnimation;
}


// ------------------------------------------------------------------------
AnimationBase* AnimationFactory::CreateHideImmediatelyAnimation(UInt32 surfaceId, const AnimationUserData& animationUserData, bool isLayerContent)
{
   SimpleAnimation* lAnimation =
      dynamic_cast<SimpleAnimation*>(SimpleAnimationPool::GetInstance().GetAnimation());

   if (0 != lAnimation)
   {
      lAnimation->SetAnimationDirection(AnimationDirection::Hide);
      lAnimation->setContentType(isLayerContent);
      lAnimation->SetSurfaceId(isLayerContent ? 0 : surfaceId);
      lAnimation->SetLayerId(isLayerContent ? surfaceId : 0);
      lAnimation->SetAnimationUserData(animationUserData);

      // set initial surface bounds to restore surface bounds on animation start
      Rectangle lBounds;
      bool lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
      if (!lRc)
      {
         return 0;
      }
      lAnimation->SetInitialSurfaceBounds(lBounds);
   }

   return lAnimation;
}


// ------------------------------------------------------------------------
bool AnimationFactory::GetInitialSurfaceBounds(UInt32 surfaceId, Rectangle& bounds)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   SurfaceBoundsMapIterator lIt = mInitialSurfaceBounds.find(surfaceId);
   if (lIt != mInitialSurfaceBounds.end())
   {
      bounds = lIt->second;
      return true;
   }

   t_ilm_int lBounds[4];
   bool lRc = IlmAccessor::GetSurfaceDestinationRectangle(surfaceId,
              lBounds[0],
              lBounds[1],
              lBounds[2],
              lBounds[3],
              lIlmError);
   if (!lRc)
   {
      return false;
   }

   bounds = Rectangle(lBounds[0], lBounds[1], lBounds[2], lBounds[3]);
   mInitialSurfaceBounds[surfaceId] = bounds;
   return true;
}


// ------------------------------------------------------------------------
bool AnimationFactory::GetInitialLayerBounds(UInt32 layerId, Rectangle& bounds)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   SurfaceBoundsMapIterator lIt = mInitialLayerBounds.find(layerId);
   if (lIt != mInitialLayerBounds.end())
   {
      bounds = lIt->second;
      return true;
   }

   t_ilm_int lBounds[4];
   bool lRc = IlmAccessor::GetLayerRectangle(layerId,
              lBounds[0],
              lBounds[1],
              lBounds[2],
              lBounds[3],
              lIlmError);
   if (!lRc)
   {
      return false;
   }

   bounds = Rectangle(lBounds[0], lBounds[1], lBounds[2], lBounds[3]);
   mInitialLayerBounds[layerId] = bounds;
   return true;
}


// ------------------------------------------------------------------------
bool AnimationFactory::InitializeSurfaceState(UInt32 surfaceId)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   Rectangle lBounds;
   bool lRc = GetInitialSurfaceBounds(surfaceId, lBounds);
   lRc = lRc && IlmAccessor::SetSurfaceOpacity(surfaceId, 1.0, false, lIlmError);
   lRc = lRc && IlmAccessor::SetSurfaceRectangle(surfaceId, lBounds.position.x, lBounds.position.y, lBounds.width, lBounds.height, false, lIlmError);
   lRc = lRc && IlmAccessor::SetSurfaceVisibility(surfaceId, ILM_TRUE, false, lIlmError);
   return lRc;
}


// ------------------------------------------------------------------------
bool AnimationFactory::InitializeLayerState(UInt32 layerId)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   Rectangle lBounds;
   bool lRc = GetInitialLayerBounds(layerId, lBounds);
   lRc = lRc && IlmAccessor::SetLayerOpacity(layerId, 1.0, false, lIlmError);
   lRc = lRc && IlmAccessor::SetLayerRectangle(layerId, lBounds.position.x, lBounds.position.y, lBounds.width, lBounds.height, false, lIlmError);
   lRc = lRc && IlmAccessor::SetLayerVisibility(layerId, ILM_TRUE, false, lIlmError);
   return lRc;
}


// ------------------------------------------------------------------------
void AnimationFactory::OnBeforeAnimation(const AnimationSurfacePropertiesBase& surfaceProperties)
{
   if (0 != surfaceProperties.layerId)
   {
      InitializeLayerState(surfaceProperties.layerId);
      PluginActions::SendLayerAnimationState(surfaceProperties.animationUserData.userData,
                                             surfaceProperties.layerId,
                                             ScreenBroker::LayerState::AnimationStarted);
   }
   else
   {
      (void)InitializeSurfaceState(surfaceProperties.surfaceId);
      // Notify application about successful change of surface state in case surface gets visible before animation
      if (PluginActions::CalculateVisibility(surfaceProperties.animationUserData.surfaceState))
      {
         PluginActions::SendNotification(surfaceProperties.surfaceId,
                                         surfaceProperties.animationUserData);
      }

      PluginActions::SendAnimationState(surfaceProperties.animationUserData.userData,
                                        surfaceProperties.surfaceId,
                                        ScreenBroker::SurfaceState::AnimationStarted);
   }
}


void AnimationFactory::OnBeforeCustomAnimation(const AnimationSurfacePropertiesBase& surfaceProperties)
{
   ilmErrorTypes lIlmError = ILM_SUCCESS;
   Rectangle lBounds;
   bool lRc = false;
   if (0 != surfaceProperties.layerId)
   {
      InitializeLayerState(surfaceProperties.layerId);
      PluginActions::SendLayerAnimationState(surfaceProperties.animationUserData.userData,
                                             surfaceProperties.layerId,
                                             ScreenBroker::LayerState::AnimationStarted);
   }
   else
   {
      lRc = GetInitialSurfaceBounds(surfaceProperties.surfaceId, lBounds);
      lRc = lRc && IlmAccessor::SetSurfaceRectangle(surfaceProperties.surfaceId, lBounds.position.x, lBounds.position.y, lBounds.width, lBounds.height, false, lIlmError);
      //lRc = lRc && IlmAccessor::SetSurfaceVisibility(surfaceProperties.surfaceId, ILM_TRUE, false, lIlmError);

      // Notify application about successful change of surface state in case surface gets visible before animation
      if (lRc && PluginActions::CalculateVisibility(surfaceProperties.animationUserData.surfaceState))
      {
         PluginActions::SendNotification(surfaceProperties.surfaceId,
                                         surfaceProperties.animationUserData);
      }

      PluginActions::SendAnimationState(surfaceProperties.animationUserData.userData,
                                        surfaceProperties.surfaceId,
                                        ScreenBroker::SurfaceState::AnimationStarted);
   }
}


}
