/* ***************************************************************************************
* FILE:          ActivePopups.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ActivePopups.h is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBrokerPlugins_Shared_ActivePopups_h)
#define ScreenBrokerPlugins_Shared_ActivePopups_h

#include "FocusList.h"
#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/Plugin/IScreenBrokerPlugin.h>
#include <list>
#include <map>

#ifndef TESTER_DECLARE_FRIEND
#define TESTER_DECLARE_FRIEND
#endif

#ifndef TESTER_DEFINE_FRIEND
#define TESTER_DEFINE_FRIEND
#endif

TESTER_DECLARE_FRIEND
namespace ScreenBroker {
///
class PopupState;

///
class ActivePopups
{
   public:
      ///
      static ActivePopups& GetInstance();

      ///
      ActivePopups();

      /**
       * @brief Creates a copy of the given original popup state and stores it into the internal popup state map,
       * if its popupId (aka. userData) not yet exists in this list.
       * If popupId already exists the given popup state parameters are simply overwritten.
       * @param sourcePopupState Popup state to insert.
       * @param destinationPopupState Created or updated popup state object, which will be maintained in active popups container.
       * @return Overall top popup state of current active application (with no respect to filter & visibility).
       */
      PopupState* Insert(const PopupState& sourcePopupState,
                         PopupState*& destinationPopupState);

      /**
       * @brief Remove given popup state.
       * @param popupState Popup state to remove.
       * @return Overall top popup state of current active application (with no respect to filter & visibility).
       */
      PopupState* Remove(PopupState* popupState);

      /**
       * @brief Retrieve the overall top most popup state, considering the priority
       * of the application modal and the system modal popups
       * @note Popup filter for given application may influence the result!
       * @param screenArea Only consider popups of a certain screen area if screenArea != 0.
       * @return Top popup state, based on priority (with no respect to modality),
       * but result is depending on filter considered or not.
       */
      PopupState* GetOverallTop(bool applyFilter = false,
                                UInt32 screenArea = 0);

      /**
       * @brief Retrieve the top most system popup state.
       * @note Popup filter for given application may influence the result!
       * @param screenArea Only consider popups of a certain screen area if screenArea != 0.
       * @return Top system popup state (based on priority).
       */
      PopupState* GetSysTop(bool applyFilter = false,
                            UInt32 screenArea = 0) const;

      /**
       * @brief Retrieve the top most popup state of given application.
       * @note Popup filter for given application may influence the result!
       * @param screenArea Only consider popups of a certain screen area if screenArea != 0.
       * @return Top application modal popup state of given application (based on priority).
       */
      PopupState* GetAppTop(bool applyFilter = false,
                            UInt32 screenArea = 0);

      /**
       * @brief Retrieve the first (lowest prior) popup state of all active popups (system and active application).
       * Note: To iterate through popup list call GetOverallNext().
       * @return First (lowest prior) popup state of all active popups (based on priority) or
       * 0 if no popup state is queued for this application.
       */
      PopupState* GetOverallFirst(UInt32 screenArea = 0);

      /**
       * @brief Retrieve the next prior popup state of all active popups (system and active application).
       * Note: To initialize the internally held iterator, call GetOverallNext().
       * @return Next popup state of all active popups (based on priority) or
       * 0 if end of popup list is reached.
       */
      PopupState* GetOverallNext(UInt32 screenArea = 0);

      /**
       * @brief Retrieve the last (highest prior) popup state of all active popups (system and active application).
       * Note: To iterate through popup list call GetOverallPrevious().
       * @return Last (highest prior) popup state of all active popups (based on priority) or
       * 0 if no popup state is queued for this application.
       */
      PopupState* GetOverallLast(UInt32 screenArea = 0);

      /**
       * @brief Retrieve the previous prior popup state of all active popups (system and active application).
       * Note: To initialize the internally held iterator, call GetOverallLast().
       * @return Previous popup state of all active popups (based on priority) or
       * 0 if end of popup list is reached.
       */
      PopupState* GetOverallPrevious(UInt32 screenArea = 0);

      /**
       * @brief Retrieve the first (lowest prior) popup state of system modal popups.
       * Note: To iterate through popup list call GetNext().
       * @param screenArea Only consider popups of a certain screen area if screenArea != 0.
       * @return First (lowest prior) popup state of system modal popups (based on priority) or
       * 0 if no popup state is queued for this application.
       */
      PopupState* GetSysFirst(UInt32 screenArea = 0);

      /**
       * @brief Retrieve the next prior popup state of given application.
       * Note: To initialize the internally held iterator, call GetFirst().
       * @param screenArea Only consider popups of a certain screen area if screenArea != 0.
       * @return Next popup state of system modal popups (based on priority) or
       * 0 if end of popup list is reached.
       */
      PopupState* GetSysNext(UInt32 screenArea = 0);

      /**
       * @brief Retrieve the first (lowest prior) popup state of given application.
       * Note: To iterate through popup list call GetNext(appId).
       * @param screenArea Only consider popups of a certain screen area if screenArea != 0.
       * @return First (lowest prior) popup state of given application (based on priority) or
       * 0 if no popup state is queued for this application.
       */
      PopupState* GetAppFirst(UInt32 screenArea = 0, bool refreshActiveApplicationList = true);

      /**
       * @brief Retrieve the next prior popup state of given application.
       * Note: To initialize the internally held iterator, call GetFirst(appId).
       * @param screenArea Only consider popups of a certain screen area if screenArea != 0.
       * @return Next popup state of given application (based on priority) or
       * 0 if end of popup list is reached.
       */
      PopupState* GetAppNext(UInt32 screenArea = 0, bool refreshActiveApplicationPopupListIt = false);

      /**
       * @brief Searches active popups container if popup with
       * given contentId (aka. popup state user data) is queued.
       * @param contentId Content ID to check against popup state user data.
       * @return Popup state instance found in active popups container, if found, 0 otherwise.
       */
      PopupState* Find(UInt32 contentId);

      /**
       * @brief Get AppId for given surface
       * @param surfaceId of popup
       * @return AppId
       */
      UInt32 GetAppId(UInt32 surfaceId);

      /// @brief Dump all active application and system popups.
      void Dump();

      /**
       * @brief Dump the priority sorted list of popups considering system popups and
       * popups of given application ID.
       * @param appId Application ID of popups considered in this priority dump.
       */
      void DumpPriorityList(UInt32 appId);

      /**
       * @brief A test driver for checking correct insertion with random
       * generated popupStates.
       * @note Resets the complete state of ActivePopups.
       * @param appCount Number of applications used for test.
       * @param popupCount Number of random generated popupStates.
       * @param maxPriority Defining priority range (1..maxPriority).
       */
      void Test(UInt32 appCount,
                UInt32 popupCount,
                UInt32 maxPriority);

      /**
       * @brief Completely reset the internal states.
       */
      void Reset();

   private:
      TESTER_DEFINE_FRIEND
      typedef std::map<UInt32, PopupState> PopupStateMap;
      typedef std::list<PopupState*> PopupList;
      typedef std::map<UInt32, PopupList> ApplicationPopupMap;
      typedef std::map<PopupState*, UInt32> PopupInsertionOrderMap;

      /// Create separate list of popup state copies to be able maintain the multiple popup scenes (contents) for one surface.
      /// This instance then has be used through-out the whole plugin implementation (even within the PopupManager).
      /// The popup state instance coming from the PopupManagerActivator interfaces ShowPopup() and HidePopup() (application request),
      /// shall only be used to find the corresponding popupState copy in this map using the popupID (the ID of the surface content,
      /// which is transferred via userData).
      PopupStateMap mPopupStateMap;

      /// Priority ordered (global) system popup list
      PopupList mSystemPopupList;

      /// Application map holding priority ordered (local) application popup list
      ApplicationPopupMap mApplicationPopupMap;

      /// Represents a merged list of system popups and the popups of the currently active application
      /// This list is initialized only via GetOverallFirst and only to be used for
      /// iteration via GetOverallNext.
      PopupList mOverallPopupList;

      /// Points to currently active application popup list (for use in iteration functions only!)
      PopupList mActiveApplicationPopupList;

      /// Iterator for current system popup list (for use in iteration functions only!)
      PopupList::const_iterator mSystemPopupListIt;

      /// Iterator for current active application popup list (for use in iteration functions only!)
      PopupList::const_iterator mActiveApplicationPopupListIt;

      /// Iterator for current overall popup list (for use in iteration functions only!)
      PopupList::const_iterator mOverallPopupListIt;

      /// Reverse Iterator for current overall popup list (for use in iteration functions only!)
      PopupList::const_reverse_iterator mOverallPopupListRIt;

      /// Overall popup insertion order map
      PopupInsertionOrderMap mPopupInsertionOrderMap;

      /**
       * @brief Update the application popup list according to active applications.
       */
      void UpdateApplicationPopupList(UInt32 screenArea);

      /**
       * @brief Request a popup insertion order ID of the given popupState.
       * @note The insertion order IDs of popups must not be continuous.
       * @param popupState PopupState a the insertion order is requested.
       * @return Popup insertion order ID of the given popupState.
       */
      UInt32 GetPopupInsertionOrder(PopupState* popupState) const;

      /**
       * @brief Checks which one of the given two popupStates is the top one.
       * @param systemPopupState The system popupState to check.
       * @param appPopupState The application popupState to check.
       * @return Top-most popup state of the given ones.
       */
      PopupState* Top(PopupState* systemPopupState,
                      PopupState* appPopupState) const;

      ///
      void UpdateOverallPopupList(UInt32 screenArea);

      ///
      static bool ComparePopupList(PopupState* const& first,
                                   PopupState* const& second);
};


}

#endif
