/* ***************************************************************************************
* FILE:          ScreenLayouter.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScreenLayouter.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
//lint -emacro(641, ENUM_MAPPING)  "Converting enum to int"
// rapidxml interface lint issues are out of scope of sbplugins.
//lint -efunc(818, rapidxml::parse_error_handler) "Pointer parameter could be declared as
// pointing to const"
// Required for global registration for plugin
//lint -esym(1502, gScreenLayouterPlugin) "defined object has no nonstatic data members
// Exposing this private object is intended
//lint -esym(1536, ScreenBroker::ScreenLayouter::mScreenAreaList) "Exposing low access member"
// =============================================================================

#include "ScreenLayouter.h"
#include "RapidXml.h"
#include <ScreenBroker/Util/Debug.h>
#include <ScreenBroker/Service/ServiceApi.h>
#include <Shared/IlmAccessor.h>

#include <Shared/Animation/AnimationProperties.h>
#include <Base.h>
#include <string.h>
#include <iostream>
#include <sstream>
#include <unistd.h>

using namespace rapidxml;
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::ScreenLayouter);
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/ScreenLayouter.cpp.trc.h"
#endif


namespace ScreenBroker {

// ------------------------------------------------------------------------
// XML logging prefix
static const Char* const cXmlDataLogPrefix = "XML: ";

// XML node structure
static const Char* const cXmlScreenLayoutsNode = "ScreenLayouts";
static const Char* const cXmlScreenLayoutNode = "ScreenLayout";
static const Char* const cXmlDisplaysNode = "Displays";
static const Char* const cXmlDisplayNode = "Display";
static const Char* const cXmlAnimationConfigNode = "AnimationConfig";
static const Char* const cXmlScreenAreaNode = "ScreenArea";
static const Char* const cXmlScreenAreaRefNode = "ScreenAreaRef";
static const Char* const cXmlSeatsNode = "Seats";
static const Char* const cXmlSeatNode = "Seat";
static const Char* const cXmlSurfacesNode = "Surfaces";
static const Char* const cXmlSurfaceNode = "Surface";
static const Char* const cXmlRegistrationNode = "Registration";
static const Char* const cXmlEventsNode = "Events";
static const Char* const cXmlEventNode = "Event";
static const Char* const cXmlInputSourcesNode = "InputSources";
static const Char* const cXmlInputSourceNode = "InputSource";
static const Char* const cXmlAnimationNode = "Animation";
static const Char* const cXmlShowNode = "Show";
static const Char* const cXmlHideNode = "Hide";
static const Char* const cXmlCustomNode = "Custom";
static const Char* const cXmlTransitionAnimationNode = "TransitionAnimation";
static const Char* const cXmlCommitNode = "Commit";

// Text to enum maps
typedef std::map<std::string, UInt32> EnumMap;
#define ENUM_MAPPING(map, ns, attr) map[#attr] = ns::attr;

// ------------------------------------------------------------------------
EnumMap CreateEventsMap()
{
   EnumMap lMap;
   ENUM_MAPPING(lMap, Event, Key);
   return lMap;
}


EnumMap cEventsMap = CreateEventsMap();

// ------------------------------------------------------------------------
EnumMap CreateInputSourcesMap()
{
   EnumMap lMap;
   ENUM_MAPPING(lMap, InputSource, Keyboard);
   ENUM_MAPPING(lMap, InputSource, Pointer);
   ENUM_MAPPING(lMap, InputSource, Touch);
   return lMap;
}


EnumMap cInputSourcesMap = CreateInputSourcesMap();


void ScreenLayouter::UpdateDisplayIdentifier(hmibase::DisplayAliasEnum displayAlias, UInt32 displayId)
{
   DisplayIdentifierMap::iterator it = mDisplayIdentifierMap.find(displayAlias);

   if (it != mDisplayIdentifierMap.end())
   {
      it->second = displayId;
   }
   else
   {
      mDisplayIdentifierMap[displayAlias] = displayId;
   }
}


hmibase::DisplayAliasEnum ScreenLayouter::GetDisplayAlias(UInt32 displayId)
{
   for (DisplayIdentifierMap::iterator it = mDisplayIdentifierMap.begin(); it != mDisplayIdentifierMap.end(); ++it)
   {
      if (it->second == displayId)
      {
         return it->first;
      }
   }

   return hmibase::DISPLAY_UNKNOWN;
}


UInt32 ScreenLayouter::GetDisplayIdentifier(hmibase::DisplayAliasEnum displayAlias)
{
   DisplayIdentifierMap::iterator it = mDisplayIdentifierMap.find(displayAlias);
   return it != mDisplayIdentifierMap.end() ? it->second : (UInt32)hmibase::DISPLAY_UNKNOWN;
}


bool ScreenLayouter::GetDisplayIdentifier(UInt32 displayAlias, UInt32& displayId)
{
   //Dispaly alias will be always >= hmibase::DEFAULT_DISPLAY i.e:0
   if (displayAlias < hmibase::DISPLAY_UNKNOWN)
   {
      displayId = GetDisplayIdentifier(static_cast<hmibase::DisplayAliasEnum>(displayAlias));
      return (displayId == hmibase::DISPLAY_UNKNOWN) ? false : true;
   }
   else
   {
      return false;
   }
}


// ------------------------------------------------------------------------
EnumMap CreateAnimationDirectionMap()
{
   EnumMap lMap;
   ENUM_MAPPING(lMap, AnimationDirection, Show);
   ENUM_MAPPING(lMap, AnimationDirection, Hide);
   ENUM_MAPPING(lMap, AnimationDirection, Custom);
   return lMap;
}


EnumMap cAnimationDirectionMap = CreateAnimationDirectionMap();

// ------------------------------------------------------------------------
EnumMap CreatePivotPointMap()
{
   EnumMap lMap;
   ENUM_MAPPING(lMap, PivotPoint, Center);
   ENUM_MAPPING(lMap, PivotPoint, TopLeft);
   ENUM_MAPPING(lMap, PivotPoint, TopRight);
   ENUM_MAPPING(lMap, PivotPoint, BottomLeft);
   ENUM_MAPPING(lMap, PivotPoint, BottomRight);
   ENUM_MAPPING(lMap, PivotPoint, Top);
   ENUM_MAPPING(lMap, PivotPoint, Bottom);
   ENUM_MAPPING(lMap, PivotPoint, Left);
   ENUM_MAPPING(lMap, PivotPoint, Right);
   return lMap;
}


EnumMap cPivotPointMap = CreatePivotPointMap();

// ------------------------------------------------------------------------
EnumMap CreateAnimationTypeMap()
{
   EnumMap lMap;
   ENUM_MAPPING(lMap, AnimationType, FadeIn);
   ENUM_MAPPING(lMap, AnimationType, FadeOut);
   ENUM_MAPPING(lMap, AnimationType, ZoomInFromTopLeft);
   ENUM_MAPPING(lMap, AnimationType, ZoomOutToTopLeft);
   ENUM_MAPPING(lMap, AnimationType, SlideInFromLeft);
   ENUM_MAPPING(lMap, AnimationType, SlideOutToLeft);
   ENUM_MAPPING(lMap, AnimationType, SlideInFromTop);
   ENUM_MAPPING(lMap, AnimationType, SlideOutToTop);
   ENUM_MAPPING(lMap, AnimationType, SlideInFromRight);
   ENUM_MAPPING(lMap, AnimationType, SlideOutToRight);
   ENUM_MAPPING(lMap, AnimationType, SlideInFromBottom);
   ENUM_MAPPING(lMap, AnimationType, SlideOutToBottom);
   ENUM_MAPPING(lMap, AnimationType, ZoomInFromTopRight);
   ENUM_MAPPING(lMap, AnimationType, ZoomOutToTopRight);
   ENUM_MAPPING(lMap, AnimationType, ZoomInFromBottomLeft);
   ENUM_MAPPING(lMap, AnimationType, ZoomOutToBottomLeft);
   ENUM_MAPPING(lMap, AnimationType, ZoomInFromBottomRight);
   ENUM_MAPPING(lMap, AnimationType, ZoomOutToBottomRight);
   ENUM_MAPPING(lMap, AnimationType, ZoomInFromCenter);
   ENUM_MAPPING(lMap, AnimationType, ZoomOutToCenter);
   ENUM_MAPPING(lMap, AnimationType, ZoomInFadeIn);
   ENUM_MAPPING(lMap, AnimationType, ZoomInFadeOut);
   ENUM_MAPPING(lMap, AnimationType, ZoomOutFadeIn);
   ENUM_MAPPING(lMap, AnimationType, ZoomOutFadeOut);
   ENUM_MAPPING(lMap, AnimationType, ShowImmediately);
   ENUM_MAPPING(lMap, AnimationType, HideImmediately);

   return lMap;
}


EnumMap cAnimationTypeMap = CreateAnimationTypeMap();

// ------------------------------------------------------------------------
EnumMap CreateAccelerationTypeMap()
{
   EnumMap lMap;
   ENUM_MAPPING(lMap, AccelerationType, Linear);
   ENUM_MAPPING(lMap, AccelerationType, OutQuad);
   ENUM_MAPPING(lMap, AccelerationType, InQuad);
   ENUM_MAPPING(lMap, AccelerationType, OutBounce);

   return lMap;
}


EnumMap cAccelerationTypeMap = CreateAccelerationTypeMap();

// ------------------------------------------------------------------------
IScreenBrokerPlugin& ScreenLayouter::Create()
{
   static ScreenBroker::ScreenLayouter sInstance;
   return sInstance;
}


// ------------------------------------------------------------------------
ScreenLayouter::ScreenLayouter() : _animationMethod(-1), mInitialized(false)
{
}


// ------------------------------------------------------------------------
ScreenLayouter::~ScreenLayouter()
{
}


// ------------------------------------------------------------------------
bool ScreenLayouter::Init()
{
   // First perform plugin version check against screen broker
   if (!VersionCheck())
   {
      return false;
   }

   /* not required here: ETG_TRACE_USR1(("Initializing plugin ScreenLayouter version %40s", SCREENBROKERPLUGINS_VERSION_STRING)); */

   // Only allow one initialization call, to keep consistent screen layout data
   if (mInitialized)
   {
      ETG_TRACE_SYS(("Already initialized - Init call is ignored!"));
      return false;
   }
#if defined(SCREENLAYOUTER_READ_SURFACE_LAYER_MAPPING)
   ETG_TRACE_USR1(("Reading static surface to layer mapping (for on-demand surface registration) is enabled."));
#endif
   bool mRequestedScreenLayoutFound = false;
   std::string lScreenLayoutXml;
   IConfigurator* lConfigurator = ServiceApi::Configurator();
   if (lConfigurator && (!(lConfigurator->GetScreenLayoutsXmlPath()).empty()))
   {
      // Read screen configuration for location of screen layout XML file
      lScreenLayoutXml = lConfigurator->GetScreenLayoutsXmlPath();
   }
   else
   {
      // Read environment variable for location of screen layout XML file
      lScreenLayoutXml = GetEnvironmentVariable(ScreenLayoutsXmlEnvVar());
   }

   ETG_TRACE_USR1(("Reading screen layouts from '%40s'", lScreenLayoutXml.c_str()));

   // Load and parse XML document
   xml_document<> doc;
   try
   {
      file<> lXmlFile(lScreenLayoutXml.c_str());
      doc.parse<0>(lXmlFile.data());

      // If schema file exists, check XML file against it
      std::string lSysCall = std::string("[ -n \"${") + ScreenLayoutsXsdEnvVar() + "}\" ] 2>&1 >/dev/null";
      bool lXmlValidationDone = false;
      // Check if environent variable for XSD is set
      if ((0 == system(lSysCall.c_str())))
      {
         lSysCall = std::string("ls ${") + ScreenLayoutsXsdEnvVar() + "} 2>&1 >/dev/null";
         // Check if XSD is available
         if (0 == system(lSysCall.c_str()))
         {
            std::string lScreenLayoutXsd = GetEnvironmentVariable(ScreenLayoutsXsdEnvVar());
            ETG_TRACE_USR1(("Validate XML %40s using %40s", lScreenLayoutXml.c_str(), lScreenLayoutXsd.c_str()));
            lSysCall = std::string("xmllint --noout --schema ${") + ScreenLayoutsXsdEnvVar() + \
                       "} ${" + ScreenLayoutsXmlEnvVar() + "} 2>&1 >/dev/null";
            // Perform XML validation using xmllint with
            if (0 == system(lSysCall.c_str()))
            {
               ETG_TRACE_USR1(("XML validation successful"));
            }
            else
            {
               ETG_TRACE_ERR(("XML validation failed!"));
               return false;
            }
            lXmlValidationDone = true;
         }
      }
      if (!lXmlValidationDone)
      {
         ETG_TRACE_SYS(("XML validation not performed (XSD not found)!"));
      }

      // Find root node
      xml_node<>* lRootNode = doc.first_node(cXmlScreenLayoutsNode);
      if (0 == lRootNode)
      {
         ETG_TRACE_ERR(("XML - Root node '%40s' found!", cXmlScreenLayoutsNode));
         return false;
      }

      // Check ScreenLayouts protocol version
      std::string lVersion = lRootNode->first_attribute("version")->value();
      ETG_TRACE_USR1(("%40s%40s version %40s",
                      cXmlDataLogPrefix,
                      cXmlScreenLayoutsNode,
                      lVersion.c_str()));

      // *********************************************************************
      // Iterate screen layouts and process only requested screen layout
      // *********************************************************************
      xml_node<>* lScreenLayoutNode = lRootNode->first_node(cXmlScreenLayoutNode);
      if (0 == lScreenLayoutNode)
      {
         ETG_TRACE_ERR(("%40s%40s: No node %40s found!",
                        cXmlDataLogPrefix,
                        cXmlScreenLayoutsNode,
                        cXmlScreenLayoutNode));
         return false;
      }

      // Read environment variable for requested of screen layout ID
      std::string lRequestedScreenLayoutIdString = GetEnvironmentVariable(ScreenLayoutIdEnvVar());
      UInt32 lRequestedScreenLayoutId;
      if (lRequestedScreenLayoutIdString.empty())
      {
         lRequestedScreenLayoutId = 0;
      }
      else
      {
         std::istringstream idString(lRequestedScreenLayoutIdString);
         idString >> lRequestedScreenLayoutId;
      }
      ETG_TRACE_USR1(("Requested screen layout has ID %u", lRequestedScreenLayoutId));

      for (;
            (!mRequestedScreenLayoutFound) && (0 != lScreenLayoutNode);
            lScreenLayoutNode = lScreenLayoutNode->next_sibling(cXmlScreenLayoutNode))
      {
         // Read the a screen layout
         UInt32 lScreenLayoutId = ConvertTo<UInt32>(lScreenLayoutNode->first_attribute("id")->value());

         // Check if current read screen layout is the requested screen layout
         mRequestedScreenLayoutFound = (lRequestedScreenLayoutId == lScreenLayoutId);
         if (mRequestedScreenLayoutFound)
         {
            ETG_TRACE_USR1(("%40s%40s id=%u",
                            cXmlDataLogPrefix,
                            cXmlScreenLayoutNode,
                            lScreenLayoutId));

            // Read displays
            ReadDisplays(lScreenLayoutNode);
         } // RequestedScreenLayout found
      } // ScreenLayout loop

      if (!mRequestedScreenLayoutFound)
      {
         ETG_TRACE_ERR(("%40s%40s: No node %40s (id=%u) found!",
                        cXmlDataLogPrefix,
                        cXmlScreenLayoutsNode,
                        cXmlScreenLayoutNode,
                        lRequestedScreenLayoutId));
      }
   }
   catch (const std::runtime_error&)
   {
      ETG_TRACE_ERR(("Failed to read XML file '%40s'", lScreenLayoutXml.c_str()));
      return false;
   }
   mInitialized = mRequestedScreenLayoutFound;
   return mRequestedScreenLayoutFound;
}


// ------------------------------------------------------------------------
void ScreenLayouter::Reset()
{
   ETG_TRACE_USR1(("Reset"));
   mScreenAreaMap.clear();
   mScreenAreaList.clear();
   mInitialized = false;
}


// ------------------------------------------------------------------------
void ScreenLayouter::Diagnosis()
{
}


// ------------------------------------------------------------------------
void ScreenLayouter::Test()
{
}


// ------------------------------------------------------------------------
ScreenArea* ScreenLayouter::GetScreenArea(UInt32 screenAreaId)
{
   ScreenAreaMap::iterator it = mScreenAreaMap.find(screenAreaId);
   if (it == mScreenAreaMap.end())
   {
      return 0;
   }

   return (*it).second;
}


// ------------------------------------------------------------------------
IScreenLayouter::ScreenAreaList& ScreenLayouter::GetScreenAreaList()
{
   return mScreenAreaList;
}


// ------------------------------------------------------------------------
void ScreenLayouter::Action(UInt32 actionId, UInt32 actionData)
{
   ETG_TRACE_SYS(("No action %u with data %u available! Request ignored!",
                  actionId,
                  actionData));
}


// ------------------------------------------------------------------------
void ScreenLayouter::RequestCurrentStatus(UInt32 requestId)
{
   ETG_TRACE_SYS(("No status for request %u available! Request ignored!", requestId));
}


// ------------------------------------------------------------------------
UInt32 ScreenLayouter::GetEvents(UInt32 surfaceId)
{
   return (mSurfacePropertiesMap[surfaceId].events);
}


// ------------------------------------------------------------------------
UInt32 ScreenLayouter::GetInputSources(UInt32 surfaceId)
{
   return (mSurfacePropertiesMap[surfaceId].inputSources);
}


AnimationDirection::Enum ScreenLayouter::GetAnimationDirection(UInt32 surfaceId, AnimationType::Enum animationType)
{
   for (UInt32 index = (UInt32)AnimationDirection::Show; index < (UInt32)AnimationDirection::End; ++index)
   {
      AnimationDirection::Enum lAnimation = (AnimationDirection::Enum)index;
      ETG_TRACE_USR1(("GetAnimationProperties-Loop, index[%u], animationType[%u]", index, (UInt32)animationType));
      if (lAnimation != AnimationDirection::Custom)
      {
         AnimationPropertiesMap::iterator it = mSurfacePropertiesMap[surfaceId].animationProperties.find(lAnimation);
         if (it == mSurfacePropertiesMap[surfaceId].animationProperties.end())
         {
            ETG_TRACE_USR1(("GetAnimationDirection - The animation type - %d, is not configured for Animation Direction - %d", animationType, index));
         }
         else
         {
            if (animationType == it->second.type)
            {
               ETG_TRACE_USR1(("GetAnimationDirection-Show/Hide - %d", it->second.type));
               return lAnimation;
            }
         }
      }
      else
      {
         CustomAnimationPropertiesMap::iterator itCustom = mSurfacePropertiesMap[surfaceId].customAnimationProperties.find(lAnimation);
         if (itCustom == mSurfacePropertiesMap[surfaceId].customAnimationProperties.end())
         {
            return AnimationDirection::Undefined;
         }
         CustomAnimationList customAnimationList = ((*itCustom).second);
         for (CustomAnimationList::iterator itList = customAnimationList.begin(); itList != customAnimationList.end(); itList++)
         {
            if (itList->type == animationType)
            {
               ETG_TRACE_USR1(("GetAnimationDirection-Custom - %d", itList->type));
               return lAnimation;
            }
         }
      }
   }
   return AnimationDirection::Undefined;
}


// ------------------------------------------------------------------------
const AnimationProperties* ScreenLayouter::GetAnimationProperties(UInt32 surfaceId,
      AnimationDirection::Enum animationDirection, AnimationType::Enum animationType)
{
   ETG_TRACE_USR1(("GetAnimationProperties - %d, %d", (UInt32)animationDirection, (UInt32)animationType));
   if (AnimationType::None == animationType)
   {
      AnimationPropertiesMap::iterator it = mSurfacePropertiesMap[surfaceId].animationProperties.find(animationDirection);
      if ((it == mSurfacePropertiesMap[surfaceId].animationProperties.end()))
      {
         return 0;
      }
      else
      {
         ETG_TRACE_USR1(("GetAnimationProperties Inside Show/Hide - %d", (UInt32)((*it).second).type));
         return &((*it).second);
      }
   }
   else
   {
      CustomAnimationPropertiesMap::iterator itCustom = mSurfacePropertiesMap[surfaceId].customAnimationProperties.find(AnimationDirection::Custom);
      if (itCustom == mSurfacePropertiesMap[surfaceId].customAnimationProperties.end())
      {
         return 0;
      }
      CustomAnimationList customAnimationList = ((*itCustom).second);
      for (CustomAnimationList::iterator itList = customAnimationList.begin(); itList != customAnimationList.end(); itList++)
      {
         if (itList->type == animationType)
         {
            ETG_TRACE_USR1(("GetAnimationProperties - %d", itList->type));
            return &*(itList);
         }
      }
   }
   return 0;
}


// ------------------------------------------------------------------------
const AnimationProperties* ScreenLayouter::GetLayerAnimationProperties(UInt32 layerId,
      AnimationDirection::Enum animationDirection, AnimationType::Enum animationType)
{
   ETG_TRACE_USR1(("GetLayerAnimationProperties - %d, %d", (UInt32)animationDirection, (UInt32)animationType));
   switch (animationDirection)
   {
      case AnimationDirection::Show:
      case AnimationDirection::Hide:
      {
         AnimationPropertiesMap::iterator it = mLayerPropertiesMap[layerId].animationProperties.find(animationDirection);
         if (it == mLayerPropertiesMap[layerId].animationProperties.end())
         {
            return 0;
         }
         else
         {
            ETG_TRACE_USR1(("GetAnimationProperties Inside Show/Hide - %d", (UInt32)animationType));
            return &((*it).second);
         }
      }
      break;
      case AnimationDirection::Custom:
      {
         CustomAnimationPropertiesMap::iterator itCustom = mLayerPropertiesMap[layerId].customAnimationProperties.find(animationDirection);
         if (itCustom == mLayerPropertiesMap[layerId].customAnimationProperties.end())
         {
            return 0;
         }
         CustomAnimationList customAnimationList = ((*itCustom).second);
         for (CustomAnimationList::iterator itList = customAnimationList.begin(); itList != customAnimationList.end(); itList++)
         {
            if (itList->type == animationType)
            {
               ETG_TRACE_USR1(("GetLayerAnimationProperties - %d", itList->type));
               return &*(itList);
            }
         }
         return 0;
      }
      break;
      default:
         return 0;
   }
   return 0;
}


// ------------------------------------------------------------------------
bool ScreenLayouter::GetExplicitCommit(UInt32 surfaceId)
{
   return (mSurfacePropertiesMap[surfaceId].explicitCommit);
}


// ------------------------------------------------------------------------
UInt32 ScreenLayouter::GetDisplayIdOfSurface(UInt32 surfaceId)
{
   UInt32 lDisplayId = 0;
   SurfaceLayerMap::const_iterator itLayer = mSurfaceLayerMap.find(surfaceId);
   if (itLayer != mSurfaceLayerMap.end())
   {
      UInt32 lLayerId = itLayer->second;

      LayerDisplayMap::const_iterator itDisplay = mLayerDisplayMap.find(lLayerId);
      if (itDisplay != mLayerDisplayMap.end())
      {
         lDisplayId = itDisplay->second;
      }
   }
   return lDisplayId;
}


// ------------------------------------------------------------------------
UInt32 ScreenLayouter::GetScreenAreaIdOfSurface(UInt32 surfaceId)
{
   UInt32 lLayerId = 0;
   SurfaceLayerMap::const_iterator itLayer = mSurfaceLayerMap.find(surfaceId);
   if (itLayer != mSurfaceLayerMap.end())
   {
      lLayerId = itLayer->second;
   }

   return lLayerId;
}


// ------------------------------------------------------------------------
bool ScreenLayouter::IsSurfaceIsObserved(UInt32 surfaceId)
{
   bool lIsSurfaceObserved = false;
   SurfaceObserveAndAddMap::const_iterator itSurfaceObserved = mSurfaceObserveAndAddMap.find(surfaceId);
   if (itSurfaceObserved != mSurfaceObserveAndAddMap.end())
   {
      lIsSurfaceObserved = itSurfaceObserved->second;
   }
   return lIsSurfaceObserved;
}


// ------------------------------------------------------------------------
UInt32 ScreenLayouter::GetDisplayIdOfScreenArea(UInt32 screenArea)
{
   UInt32 lDisplayId = 0;

   LayerDisplayMap::const_iterator itDisplay = mLayerDisplayMap.find(screenArea);
   if (itDisplay != mLayerDisplayMap.end())
   {
      lDisplayId = itDisplay->second;
   }

   return lDisplayId;
}


// ------------------------------------------------------------------------
IScreenLayouter::SeatList* ScreenLayouter::GetSeatList(UInt32 surfaceId)
{
   SeatList* lRc = 0;
   if (mSeatMap.find(surfaceId) != mSeatMap.end())
   {
      lRc = &(mSeatMap.at(surfaceId));
   }

   return lRc;
}


// ------------------------------------------------------------------------
IScreenLayouter::DisplayIdList& ScreenLayouter::GetDisplayIdList()
{
   return mDisplayIdList;
}


// ------------------------------------------------------------------------
std::string ScreenLayouter::GetEnvironmentVariable(const std::string& key) const
{
   Char* value = std::getenv(key.c_str());
   return ((0 != value) ? value : "");
}


// ------------------------------------------------------------------------
template <typename T> T ScreenLayouter::ConvertTo(std::string from)
{
   std::stringstream lFrom(from);

   T lTo;

   if ((lFrom >> lTo).fail())
   {
      ETG_TRACE_ERR(("XML - Failed to convert %40s to native type", from.c_str()));
   }
   return lTo;
}


// ------------------------------------------------------------------------
bool ScreenLayouter::ConfigureDisplay(UInt32 displayId)
{
   bool lRc = false;

   // Set display render order
   ScreenAreaRefList* lScreenAreaRefList = &mLayerRenderOrder[displayId];
   if (lScreenAreaRefList->size() > 0)
   {
      UInt32 lLayerArray[lScreenAreaRefList->size()];
      UInt32 lLayerCount = 0;
      std::stringstream lSs;
      for (ScreenAreaRefList::iterator sIt = lScreenAreaRefList->begin();
            lScreenAreaRefList->end() != sIt;
            ++sIt)
      {
         UInt32 lLayerId = (*sIt)->GetLayerId();
         lSs << ((0 != lLayerCount) ? "," : "") << lLayerId;
         lLayerArray[lLayerCount++] = lLayerId;
      }
      ETG_TRACE_USR1(("Set layer render order for display %u to [%40s]",
                      displayId,
                      lSs.str().c_str()));

      //Commit changes for all layers also will be done
      ilmErrorTypes lIlmError = ILM_SUCCESS;
      lRc = IlmAccessor::SetLayerRenderOrder(displayId,
                                             lLayerArray,
                                             lLayerCount,
                                             true,
                                             lIlmError);
   }
   return lRc;
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadDisplays(const rapidxml::xml_node<char>* screenLayoutNode)
{
   hmibase::DisplayAliasEnum displayAlias = hmibase::DEFAULT_DISPLAY;

   if (0 != screenLayoutNode)
   {
      xml_node<>* lDisplaysNode = screenLayoutNode->first_node(cXmlDisplaysNode);
      if (0 != lDisplaysNode)
      {
         std::vector<UInt32> screenIds;
         IlmAccessor::GetScreenIds(screenIds);

         xml_node<>* lDisplayNode = lDisplaysNode->first_node(cXmlDisplayNode);
         bool displayFound = false;
         bool acceptFirstDisplayInConfig = false;

         while (!displayFound)
         {
            for (; 0 != lDisplayNode;
                  lDisplayNode = lDisplayNode->next_sibling(cXmlDisplayNode))
            {
               UInt32 lDisplayId = 0;

               if (!screenIds.empty())
               {
                  displayFound = false;
                  if (screenIds.size() > 1)
                  {
                     // there is more than one display, so we need to figure out which one to use for which DisplayNode in Layuout.xml
                     IConfigurator* lConfigurator = ServiceApi::Configurator();

                     if (lConfigurator)
                     {
                        // get info from project specific plugin
                        if (lConfigurator->GetDisplayIdFromAlias(displayAlias, lDisplayId))
                        {
                           displayFound = true;
                        }
                        else
                        {
                           displayFound = false;
                        }
                     }

                     if (!displayFound)
                     {
                        // get info from Layout xml
                        lDisplayId = ConvertTo<UInt32>(lDisplayNode->first_attribute("id")->value());
                        displayFound = true;
                     }
                  }
                  else
                  {
                     // only one display available, so figure out main display in ScreenLayouts
                     if (acceptFirstDisplayInConfig || (0 == strcmp(lDisplayNode->first_attribute("name")->value(), ":0")))
                     {
                        // this is my main display
                        lDisplayId = screenIds[0];
                        displayFound = true;
                     }
                     else
                     {
                        continue;
                     }
                  }
               }
               else
               {
                  ETG_TRACE_FATAL(("no screens connected"));
                  return;
               }

               //Update Display Index with Real Display
               UpdateDisplayIdentifier(displayAlias, lDisplayId);
               displayAlias = static_cast<hmibase::DisplayAliasEnum>(static_cast<UInt32>(displayAlias) + 1);
               ETG_TRACE_USR1(("%40s  %40s id=%u",
                               cXmlDataLogPrefix,
                               cXmlDisplayNode,
                               lDisplayId));

               // Read display childs (ScreenArea and ScreenAreaRef)
               xml_node<>* lDisplayChildNode = lDisplayNode->first_node();
               for (; 0 != lDisplayChildNode;
                     lDisplayChildNode = lDisplayChildNode->next_sibling())
               {
                  // Distinguish between ScreenArea and ScreenAreaRef tag
                  if (0 == strcmp(lDisplayChildNode->name(), cXmlScreenAreaNode))
                  {
                     ReadScreenArea(lDisplayChildNode, lDisplayId);
                  }
                  else if (0 == strcmp(lDisplayChildNode->name(), cXmlScreenAreaRefNode))
                  {
                     ReadScreenAreaRef(lDisplayChildNode, lDisplayId);
                  }
               } // DisplayChild loop

               // Add to displayId list
               mDisplayIdList.push_back(lDisplayId);

               // add build-in layer at the end of the list
               CreateBuildInLayer(lDisplayId);

               // Propagate layer render order for given display
               ConfigureDisplay(lDisplayId);
            } // Display loop

            if (acceptFirstDisplayInConfig && !displayFound)
            {
               // this should not happen
               ETG_TRACE_FATAL(("Neither matching display nor fallback display found"));
               // ToDo: maybe we should abort here
               break;
            }
            if (displayFound == false)
            {
               // no matching display found in first pass, so fallback to first display in config and take this one
               lDisplayNode = lDisplaysNode->first_node(cXmlDisplayNode);
               acceptFirstDisplayInConfig = true;
            }
         }

         readAnimationConfig(lDisplaysNode);
         // Read surfaces
         ReadSurfaces(lDisplaysNode);
      } // Displays found
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadScreenArea(const rapidxml::xml_node<char>* screenAreaNode, UInt32 displayId)
{
   if (0 != screenAreaNode)
   {
      UInt32 lScreenAreaId = ConvertTo<UInt32>(screenAreaNode->first_attribute("id")->value());

      // Per default set popup flag to false, if attribute is missing
      bool lIsPopup = (0 != screenAreaNode->first_attribute("popup")) ?
                      (std::string(screenAreaNode->first_attribute("popup")->value()) == std::string("yes")) :
                      false;

      // Per default set managed flag to true, if attribute is missing
      bool lIsManaged = (0 != screenAreaNode->first_attribute("managed")) ?
                        (std::string(screenAreaNode->first_attribute("managed")->value()) == std::string("yes")) :
                        true;

      // Per default set visible flag to true, if attribute is missing
      bool lIsVisible = (0 != screenAreaNode->first_attribute("visible")) ?
                        (std::string(screenAreaNode->first_attribute("visible")->value()) == std::string("yes")) :
                        true;

      // Per default notifyOnPopupStatus flag to false, if attribute is missing
      bool lIsNotifyOnPopupStatus = (0 != screenAreaNode->first_attribute("notifyOnPopupStatus")) ?
                                    (std::string(screenAreaNode->first_attribute("notifyOnPopupStatus")->value()) == std::string("yes")) :
                                    false;
      ReadAnimation(screenAreaNode, lScreenAreaId, true);
      ScreenArea lScreenArea;
      if (lScreenArea.Init(lScreenAreaId,
                           ConvertTo<UInt32>(screenAreaNode->first_attribute("width")->value()),
                           ConvertTo<UInt32>(screenAreaNode->first_attribute("height")->value()),
                           ConvertTo<UInt32>(screenAreaNode->first_attribute("x")->value()),
                           ConvertTo<UInt32>(screenAreaNode->first_attribute("y")->value()),
                           lIsPopup,
                           displayId,
                           GetDisplayAlias(displayId),
                           lIsManaged,
                           lIsVisible,
                           lIsNotifyOnPopupStatus))
      {
         {
            char str[230];
            sprintf(str, "%s    %s id=%u (dimension=%ux%u, position=%u/%u, isPopup: %s, isManaged: %s isVisible: %s IsNotifyOnPopupStatus: %s)",
                    cXmlDataLogPrefix,
                    screenAreaNode->name(),
                    lScreenAreaId,
                    lScreenArea.GetWidth(),
                    lScreenArea.GetHeight(),
                    lScreenArea.GetXPos(),
                    lScreenArea.GetYPos(),
                    lScreenArea.IsPopupArea() ? "true" : "false",
                    lScreenArea.IsManaged() ? "true" : "false",
                    lScreenArea.IsVisible() ? "true" : "false",
                    lScreenArea.IsNotifyOnPopupStatus() ? "true" : "false");
            ETG_TRACE_USR4(("%s", str));
         }
// BOSCH_CHANGE_START
#ifdef GEN3X86
         mLayerDisplayMap[lScreenAreaId] = displayId;
         // Add (only) new ScreenArea to list
         if (0 == GetScreenArea(lScreenAreaId))
         {
            mScreenAreaList.push_back(lScreenArea);
            ScreenArea* lScreenAreaListElement = &mScreenAreaList.back();
            mScreenAreaMap[lScreenAreaId] = lScreenAreaListElement;

            // Request layer for screen area from ILM
            if (lScreenAreaListElement->InitializeLayering())
            {
               // Fill up render order per display
               mLayerRenderOrder[displayId].push_back(lScreenAreaListElement);
            }
         }
         else
         {
            //Update secondary display id to screen area
            ScreenArea* screenAreaPtr = mScreenAreaMap[lScreenAreaId];
            screenAreaPtr->ReInit(lScreenArea);
         }
#else
         if (0 == GetScreenArea(lScreenAreaId))
         {
            mScreenAreaList.push_back(lScreenArea);
            ScreenArea* lScreenAreaListElement = &mScreenAreaList.back();
            mScreenAreaMap[lScreenAreaId] = lScreenAreaListElement;
            mLayerDisplayMap[lScreenAreaId] = displayId;
            if (lScreenAreaListElement->InitializeLayering())
            {
               mLayerRenderOrder[displayId].push_back(lScreenAreaListElement);
            }
         }
         else
         {
            ETG_TRACE_USR1(("Screen area ID:%u already imported! Ignoring multiple screen areas!", lScreenAreaId));
         }
#endif
// BOSCH_CHANGE_END
      }
      else
      {
         ETG_TRACE_ERR(("Initialization of screen area ID:%u failed!",
                        lScreenAreaId));
      }
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadScreenAreaRef(const rapidxml::xml_node<char>* screenAreaRefNode, UInt32 displayId)
{
   if (0 != screenAreaRefNode)
   {
      UInt32 lScreenAreaId = ConvertTo<UInt32>(screenAreaRefNode->first_attribute("id")->value());
      ScreenArea* lScreenArea = GetScreenArea(lScreenAreaId);

      if (0 != lScreenArea)
      {
         {
            char str[230];
            sprintf(str, "%s    %s id=%u (dimension=%ux%u, position=%u/%u, isPopup: %s, isManaged: %s)",
                    cXmlDataLogPrefix,
                    screenAreaRefNode->name(),
                    lScreenAreaId,
                    lScreenArea->GetWidth(),
                    lScreenArea->GetHeight(),
                    lScreenArea->GetXPos(),
                    lScreenArea->GetYPos(),
                    lScreenArea->IsPopupArea() ? "true" : "false",
                    lScreenArea->IsManaged() ? "true" : "false");
            ETG_TRACE_USR4(("%40s", str));
         }
         // Fill up render order per display
         mLayerRenderOrder[displayId].push_back(lScreenArea);
      }
      else
      {
         ETG_TRACE_SYS(("%40s    %40s id=%u: No screen area to reference to found in previous defined displays. Ignoring...",
                        cXmlDataLogPrefix,
                        screenAreaRefNode->name(),
                        lScreenAreaId));
      }
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadSurfaces(const rapidxml::xml_node<char>* displayNode)
{
   if (0 != displayNode)
   {
      xml_node<>* lSurfacesNode = displayNode->next_sibling(cXmlSurfacesNode);
      if (0 != lSurfacesNode)
      {
         xml_node<>* lSurfaceNode = lSurfacesNode->first_node(cXmlSurfaceNode);
         for (; 0 != lSurfaceNode;
               lSurfaceNode = lSurfaceNode->next_sibling(cXmlSurfaceNode))
         {
            UInt32 lSurfaceId = ConvertTo<UInt32>(lSurfaceNode->first_attribute("id")->value());
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
            // Per default observeAndAdd flag to false, if attribute is missing
            bool lIsobserveAndAdd = (0 != lSurfaceNode->first_attribute("observeAndAdd")) ?
                                    (std::string(lSurfaceNode->first_attribute("observeAndAdd")->value()) == std::string("yes")) :
                                    false;
            mSurfaceObserveAndAddMap[lSurfaceId] = lIsobserveAndAdd;
#endif
            ETG_TRACE_USR1(("%40s  %40s id=%u",
                            cXmlDataLogPrefix,
                            cXmlSurfaceNode,
                            lSurfaceId));

            // Read static surface registration data
            ReadRegistrationData(lSurfaceNode, lSurfaceId);

            // Read events
            ReadEvents(lSurfaceNode, lSurfaceId);

            // Read input sources
            ReadInputSources(lSurfaceNode, lSurfaceId);

            // Read seats
            ReadSeats(lSurfaceNode, lSurfaceId);

            // Read animation
            ReadAnimation(lSurfaceNode, lSurfaceId);

            // Read commit
            ReadCommit(lSurfaceNode, lSurfaceId);
         } // Surface loop
      } // Surfaces found
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadRegistrationData(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId)
{
   if (0 != surfaceNode)
   {
      xml_node<>* lRegistrationNode = surfaceNode->first_node(cXmlRegistrationNode);
      if (0 != lRegistrationNode)
      {
         UInt32 lScreenAreaId = ConvertTo<UInt32>(lRegistrationNode->first_attribute("screenArea")->value());
         UInt32 lUserData = ConvertTo<UInt32>(lRegistrationNode->first_attribute("userData")->value());
         ETG_TRACE_USR1(("%40s    %40s screenArea=%u userData=%u",
                         cXmlDataLogPrefix,
                         cXmlRegistrationNode,
                         lScreenAreaId,
                         lUserData));

         mSurfaceLayerMap[surfaceId] = lScreenAreaId;
      }
      else
      {
         ETG_TRACE_USR1(("lRegistrationNode not created"));
      }

#if defined(SCREENLAYOUTER_READ_SURFACE_LAYER_MAPPING)
      // Initialize surface to layer mapping
      ServiceApi::RegisterSurfaceToLayer(surfaceId, lScreenAreaId, lUserData);
#else
      (void)surfaceId;
#endif
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadEvents(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId)
{
   if (0 != surfaceNode)
   {
      xml_node<>* lEventsNode = surfaceNode->first_node(cXmlEventsNode);
      if (0 != lEventsNode)
      {
         xml_node<>* lEventNode = lEventsNode->first_node(cXmlEventNode);
         for (; 0 != lEventNode;
               lEventNode = lEventNode->next_sibling(cXmlEventNode))
         {
            std::string lEventType = lEventNode->first_attribute("type")->value();
            {
               char str[230];
               sprintf(str, "%s    %s type=%s",
                       cXmlDataLogPrefix,
                       cXmlEventNode,
                       lEventType.c_str());
               ETG_TRACE_USR4(("%40s", str));
            }

            mSurfacePropertiesMap[surfaceId].events |= cEventsMap[lEventType];
         } // Event loop
      } // Events found
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadInputSources(const xml_node<>* surfaceNode, UInt32 surfaceId)
{
   if (0 != surfaceNode)
   {
      xml_node<>* lInputSourcesNode = surfaceNode->first_node(cXmlInputSourcesNode);
      if (0 != lInputSourcesNode)
      {
         xml_node<>* lInputSourceNode = lInputSourcesNode->first_node(cXmlInputSourceNode);
         for (; 0 != lInputSourceNode;
               lInputSourceNode = lInputSourceNode->next_sibling(cXmlInputSourceNode))
         {
            std::string lInputSource = lInputSourceNode->first_attribute("type")->value();
            {
               char str[230];
               sprintf(str, "%s    %s type=%s",
                       cXmlDataLogPrefix,
                       cXmlInputSourceNode,
                       lInputSource.c_str());
               ETG_TRACE_USR4(("%40s", str));
            }

            mSurfacePropertiesMap[surfaceId].inputSources |= cInputSourcesMap[lInputSource];
         } // InputSource loop

#ifdef VARIANT_S_FTR_ENABLE_TREAT_TOUCH_AS_POINTER
         if (mSurfacePropertiesMap[surfaceId].inputSources & InputSource::Touch)
         {
            mSurfacePropertiesMap[surfaceId].inputSources |= InputSource::Pointer;
            ETG_TRACE_USR4(("ScreenLayouter: treat touch input source for surface %d as pointer", surfaceId));
         }
#endif
      } // InputSources found
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadSeats(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId)
{
   if (0 != surfaceNode)
   {
      xml_node<>* lSeatsNode = surfaceNode->first_node(cXmlSeatsNode);

      if (0 != lSeatsNode)
      {
         xml_node<>* lSeatNode = lSeatsNode->first_node(cXmlSeatNode);
         for (; 0 != lSeatNode;
               lSeatNode = lSeatNode->next_sibling(cXmlSeatNode))
         {
            mSeatMap[surfaceId].push_back(lSeatNode->first_attribute("name")->value());
         }
      }
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadAnimation(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId, bool isLayerContent)
{
   if (0 != surfaceNode)
   {
      xml_node<>* lAnimationNode = surfaceNode->first_node(cXmlAnimationNode);
      if (0 != lAnimationNode)
      {
         ETG_TRACE_USR1(("%40s    %40s",
                         cXmlDataLogPrefix,
                         cXmlAnimationNode));
         ReadAnimationProperties(lAnimationNode, cXmlShowNode, surfaceId, isLayerContent);
         ReadAnimationProperties(lAnimationNode, cXmlHideNode, surfaceId, isLayerContent);

         xml_node<>* lTransitionAnimationList = lAnimationNode->first_node(cXmlTransitionAnimationNode);
         if (lTransitionAnimationList != 0)
         {
            ReadCustomAnimationProperties(lTransitionAnimationList, cXmlCustomNode, surfaceId, isLayerContent);
         }
      } // Animation found
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadCustomAnimationProperties(const rapidxml::xml_node<char>* animationNode,
      const char* xmlNode,
      UInt32 surfaceId, bool isLayerContent)
{
   std::string lXmlNode(xmlNode);
   std::string lStrScale;
   CustomAnimationList lCustomAnimationList;
   AnimationDirection::Enum lAnimationDirection = AnimationDirection::Undefined;
   xml_node<>* lNode = animationNode->first_node(lXmlNode.c_str());
   while (0 != lNode)
   {
      AnimationProperties lAnimationProperties;
      lAnimationDirection = static_cast<AnimationDirection::Enum>(cAnimationDirectionMap[lXmlNode]);
      std::string lType = lNode->first_attribute("type")->value();
      lAnimationProperties.type = static_cast<AnimationType::Enum>(cAnimationTypeMap[lType]);
      xml_attribute<>* lDurationAttr = lNode->first_attribute("duration");
      lAnimationProperties.duration = (0 != lDurationAttr) ?
                                      ConvertTo<UInt32>(lDurationAttr->value()) :
                                      AnimationProperties::cDefaultDuration;
      xml_attribute<>* lStepsAttr = lNode->first_attribute("steps");
      lAnimationProperties.steps = (0 != lStepsAttr) ?
                                   ConvertTo<UInt32>(lStepsAttr->value()) :
                                   AnimationProperties::cDefaultUpdateCount;
      xml_attribute<>* lAccelerationAttr = lNode->first_attribute("acceleration");
      lAnimationProperties.acceleration = (0 != lAccelerationAttr) ?
                                          static_cast<AccelerationType::Enum>(cAccelerationTypeMap[lAccelerationAttr->value()]) :
                                          AnimationProperties::cDefaultAccelerationType;
      xml_attribute<>* lAmplitudeAttr = lNode->first_attribute("amplitude");
      lAnimationProperties.amplitude = (0 != lAmplitudeAttr) ?
                                       ConvertTo<Float>(lAmplitudeAttr->value()) :
                                       AnimationProperties::cDefaultAmplitude;

      xml_attribute<>* lScaleFromAttr = lNode->first_attribute("scaleFrom");
      lAnimationProperties.scaleFrom = (0 != lScaleFromAttr) ?
                                       ConvertTo<UInt32>(lScaleFromAttr->value()) :
                                       AnimationProperties::cScaleFrom;
      xml_attribute<>* lScaleToAttr = lNode->first_attribute("scaleTo");
      lAnimationProperties.scaleTo = (0 != lScaleToAttr) ?
                                     ConvertTo<UInt32>(lScaleToAttr->value()) :
                                     AnimationProperties::cScaleTo;

      xml_attribute<>* lPivotPointAttr = lNode->first_attribute("pivot");
      lAnimationProperties.pivotPoint = (0 != lPivotPointAttr) ? static_cast<PivotPoint::Enum>(cPivotPointMap[lPivotPointAttr->value()]) : PivotPoint::Undefined;
      {
         char str[230];
         sprintf(str, "%s      %s type=%s duration=%u steps=%u acceleration=%s ampliutde=%0.2f, scale(%u, %u), pivot=%s",
                 cXmlDataLogPrefix,
                 lXmlNode.c_str(),
                 lType.c_str(),
                 lAnimationProperties.duration,
                 lAnimationProperties.steps,
                 (0 != lAccelerationAttr) ? lAccelerationAttr->name() : "",
                 lAnimationProperties.amplitude,
                 lAnimationProperties.scaleFrom,
                 lAnimationProperties.scaleTo,
                 (lPivotPointAttr != 0) ? lPivotPointAttr->value() : "NULL");
         ETG_TRACE_USR4(("%s", str));
      }

      //mSurfacePropertiesMap[surfaceId].customAnimationProperties[lAnimationDirection].push_back(lAnimationProperties);
      lCustomAnimationList.push_back(lAnimationProperties);
      lNode = lNode->next_sibling(lXmlNode.c_str());
      ETG_TRACE_USR4(("%50s, %d", lXmlNode.c_str(), (lNode != 0)));
   } // lNode found
   if (isLayerContent)
   {
      mLayerPropertiesMap[surfaceId].customAnimationProperties[lAnimationDirection] = lCustomAnimationList;
      ETG_TRACE_USR4(("Custom Animation properties for layer %d", surfaceId));
   }
   else
   {
      mSurfacePropertiesMap[surfaceId].customAnimationProperties[lAnimationDirection] = lCustomAnimationList;
   }
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadAnimationProperties(const rapidxml::xml_node<char>* animationNode,
      const char* xmlNode,
      UInt32 surfaceId, bool isLayerContent)
{
   std::string lXmlNode(xmlNode);
   std::string lStrScale;
   xml_node<>* lNode = animationNode->first_node(lXmlNode.c_str());
   if (0 != lNode)
   {
      AnimationProperties lAnimationProperties;
      AnimationDirection::Enum lAnimationDirection =
         static_cast<AnimationDirection::Enum>(cAnimationDirectionMap[lXmlNode]);
      std::string lType = lNode->first_attribute("type")->value();
      lAnimationProperties.type = static_cast<AnimationType::Enum>(cAnimationTypeMap[lType]);
      xml_attribute<>* lDurationAttr = lNode->first_attribute("duration");
      lAnimationProperties.duration = (0 != lDurationAttr) ?
                                      ConvertTo<UInt32>(lDurationAttr->value()) :
                                      AnimationProperties::cDefaultDuration;
      xml_attribute<>* lStepsAttr = lNode->first_attribute("steps");
      lAnimationProperties.steps = (0 != lStepsAttr) ?
                                   ConvertTo<UInt32>(lStepsAttr->value()) :
                                   AnimationProperties::cDefaultUpdateCount;
      xml_attribute<>* lAccelerationAttr = lNode->first_attribute("acceleration");
      lAnimationProperties.acceleration = (0 != lAccelerationAttr) ?
                                          static_cast<AccelerationType::Enum>(cAccelerationTypeMap[lAccelerationAttr->value()]) :
                                          AnimationProperties::cDefaultAccelerationType;
      xml_attribute<>* lAmplitudeAttr = lNode->first_attribute("amplitude");
      lAnimationProperties.amplitude = (0 != lAmplitudeAttr) ?
                                       ConvertTo<Float>(lAmplitudeAttr->value()) :
                                       AnimationProperties::cDefaultAmplitude;
      xml_attribute<>* lScaleFromAttr = lNode->first_attribute("scaleFrom");
      lAnimationProperties.scaleFrom = (0 != lScaleFromAttr) ?
                                       ConvertTo<UInt32>(lScaleFromAttr->value()) :
                                       AnimationProperties::cScaleFrom;
      xml_attribute<>* lScaleToAttr = lNode->first_attribute("scaleTo");
      lAnimationProperties.scaleTo = (0 != lScaleToAttr) ?
                                     ConvertTo<UInt32>(lScaleToAttr->value()) :
                                     AnimationProperties::cScaleTo;
      xml_attribute<>* lPivotPointAttr = lNode->first_attribute("pivot");
      lAnimationProperties.pivotPoint = (0 != lPivotPointAttr) ? static_cast<PivotPoint::Enum>(cPivotPointMap[lPivotPointAttr->value()]) : PivotPoint::Undefined;
      xml_attribute<>* lDelayDurationAttr = lNode->first_attribute("delay");
      lAnimationProperties.delayduration = (0 != lDelayDurationAttr) ?
                                           ConvertTo<UInt32>(lDelayDurationAttr->value()) :
                                           AnimationProperties::cDefaultDelayDuration;
      {
         char str[230];
         sprintf(str, "%s      %s type=%s duration=%u delay=%u steps=%u acceleration=%s amplitude=%f, scale(%u, %u), pivot - %s",
                 cXmlDataLogPrefix,
                 lXmlNode.c_str(),
                 lType.c_str(),
                 lAnimationProperties.duration,
                 lAnimationProperties.delayduration,
                 lAnimationProperties.steps,
                 (0 != lAccelerationAttr) ? lAccelerationAttr->name() : "",
                 lAnimationProperties.amplitude,
                 lAnimationProperties.scaleFrom,
                 lAnimationProperties.scaleTo,
                 (lPivotPointAttr != 0) ? lPivotPointAttr->value() : "NULL");
         ETG_TRACE_USR4(("%s", str));
      }
      if (isLayerContent)
      {
         mLayerPropertiesMap[surfaceId].animationProperties[lAnimationDirection] = lAnimationProperties;
         ETG_TRACE_USR4(("Animation properties for layer %d", surfaceId));
      }
      else
      {
         mSurfacePropertiesMap[surfaceId].animationProperties[lAnimationDirection] = lAnimationProperties;
      }
   } // lNode found
}


// ------------------------------------------------------------------------
void ScreenLayouter::ReadCommit(const rapidxml::xml_node<char>* surfaceNode, UInt32 surfaceId)
{
   if (0 != surfaceNode)
   {
      xml_node<>* lCommitNode = surfaceNode->first_node(cXmlCommitNode);
      bool lExplicit = false;

      if (0 != lCommitNode)
      {
         lExplicit = (0 == strcmp(lCommitNode->first_attribute("explicit")->value(), "true"));

         ETG_TRACE_USR1(("%40s      %40s explicit=%u ",
                         cXmlDataLogPrefix,
                         lCommitNode->name(),
                         lExplicit));
      }

      mSurfacePropertiesMap[surfaceId].explicitCommit = lExplicit;
   }
}


void ScreenLayouter::CreateBuildInLayer(UInt32 displayId)
{
   // add one layer as build-in layer to each display, this can be used testing purpose like showng test images via testimagedaemon
   t_ilm_uint width = 0;
   t_ilm_uint height = 0;
   ilmErrorTypes errorCode = ILM_SUCCESS;
   if (IlmAccessor::GetScreenDimensions(displayId, width, height, errorCode) == true)
   {
      static UInt32 layerId = hmibase::LAYERID_BUILDINLAYERBASE;
      static UInt32 surfaceId = hmibase::SURFACEID_BUILDINSURFACEBASE;
      ++layerId;
      ++surfaceId;

      if (layerId < hmibase::LAYERID_BUILDINLAYERBASE_MAX)
      {
         ScreenArea screenArea;
         screenArea.Init(layerId, width, height, 0, 0, false, displayId, GetDisplayAlias(displayId), true, true, false);

         mScreenAreaList.push_back(screenArea);
         ScreenArea* lScreenAreaListElement = &mScreenAreaList.back();
         mScreenAreaMap[layerId] = lScreenAreaListElement;
         mLayerDisplayMap[layerId] = displayId;
         if (lScreenAreaListElement->InitializeLayering())
         {
            mLayerRenderOrder[displayId].push_back(lScreenAreaListElement);
         }

         mSurfaceLayerMap[surfaceId] = layerId;
         ServiceApi::AddBuildInLayerInformation(layerId, surfaceId, displayId, width, height);
      }
      else
      {
         ETG_TRACE_ERR(("LayerId for build in layer exceed the maximum"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Error reading screen resolution for screen %d from ILM, error %d", displayId, errorCode));
   }
}


void ScreenLayouter::readAnimationConfig(const rapidxml::xml_node<char>* DisplayConfigNode)
{
   if (0 != DisplayConfigNode)
   {
      xml_node<>* lAnimationConfigNode = DisplayConfigNode->next_sibling(cXmlAnimationConfigNode);
      if (0 != lAnimationConfigNode)
      {
         _animationMethod = ConvertTo<Int16>(lAnimationConfigNode->first_attribute("type")->value());
         ETG_TRACE_USR1(("_animationMethod = %d", _animationMethod));
      }
   }
}


bool ScreenLayouter::getAnimationMethod(UInt& method)
{
   if (0 <= _animationMethod)
   {
      method = static_cast<UInt>(_animationMethod);
   }
   return (_animationMethod != -1) ? true : false;
}


}


// ========================================================================

namespace rapidxml {
///
void parse_error_handler(const char* what, void* where)
{
   ETG_TRACE_ERR(("XML parse error: %40s", what));
   if (0 != where)
   {
      ETG_TRACE_ERR(("    at: '%40s'", reinterpret_cast<const char*>(where)));
      std::abort();
   }
}


}

// ========================================================================
// Global registration for plugin
// ========================================================================
extern "C" {
   // ------------------------------------------------------------------------
   class ScreenLayouterPluginProxy
   {
      public:
         ScreenLayouterPluginProxy()
         {
            // Register the plugin instance creation at the global factory map
            gScreenBrokerPluginFactoryMap[ScreenBroker::IScreenLayouter::Name()] = ScreenBroker::ScreenLayouter::Create;
         }
   };

   // Create one instance of the proxy to perform registration
   ScreenLayouterPluginProxy gScreenLayouterPlugin;
}
