/* ***************************************************************************************
* FILE:          ScreenBrokerActivator.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScreenBrokerActivator.cpp is part of HMI-Base ScreenBrokerPlugins
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

// =============================================================================
// Required for global registration for plugin
//lint -esym(1502, gScreenBrokerActivatorPlugin) "defined object has no nonstatic data members
//lint -esym(1502, sInstance) "defined object has no nonstatic data members
// =============================================================================

#include "ScreenBrokerActivator.h"
#include <Base.h>
#include <ScreenBroker/Service/ServiceApi.h>
#include <ScreenBroker/SurfaceState.h>
#include <Shared/ActivePopups.h>
#include <Shared/ActiveSurfaces.h>
#include <Shared/AnimationHandler.h>
#include <Shared/FocusHandler.h>
#include <Shared/IlmAccessor.h>
#include <ScreenLayouter/ScreenLayouter.h>

//added for ulseep
#include <unistd.h>

#include <Shared/PluginActions.h>
#include <bitset>
#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_PLUGINS
#include "trcGenProj/Header/ScreenBrokerActivator.cpp.trc.h"
#endif


namespace ScreenBroker {
// SCREENBROKER_LOG_SET_REALM(LogRealm::ScreenBrokerActivator);

// ------------------------------------------------------------------------
// Instantiate (plugin) global handled ActiveSurfaces object
ActiveSurfaces gActiveSurfacesInstance;

// ------------------------------------------------------------------------
// Instantiate (plugin) global focus handling
FocusHandler gFocusHandlerInstance;

// ------------------------------------------------------------------------
IScreenBrokerPlugin& ScreenBrokerActivator::Create()
{
   static ScreenBroker::ScreenBrokerActivator sInstance;
   return sInstance;
}


// ------------------------------------------------------------------------
ScreenBrokerActivator::ScreenBrokerActivator()
{
   // Note: Shared screen broker plugin realms have only to be initialized in one plugin
}


// ------------------------------------------------------------------------
ScreenBrokerActivator::~ScreenBrokerActivator()
{
   // Note: Shared screen broker plugin realms have only to be destroyed in one plugin
}


// ------------------------------------------------------------------------
bool ScreenBrokerActivator::Init()
{
   bool lRc = VersionCheck();
   if (lRc)
   {
      /* not required here: ETG_TRACE_USR1(("Initializing plugin ScreenBrokerActivator version %40s", SCREENBROKERPLUGINS_VERSION_STRING)); */
   }
   return lRc;
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::Reset()
{
   ETG_TRACE_USR1(("*** Reset ***"));
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   // Remove active popups
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
   IScreenLayouter* lScreenLayouter = ServiceApi::ScreenLayouter();
   if (0 != lScreenLayouter)
   {
      IScreenLayouter::DisplayIdList lDisplayIdList = lScreenLayouter->GetDisplayIdList();
      for (IScreenLayouter::DisplayIdList::iterator lIt = lDisplayIdList.begin(); lIt != lDisplayIdList.end(); ++lIt)
      {
         UInt32 lSurfaceId = lActiveSurfaces.GetTop(*lIt, false);
         while (0 != lSurfaceId)
         {
            ETG_TRACE_USR1(("Resetting surface %u", lSurfaceId));
            (void) FocusHandler::EnableInputEvents(lSurfaceId, false);
            (void) IlmAccessor::SetSurfaceVisibility(lSurfaceId, ILM_FALSE, false, lIlmError);
            lActiveSurfaces.Remove(lSurfaceId);
            lSurfaceId = lActiveSurfaces.GetTop(*lIt, false);
         }

         // Update input focus to active surface ID
         (void) FocusHandler::GetInstance().UpdateInputFocus(ActiveSurfaces::GetInstance().GetTop(*lIt));
      }
   }
   else
   {
      ETG_TRACE_SYS(("@ Resetting surfaces failed: ScreenLayouter instance not accessible!"));
   }
   (void) IlmAccessor::Commit(lIlmError);
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::Diagnosis()
{
   ActiveSurfaces::GetInstance().Dump();
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::Test()
{
   IlmAccessor::TestInputAcceptance(1);
}


// ------------------------------------------------------------------------
bool ScreenBrokerActivator::ActivationPostProcessing(UInt32 surfaceId,
      bool showSurface,
      UInt32 userData,
      UInt32 appId,
      UInt32 entryCustomAnimationType,
      UInt32 exitCustomAnimationType)
{
   bool lRc = true;
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
   UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(surfaceId);

   // Retrieve current top popup
   UInt32 lCurrentSurfaceId = lActiveSurfaces.GetTop(lDisplayId);

   if (showSurface)
   {
      // First handle previous surface and its popups
      if (lCurrentSurfaceId != surfaceId)
      {
         ETG_TRACE_USR1(("Requested surface %u is %40spermanent",
                         surfaceId,
                         ProjectSettings::IsPermanent(surfaceId) ? "" : "non-"));
         if (0 != lCurrentSurfaceId)
         {
            ETG_TRACE_USR1(("Current surface %u is %40spermanent",
                            lCurrentSurfaceId,
                            ProjectSettings::IsPermanent(lCurrentSurfaceId) ? "" : "non-"));
            if (!ProjectSettings::IsPermanent(surfaceId))
            {
               // Re-organize application modal popups of closing surface ID
               PluginActions::ProcessPopupsOnAppHide(lCurrentSurfaceId);
               // Hide & notify previous surface
               lRc = lRc && PluginActions::NotifyVisibility(lCurrentSurfaceId,
                     SurfaceState::Unmapped,
                     lActiveSurfaces.GetUserData(lCurrentSurfaceId),
                     exitCustomAnimationType);

               // Close given surface from list of active surfaces,
               // considering also attached application popups
               if (lRc)
               {
                  // Remove given surface from list of active surfaces
                  (void) lActiveSurfaces.Remove(lCurrentSurfaceId);
               }
            }
         }

         // Update focus list of active surfaces
         (void) lActiveSurfaces.PutOnTop(surfaceId, userData, appId);

         // Show & notify new surface
         lRc = lRc && PluginActions::NotifyVisibility(surfaceId, SurfaceState::Mapped, userData, entryCustomAnimationType);

         if (!ProjectSettings::IsPermanent(surfaceId))
         {
            // Re-activate application popup, if applicable
            PluginActions::ProcessPopupsOnAppShow(surfaceId);
         }
         // Set input focus, if applicable
         (void)FocusHandler::GetInstance().UpdateInputFocus(surfaceId);
      }
   }
   else
   {
      // Get id of current application
      UInt32 lCurrentAppId = lActiveSurfaces.GetAppId(surfaceId);

      ActivePopups& lActivePopups = ActivePopups::GetInstance();
      lActivePopups.GetAppFirst();
      // Close given surface from list of active surfaces, considering also attached application popups
      // Remove given surface from list of active surfaces
      UInt32 lNewSurfaceId = lActiveSurfaces.Remove(surfaceId);

      // Process popups only for new application
      if (lCurrentAppId != lActiveSurfaces.GetAppId(lNewSurfaceId))
      {
         if (!ProjectSettings::IsPermanent(surfaceId))
         {
            // Re-organize application modal popups of closing surface ID
            PluginActions::ProcessPopupsOnAppHide(surfaceId);
         }
      }

      if ((0 != lNewSurfaceId) && (lNewSurfaceId != lCurrentSurfaceId))
      {
         // Show & notify newly focused surface
         lRc = lRc && PluginActions::NotifyVisibility(lNewSurfaceId,
               SurfaceState::Mapped,
               lActiveSurfaces.GetUserData(lNewSurfaceId),
               entryCustomAnimationType);
      }
      else
      {
         // Set new focused surface ID to a permanent surface of no other surface is available
         lNewSurfaceId = lActiveSurfaces.GetTop(lDisplayId, false);
      }

      // Process popups only for new application
      if ((0 != lNewSurfaceId) && (lCurrentAppId != lActiveSurfaces.GetAppId(lNewSurfaceId)))
      {
         // Re-activate popups, if applicable
         PluginActions::ProcessPopupsOnAppShow(lNewSurfaceId);
      }

      // Update input focus
      FocusHandler::GetInstance().UpdateInputFocus(lNewSurfaceId);

      // Hide surface & notify
      lRc = lRc && PluginActions::NotifyVisibility(surfaceId, SurfaceState::Unmapped, userData, exitCustomAnimationType);
   }

   return lRc;
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::Activate(const ServiceRequestArg& serviceRequestArg,
                                     UInt32 layerId,
                                     UInt32 surfaceId,
                                     UInt32 entryCustomAnimationType,
                                     UInt32 exitCustomAnimationType,
                                     UInt32 hint)
{
   ETG_TRACE_USR1(("Activate called { { %u, %u, %u }, %u, %u, %u, %u, %u }",
                   serviceRequestArg.RequestId(),
                   serviceRequestArg.AppId(),
                   serviceRequestArg.UserData(),
                   layerId,
                   surfaceId,
                   entryCustomAnimationType,
                   exitCustomAnimationType,
                   hint));

   // TODO: provide synchronization mechanism for surface creation
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   //workaround fix for Bug 324603
   usleep(50000);
   ETG_TRACE_USR1(("Activate delayed for 50 ms"));
#endif

   bool lRc = (0 != surfaceId);
   ilmErrorTypes lIlmError = ILM_SUCCESS;

   // Show surface if requestId's Bit 0 is set, hide surface otherwise
   const bool lShowSurface = ((serviceRequestArg.RequestId() & 0x1) == 0x1);

   if (lShowSurface)
   {
      // Adds given surface to given layer, if not already done
      lRc = IlmAccessor::AddSurfaceToLayer(layerId, surfaceId, false, lIlmError);
   }

   // -----------
   // Switch surface visibility and notify clients about the surface state change
   if (lRc)
   {
      // Cancel all running animations (if any) with new possible surface change request
      AnimationHandler::AbortAnimations(surfaceId);

      // Make surface transition, if new surface is to show (hiding the old one)
      // Else only hide the requested surface
      lRc = ActivationPostProcessing(surfaceId,
                                     lShowSurface,
                                     serviceRequestArg.UserData(),
                                     serviceRequestArg.AppId(),
                                     entryCustomAnimationType,
                                     exitCustomAnimationType);

      // Perform visibility transformation (if available)
      AnimationHandler::TriggerAnimations(false);
   }

   // Do a final all-in-one commit of all transactions towards layer manager
   if (lRc)
   {
      PluginActions::CheckExplicitCommit(surfaceId);
   }

   // Dump active surfaces & popups
   ActiveSurfaces::GetInstance().Dump();
   ActivePopups::GetInstance().Dump();
}


void ScreenBrokerActivator::TransitionAnimation(const ServiceRequestArg& serviceRequestArg,
      UInt32 layerId,
      UInt32 surfaceId,
      UInt32 animationType,
      UInt32 hint)
{
   AnimationDirection::Enum lAnimationDirection = AnimationDirection::Undefined;
   ETG_TRACE_USR1(("TransitionAnimation called { { %u, %u, %u }, %u, %u, %u, %u }",
                   serviceRequestArg.RequestId(),
                   serviceRequestArg.AppId(),
                   serviceRequestArg.UserData(),
                   layerId,
                   surfaceId,
                   animationType,
                   hint));

   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
   UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(surfaceId);

   // Retrieve current top popup
   UInt32 lCurrentSurfaceId = lActiveSurfaces.GetTop(lDisplayId);

   if (surfaceId == lCurrentSurfaceId)
   {
      AnimationHandler::AbortAnimations(surfaceId);
      lAnimationDirection = (AnimationDirection::Enum)AnimationDirection::Custom;
      // Show surface if requestId's Bit 0 is set, hide surface otherwise
      //const bool lShowSurface = ((serviceRequestArg.RequestId() & 0x1) == 0x1);
      AnimationUserData lAnimationUserData(serviceRequestArg.UserData(), SurfaceState::Mapped, true, true);

      bool lAnimationEstablished = false;
      //Schedule animation for Exit/Entry direction
      // Establish animation, dependant on XML configuration
      lAnimationEstablished = AnimationHandler::EstablishAnimation(lAnimationDirection,
                              surfaceId,
                              lAnimationUserData,
                              (AnimationType::Enum)animationType);

      if (!lAnimationEstablished)
      {
         ETG_TRACE_ERR(("Failed to establish TransitionAnimation !!!"));
      }
      else
      {
         // Perform visibility transformation (if available)
         AnimationHandler::TriggerAnimations();
      }
   }
   else
   {
      ETG_TRACE_ERR(("TransitionAnimation can't be established, due to mismatch in requested surfaceId [%u] and active surfaceId [%u]",
                     surfaceId,
                     lCurrentSurfaceId));
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::TransitionAnimationOfLayer(const ServiceRequestArg& serviceRequestArg,
      UInt32 layerId,
      std::vector<UInt32> surfaceIdList,
      UInt32 animationType,
      UInt32 hint)
{
   AnimationDirection::Enum lAnimationDirection = AnimationDirection::Undefined;
   ETG_TRACE_USR1(("TransitionAnimation called { { %u, %u, %u }, %u, %u, %u }",
                   serviceRequestArg.RequestId(),
                   serviceRequestArg.AppId(),
                   serviceRequestArg.UserData(),
                   layerId,
                   animationType,
                   hint));

   if ((0 != IlmAccessor::isLayerAvailable(layerId)))
   {
      ilmErrorTypes lIlmError = ILM_SUCCESS;
      AnimationHandler::AbortAnimations(layerId);
      if ((!surfaceIdList.empty()) && (0 != surfaceIdList.front()))
      {
         IlmAccessor::removeAllAttachedSurfaces(layerId);
         while (!surfaceIdList.empty())
         {
            UInt32 surfaceId = surfaceIdList.back();
            surfaceIdList.pop_back();
            if (IlmAccessor::AddSurfaceToLayer(layerId, surfaceId, true, lIlmError))
            {
               ETG_TRACE_USR4(("Surface %u attached to  %u: Layer", surfaceId, layerId));
               AnimationHandler::AbortAnimations(surfaceId);
            }
            else
            {
               ETG_TRACE_ERR((" Surface %u failed to attach to %u: Layer", surfaceId, layerId));
            }
         }
      }
      if (0 == IlmAccessor::getAttachedSurfaceCount(layerId))
      {
         return;
      }
   }
   else
   {
      ETG_TRACE_ERR(("Failed to Establish Layer animation for  %u: Layer!", layerId));
      return;
   }

   lAnimationDirection = (AnimationDirection::Enum)AnimationDirection::Custom;
   // Show surface if requestId's Bit 0 is set, hide surface otherwise
   //const bool lShowSurface = ((serviceRequestArg.RequestId() & 0x1) == 0x1);
   AnimationUserData lAnimationUserData(serviceRequestArg.UserData(), SurfaceState::Mapped, true, true);

   bool lAnimationEstablished = false;
   //Schedule animation for Exit/Entry direction
   // Establish animation, dependant on XML configuration
   lAnimationEstablished = AnimationHandler::EstablishLayerAnimation(lAnimationDirection,
                           layerId,
                           lAnimationUserData,
                           (AnimationType::Enum)animationType);

   if (!lAnimationEstablished)
   {
      ETG_TRACE_ERR(("Failed to establish TransitionAnimation for Layer %u !!!", layerId));
   }
   else
   {
      AnimationHandler::TriggerAnimations();
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::SetInputFocus(UInt32 surfaceId,
      bool keyboardFocus,
      bool pointerFocus,
      Int32 priority,
      UInt32 focusData)
{
   ETG_TRACE_USR1(("SetInputFocus calls {%u, %d, %d, %d, %u}",
                   surfaceId,
                   keyboardFocus,
                   pointerFocus,
                   priority,
                   focusData));

   // Set focus priority independent of surface status
   switch (focusData)
   {
      case ScreenBrokerProtocol::InputFocus::Force:
         if (keyboardFocus)
         {
            FocusHandler::GetInstance().ForceKeyboardFocus(surfaceId, priority);
         }

         if (pointerFocus)
         {
            FocusHandler::GetInstance().ForcePointerFocus(surfaceId, priority);
         }
         break;
      case ScreenBrokerProtocol::InputFocus::Reset:
         if (keyboardFocus)
         {
            FocusHandler::GetInstance().ForceKeyboardFocus(0, 0);
         }

         if (pointerFocus)
         {
            FocusHandler::GetInstance().ForcePointerFocus(0, 0);
         }
         break;
      default:
         if (keyboardFocus)
         {
            FocusHandler::GetInstance().SetKeyboardPriority(surfaceId, priority);
         }

         if (pointerFocus)
         {
            FocusHandler::GetInstance().SetPointerPriority(surfaceId, priority);
         }
   }

   // Get instance of active surfaces container
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
   UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(surfaceId);

   UInt32 lSurface = lActiveSurfaces.GetFirst(lDisplayId, false);

   while ((0 != lSurface) && (lSurface != surfaceId))
   {
      lSurface = lActiveSurfaces.GetNext(lDisplayId, false);
   }

   if ((0 != lSurface) && (lSurface == surfaceId))
   {
      if (keyboardFocus)
      {
         FocusHandler::GetInstance().SetKeyboardFocus(surfaceId);
      }

      if (pointerFocus)
      {
         FocusHandler::GetInstance().SetPointerFocus(surfaceId);
      }

      if (keyboardFocus || pointerFocus)
      {
         if (FocusHandler::GetInstance().UpdateInputFocus(surfaceId))
         {
            ilmErrorTypes lIlmError = ILM_SUCCESS;
            (void) IlmAccessor::Commit(lIlmError);
         }
      }
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::DeregisterSurface(UInt32 surfaceId)
{
   ETG_TRACE_USR1(("DeregisterSurface calls {%u}",
                   surfaceId));

   // Abort animations of surface with given surfaceId
   AnimationHandler::AbortAnimations(surfaceId);

   // Get instance of active surfaces container
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();
   UInt32 lDisplayId = PluginActions::GetDisplayIdOfSurface(surfaceId);

   UInt32 lSurface = lActiveSurfaces.GetFirst(lDisplayId, false);

   while ((0 != lSurface) && (lSurface != surfaceId))
   {
      lSurface = lActiveSurfaces.GetNext(lDisplayId, false);
   }

   if ((0 != lSurface) && (lSurface == surfaceId))
   {
      ilmErrorTypes lIlmError = ILM_SUCCESS;
      t_ilm_bool lVisible;
      IlmAccessor::GetSurfaceVisibility(surfaceId, lVisible, lIlmError);

      if (lVisible)
      {
         AnimationUserData lAnimationUserData(lActiveSurfaces.GetUserData(surfaceId),
                                              SurfaceState::Unmapped, true, false);
         PluginActions::SendNotification(surfaceId, lAnimationUserData);
      }
   }

   // Update input focus
   FocusHandler::GetInstance().RemoveSurface(surfaceId);

   FocusHandler::UpdateInputAcceptance(surfaceId, false);
   if (FocusHandler::GetInstance().TransferInputFocus())
   {
      ilmErrorTypes lIlmError = ILM_SUCCESS;
      (void) IlmAccessor::Commit(lIlmError);
   }
}


// ------------------------------------------------------------------------
DimensionArg ScreenBrokerActivator::GetPreferredDimensions(const ServiceRequestArg& serviceRequestArg,
      UInt32 layerId,
      const DimensionArg& defaultDimension)
{
   ETG_TRACE_USR1(("GetPreferredDimensions called { { %u, %u, %u }, %u, { %ux%u } }",
                   serviceRequestArg.RequestId(),
                   serviceRequestArg.AppId(),
                   serviceRequestArg.UserData(),
                   layerId,
                   defaultDimension.Width(),
                   defaultDimension.Height()));
   (void) serviceRequestArg;
   (void) layerId;

   // Return default screen area (aka. layer) dimension as preferred dimensions for surfaces on this layer
   DimensionArg lDimensionsArg(defaultDimension);

   return lDimensionsArg;
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::Action(UInt32 actionId, UInt32 actionData)
{
   switch (actionId)
   {
      case ScreenBrokerProtocol::ActionId::StopAnimations:
      {
         AnimationHandler::AbortAnimations();
         break;
      }
      case ScreenBrokerProtocol::ActionId::Commit:
      {
         // Do a commit towards layer manager
         ilmErrorTypes lIlmError = ILM_SUCCESS;
         (void) IlmAccessor::Commit(lIlmError);
         break;
      }
      default:
         ETG_TRACE_SYS(("No action %u with data %u available! Request ignored!",
                        actionId,
                        actionData));
         break;
   }
}


// ------------------------------------------------------------------------
void ScreenBrokerActivator::RequestCurrentStatus(UInt32 requestId)
{
   // Get instance of active surfaces container
   ActiveSurfaces& lActiveSurfaces = ActiveSurfaces::GetInstance();

   IScreenLayouter* lScreenLayouter = ServiceApi::ScreenLayouter();
   if (0 != lScreenLayouter)
   {
      IScreenLayouter::DisplayIdList lDisplayIdList = lScreenLayouter->GetDisplayIdList();
      switch (requestId)
      {
         case ScreenBrokerProtocol::CurrentStatusRequestId::TopActiveSurfaceNonPermanent:
         case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveSurfaceNonPermanent:
         {
            bool lSurfaceFound = false;
            for (IScreenLayouter::DisplayIdList::iterator lIt = lDisplayIdList.begin(); lIt != lDisplayIdList.end(); ++lIt)
            {
               UInt32 lSurfaceId = lActiveSurfaces.GetTop(*lIt);
               if (0 != lSurfaceId)
               {
                  std::bitset<sizeof(UInt32)> lStatus(0);
                  ServiceApi::NotifyCurrentStatus(requestId,
                                                  lSurfaceId,
                                                  UInt32(lStatus.to_ulong()),
                                                  lActiveSurfaces.GetUserData(lSurfaceId));
                  ETG_TRACE_USR1(("%40s (TopActiveSurfaceNonPermanent): %u/0x%08x: status(%u)",
                                  __FUNCTION__,
                                  lSurfaceId,
                                  lActiveSurfaces.GetUserData(lSurfaceId),
                                  UInt32(lStatus.to_ulong())));
                  lSurfaceFound = true;
               }
            }
            if (!lSurfaceFound)
            {
               ETG_TRACE_SYS(("%40s (TopActiveSurfaceNonPermanent): No top non-permanent surface found! Request ignored!", __FUNCTION__));
            }
            break;
         }
         case ScreenBrokerProtocol::CurrentStatusRequestId::TopActiveSurface:
         case ScreenBrokerProtocol::CurrentStatusRequestIdForTTFisCMD::TopActiveSurface:
         {
            bool lSurfaceFound = false;
            for (IScreenLayouter::DisplayIdList::iterator lIt = lDisplayIdList.begin(); lIt != lDisplayIdList.end(); ++lIt)
            {
               UInt32 lSurfaceId = lActiveSurfaces.GetTop(*lIt, false);
               if (0 != lSurfaceId)
               {
                  std::bitset<sizeof(UInt32)> lStatus(0);
                  ServiceApi::NotifyCurrentStatus(requestId,
                                                  lSurfaceId,
                                                  UInt32(lStatus.to_ulong()),
                                                  lActiveSurfaces.GetUserData(lSurfaceId));
                  ETG_TRACE_USR1(("%40s (TopActiveSurface): %u/0x%08x: status(%u)",
                                  __FUNCTION__,
                                  lSurfaceId,
                                  lActiveSurfaces.GetUserData(lSurfaceId),
                                  UInt32(lStatus.to_ulong())));
                  lSurfaceFound = true;
               }
            }
            if (!lSurfaceFound)
            {
               ETG_TRACE_SYS(("%40s (TopActiveSurface): No top surface found! Request ignored!", __FUNCTION__));
            }
            break;
         }
         default:
            ETG_TRACE_SYS(("No status for request %u available! Request ignored!", requestId));
            break;
      }
   }
   else
   {
      ETG_TRACE_SYS(("@ Requesting current status failed: ScreenLayouter instance not accessible!"));
   }
}


}

// ========================================================================
// Global registration for plugin
// ========================================================================
// ------------------------------------------------------------------------
class ScreenBrokerActivatorPluginProxy
{
   public:
      ScreenBrokerActivatorPluginProxy()
      {
         // Register the plugin instance creation at the global factory map
         gScreenBrokerPluginFactoryMap[ScreenBroker::IScreenBrokerActivator::Name()] = ScreenBroker::ScreenBrokerActivator::Create;
      }
};


// Create one instance of the proxy to perform registration
ScreenBrokerActivatorPluginProxy gScreenBrokerActivatorPlugin;
