/* ***************************************************************************************
* FILE:          ServiceAdaptor.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ServiceAdaptor.h is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


#if !defined(ScreenBroker_Service_ServiceAdaptor_h)
#define ScreenBroker_Service_ServiceAdaptor_h

#include <ScreenBroker/ScreenBroker.h>
#include <ScreenBroker/DimensionArg.h>
#include <ScreenBroker/KeyState.h>
#include <ScreenBroker/Plugin/IScreenLayouter.h>
#include <ScreenBroker/Service/ServiceRequestArg.h>
#include <ScreenBroker/Service/ScreenArea.h>
#include <ScreenBroker/Service/Service.h>
#include <ScreenBroker/SurfaceState.h>
#include <ScreenBroker/LayerState.h>
#include <ScreenBroker/Service/ClientState.h>
#include <ScreenBroker/Util/CriticalSection.h>
#include <map>

namespace ScreenBroker {
class Synchronize;

namespace Internal {
/// @addtogroup ScreenBrokerService
/// @{

/**
 * @brief The ServiceAdaptor class wraps the communication layer and manages its interface to the client.
 *
 * Additionally it holds the list of currently connected client applications (ClientState).
 */
class ServiceAdaptor
{
      friend class ::ScreenBroker::Synchronize;
   public:

      ///
      static ServiceAdaptor* GetInstance();

      ///
      virtual ~ServiceAdaptor();

      /**
       * @brief Initializes the service provided to the clients and reads in the
       * screen area definition provided by the plugin ScreenLayouter.
       * @return True if initialization was successful, false otherwise.
       */
      virtual bool Init();

      /**
       * @brief runs the service
       * the function will return when the service has been terminated
       */
      virtual void Run() = 0;

      /**
       * @brief terminates the service
       */
      virtual void Terminate() = 0;

      /**
       * @brief Triggers a reset the screen broker service internal states based on the given reset level
       * without restarting the service.
       *
       * @param mode Level of reset.
       */
      void Reset(ResetMode::Enum mode);

      /**
       * @brief Triggers a diagnosis call towards plugins for e.g. dumping current states, etc.
       */
      void Diagnosis();

      /**
       * @brief Requests the unique service ID (normally provided by the respective transport layer instance).
       *
       * @return Unique service ID.
       */
      virtual const Char* GetServiceId() const = 0;

      /**
       * @brief Registers the surface with the given ID to the given layer ID.
       *
       * @param surfaceId The ID of the surface to register.
       * @param layerId The layer ID where the surface shall be registered for.
       * @param hint Additional registration userData, which is provided as hint
       * in subsequent plugins calls (e.g. ScreenBrokerActivator::Activate).
       */
      void RegisterSurfaceToLayer(UInt32 surfaceId,
                                  UInt32 layerId,
                                  UInt32 userData);

      /**
       * @brief Notifies clients that the state of the indicated surface has changed.
       *
       * @note The notification of the state change is triggered by the respective screen broker plugin
       * (ScreenbrokerActivator or PopupManagerActivator).
       *
       * @param userData User data which may give additional information from plugin to application.
       * @param surfaceId Surface ID which this state change applies to.
       * @param surfaceState New state of surface.
       */
      void NotifySurfaceStateChange(UInt32 userData,
                                    UInt32 surfaceId,
                                    SurfaceState::Enum surfaceState);

      /**
       * @brief Notifies clients that a runtime error has occurred with given request.
       *
       * @param serviceRequestArg Request information comprises request ID, application ID and user data, which is provided
       * by the application request returned in the error response.
       * @param systemErrno System error codes (like linux extended error codes defined in linux/errno.h).
       * @param presentationErrno Error codes specific to the visual presentation module screen broker
       * is connected to (e.g. layer manager).
       */
      void NotifyError(const ServiceRequestArg& serviceRequestArg,
                       Int32 systemErrno,
                       Int32 presentationErrno);

      /**
       * @brief Notifies a key event to a client which is holding the focused surface (normal or popup).
       *
       * @param userData User data which may give additional information from plugin to application.
       * @param surfaceId ID of focused surface to which the key event shall be sent.
       * @param keyCode 32bit code of the key event.
       * @param keyState The state of the key event.
       */
      void NotifyKeyEvent(UInt32 userData,
                          UInt32 surfaceId,
                          UInt32 keyCode,
                          KeyState::Enum keyState);

      /**
       * @brief Notifies about current status details based on the request.
       *
       * @note The notification is sent to the owner of the surface defined via surfaceId.
       *
       * @param requestId 32bit information to inform about details of the requests nature.
       * @param surfaceId The ID of the surface, which identifies the response path, may also be part of the status response.
       * @param status 32bit of response data carrying the status information.
       * @param userData Additional 32bit of user data for whatever use.
       */
      void NotifyCurrentStatus(UInt32 requestId,
                               UInt32 surfaceId,
                               UInt32 status,
                               UInt32 userData);

      /**
       * @brief Notifies about Layer State details based on the update.
       *
       * @note The notification is sent to the owner of the first valid attached surface.
       *
       * @param requestId 32bit information to inform about details of the requests nature.
       * @param layerId layer ID which this state change applies to.
       * @param surfaceState New state of surface.
       */
      void NotifyLayerStateChanged(UInt32 requestId,
                                   UInt32 layerId,
                                   LayerState::Enum state);

      /**
       * @brief Notifies clients that close on external touch is enabled or disabled.
       * @param status Enabled or Disabled
       */
      void NotifyCloseOnExternalTouch(bool status);

      /**
       * @brief Add information about build-in layers/surfaces
       *
       * @param layerid
       * @param surfaceId
       * @param screenId
       * @param connectorName
       */
      virtual void UpdateBuildInLayerInformation(UInt32 layerId, UInt32 surfaceId, UInt32 screenId, UInt32 width, UInt32 height, std::string connectorName = "") = 0;

   protected:
      friend class Service;
      class StaticSurfaceContext
      {
         public:
            StaticSurfaceContext() : mLayerId(0), mHint(0) {}
            StaticSurfaceContext(UInt32 layerId, UInt32 hint)
               : mLayerId(layerId),
                 mHint(hint) {}
            UInt32 LayerId() const
            {
               return mLayerId;
            }
            UInt32 Hint() const
            {
               return mHint;
            }
         private:
            UInt32 mLayerId;
            UInt32 mHint;
      };
      typedef std::map<std::string, ClientState> ClientStateMap;
      typedef std::map<UInt32, StaticSurfaceContext> StaticSurfaceContextMap;

      /// Map holding the active client states.
      ClientStateMap mClientStateMap;

      /// Map holding the assignment from surfaceId to a static surface context
      /// used for on demand surface registration.
      StaticSurfaceContextMap mStaticSurfaceContextMap;

      ///
      ServiceAdaptor();

      ///
      static ScreenArea* GetScreenArea(UInt32 screenAreaId);

      /**
       * @brief Provides client state by given client ID.
       *
       * @note If client ID not found in list, then a new entry with given client ID is created.
       *
       * @param clientId Client ID to search the corresponding client state for.
       * @return Client state corresponding to given client ID.
       */
      ClientState& GetClientState(const std::string& clientId);

      /**
       * @brief Provides client state by given surface ID.
       *
       * @param surfaceId Surface ID to search the corresponding client state for.
       * @return Client state corresponding to given surface ID if found, 0 otherwise.
       */
      ClientState* GetClientState(UInt32 surfaceId);

      /**
       * @brief Destroys the service and cleans up internal states.
       */
      void Destroy();

      ///
      bool HandleClientTermination(const std::string& clientId);

      ///
      void HandleGetDisplayId(const std::string& clientId,
                              const RequestArg& reqArg,
                              UInt32 screenAreaId);

      ///
      void HandleGetPreferredDimensions(const std::string& clientId,
                                        const RequestArg& reqArg,
                                        UInt32 screenAreaId);

      ///
      void HandleRegisterSurface(const std::string& clientId,
                                 const RequestArg& reqArg,
                                 UInt32 screenAreaId,
                                 UInt32 surfaceId);

      ///
      void HandleDeregisterSurface(const std::string& clientId,
                                   const RequestArg& reqArg,
                                   UInt32 surfaceId,
                                   bool sendToIlm);

      ///
      void HandleActivateApplication(const std::string& clientId,
                                     const RequestArg& reqArg,
                                     UInt32 surfaceId,
                                     UInt32 entryCustomAnimationType,
                                     UInt32 exitCustomAnimationType);

      ///
      void HandleStartAnimation(const std::string& clientId,
                                const RequestArg& reqArg,
                                UInt32 surfaceId,
                                UInt32 animationType,
                                UInt32 hint);
      ///
      void HandleAttachandAnimateLayer(const std::string& clientId,
                                       const RequestArg& reqArg,
                                       UInt32 layerId,
                                       std::vector<UInt32> surfaceIdList,
                                       UInt32 animationType,
                                       UInt32 hint);

      ///
      void HandleShowPopup(const std::string& clientId,
                           const RequestArg& reqArg,
                           UInt32 surfaceId,
                           const PopupPresentationArg& popupPresentationArg);

      ///
      void HandleHidePopup(const std::string& clientId,
                           const RequestArg& requestArg,
                           UInt32 surfaceId);

      ///
      void HandleCloseOnExternalTouch(const std::string& clientId,
                                      const RequestArg& reqArg) const;

      ///
      void HandleSetPopupFilter(const std::string& clientId,
                                const RequestArg& reqArg,
                                bool disableAll,
                                const PopupPresentationArg& popupPresentationArg) const;

      ///
      void HandleClearPopupFilter(const std::string& clientId,
                                  const RequestArg& requestArg) const;

      ///
      void HandleKeyFeedback(const std::string& clientId,
                             UInt32 userData,
                             UInt32 keyCode,
                             KeyState::Enum keyState,
                             bool consumed) const;

      ///
      void HandleAction(const std::string& clientId,
                        UInt32 actionId,
                        UInt32 actionData) const;

      ///
      void HandleSetInputFocus(const std::string& clientId,
                               UInt32 surfaceId,
                               bool keyboardFocus,
                               bool pointerFocus,
                               Int32 priority,
                               UInt32 focusData);

      ///
      void HandleRequestCurrentStatus(const std::string& clientId,
                                      UInt32 requestId) const;

      /**
       * @brief Does an on demand surface registration of given client ID.
       *
       * @note If client ID not found in list, then a new entry with given client ID is created.
       *
       * @param clientId Client ID to search the corresponding client state for.
       * @param surfaceId Surface ID which shall be registered on demand.
       */
      void OnDemandSurfaceRegistration(const RequestArg& reqArg,
                                       const std::string& clientId,
                                       UInt32 surfaceId);

      ///
      virtual void SendOnGetDisplayId(const std::string& clientId,
                                      const RequestArg& reqArg,
                                      UInt32 displayId) = 0;

      ///
      virtual void SendOnGetPreferredDimensions(const std::string& clientId,
            const RequestArg& reqArg,
            const DimensionArg& dimArg) = 0;

      ///
      virtual void SendNotifySurfaceStateChange(const std::string& clientId,
            UInt32 userData,
            UInt32 surfaceId,
            UInt32 displayId,
            UInt32 displayAlias,
            SurfaceState::Enum state) = 0;

      ///
      virtual void SendNotifyError(const ServiceRequestArg& serviceRequestArg,
                                   Int32 systemErrno,
                                   Int32 presentationErrno) = 0;

      ///
      virtual void SendNotifyKeyEvent(const std::string& clientId,
                                      UInt32 userData,
                                      UInt32 surfaceId,
                                      UInt32 keyCode,
                                      KeyState::Enum keyState) = 0;

      ///
      virtual void SendNotifyCurrentStatus(const std::string& clientId,
                                           UInt32 requestId,
                                           UInt32 surfaceId,
                                           UInt32 status,
                                           UInt32 userData) = 0;
      ///
      virtual void SendNotifyLayerStateChanged(const std::string& clientId,
            UInt32 requestId,
            UInt32 layerId,
            LayerState::Enum state) = 0;
      ///
      virtual void SendNotifyCloseOnExternalTouch(bool status) = 0;

   private:
      /// Critical section to secure a sequential processing of external
      /// (e.g. reset, diagnosis, ..) and internal requests (e.g. plugin calls)
      CriticalSection mCs;
};


/// @}
}


}

#endif
