/* ***************************************************************************************
* FILE:          ServiceAdaptor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ServiceAdaptor.cpp is part of HMI-Base ScreenBroker
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */


// =============================================================================
// Do not make this method const, as it calls a plugin interface which depending
// on the implementation, may not be const.
//lint -efunc(1762, ScreenBroker::Internal::ServiceAdaptor::Diagnosis)    "Member function could be made const"
// =============================================================================

#include <ScreenBroker/Version.h>
#include "ServiceAdaptor.h"
#include "ClientState.h"
#include <ScreenBroker/Service/ASF/AsfServiceAdaptor.h>

#include "PluginManager.h"
#include <ScreenBroker/DimensionArg.h>
#include <ScreenBroker/RequestArg.h>
#include <ScreenBroker/PopupPresentationArg.h>
#include <ScreenBroker/Service/Synchronize.h>
#include <ScreenBroker/Service/PopupManager/PopupManager.h>

#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
#include "LayerManagerAccessor.h"

#endif

#include "ScreenBroker/ScreenBroker_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_SB_SCREENBROKERSERVICE
#include "trcGenProj/Header/ServiceAdaptor.cpp.trc.h"
#endif


namespace ScreenBroker {
namespace Internal {
// SCREENBROKER_LOG_SET_REALM(ScreenBroker::LogRealm::ScreenBrokerService);

// ------------------------------------------------------------------------
ServiceAdaptor* ServiceAdaptor::GetInstance()
{
   return AsfServiceAdaptor::DoGetInstance();
}


// ------------------------------------------------------------------------
ServiceAdaptor::ServiceAdaptor()
{
}


// ------------------------------------------------------------------------
ServiceAdaptor::~ServiceAdaptor()
{
   mClientStateMap.clear();
   mStaticSurfaceContextMap.clear();
}


// ------------------------------------------------------------------------
bool ServiceAdaptor::Init()
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();
   return true;
}


// ------------------------------------------------------------------------
void ServiceAdaptor::Destroy()
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();
   mClientStateMap.clear();
   mStaticSurfaceContextMap.clear();
}


// ------------------------------------------------------------------------
void ServiceAdaptor::Reset(ResetMode::Enum mode)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ETG_TRACE_USR1(("Executing reset with mode %d!", mode));

   switch (mode)
   {
      //TODO: ScreenLayouterPlugin reset (including layer reset)
//            case ResetMode::ScreenLayouterPlugin: {
//                IScreenBrokerPlugin * lPlugin = Internal::PluginManager::GetInstance().ScreenLayouter();
//                if (0 != lPlugin) {
//                    lPlugin->Reset();
//                    lPlugin->Init();
//                }
//                break;
//            }
      case ResetMode::ActivatorPlugins:
      {
         //SCREENBROKER_LOG_FN_MARK("ResetMode::ActivatorPlugins");
         IScreenBrokerPlugin* lPlugin =
            Internal::PluginManager::GetInstance().GetPlugin(PluginId::PopupManagerActivator);
         if (0 != lPlugin)
         {
            lPlugin->Reset();
            lPlugin->Init();
         }
         lPlugin =
            Internal::PluginManager::GetInstance().GetPlugin(PluginId::ScreenBrokerActivator);
         if (0 != lPlugin)
         {
            lPlugin->Reset();
            lPlugin->Init();
         }
         break;
      }

      //TODO: plugin reset (including layer reset via ScreenLayouterPlugin reset)
//            case ResetMode::Plugins: {
//                Reset(ResetMode::ScreenLayouterPlugin);
//                Reset(ResetMode::ActivatorPlugins);
//                break;
//            }

      case ResetMode::SurfaceInstances:
      {
         //SCREENBROKER_LOG_FN_MARK("ResetMode::SurfaceInstances");
         Reset(ResetMode::ActivatorPlugins);
         PopupManager::GetInstance().Reset();
         for (ClientStateMap::iterator it = mClientStateMap.begin();
               mClientStateMap.end().operator != (it); ++it)
         {
            ClientState& lClientState = (*it).second;
            lClientState.Reset();
         }
         break;
      }
      case ResetMode::ClientInstances:
      {
         //SCREENBROKER_LOG_FN_MARK("ResetMode::ClientInstances");
         Reset(ResetMode::ActivatorPlugins);
         PopupManager::GetInstance().Reset();
         mClientStateMap.clear();
         break;
      }

      //TODO: full reset (including layer reset via ScreenBrokerActivator
//            case ResetMode::Full: {
//                Reset(ResetMode::ScreenLayouterPlugin);
//                Reset(ResetMode::ClientInstances);
//                mClientStateMap.clear();
//                break;
//            }
      default:
         ETG_TRACE_SYS(("Reset mode %d not available!", mode));
         break;
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::Diagnosis()
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   for (Int i = static_cast<Int>(PluginId::First);
         i < static_cast<Int>(PluginId::Count);
         ++i)
   {
      IScreenBrokerPlugin* lPlugin =
         Internal::PluginManager::GetInstance().GetPlugin(static_cast<PluginId::Enum>(i));
      if (0 != lPlugin)
      {
         lPlugin->Diagnosis();
      }
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::RegisterSurfaceToLayer(UInt32 surfaceId,
      UInt32 layerId,
      UInt32 userData)
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR4(("Memorize on-demand surface registration mapping (surface: %u, layer: %u, userData: %u)",
                   surfaceId, layerId, userData));
   StaticSurfaceContextMap::iterator it = mStaticSurfaceContextMap.find(surfaceId);
   if (mStaticSurfaceContextMap.end().operator != (it))
   {
      ETG_TRACE_SYS(("Existing on-demand surface registration mapping for surface %u will be overwritten (layer: %u -> %u, userData: %u -> %u)!",
                     surfaceId,
                     (*it).second.LayerId(),
                     layerId,
                     (*it).second.Hint(),
                     userData));
   }
   mStaticSurfaceContextMap[surfaceId] = StaticSurfaceContext(layerId, userData);
}


// ------------------------------------------------------------------------
void ServiceAdaptor::NotifySurfaceStateChange(UInt32 userData,
      UInt32 surfaceId,
      SurfaceState::Enum surfaceState)
{
   //SCREENBROKER_LOG_FN();
   ClientState* lClientState = GetClientState(surfaceId);

   if (0 != lClientState)
   {
      UInt32 lDisplayId = 0;
      UInt32 lDisplayAlias = 0;
      const ScreenArea* lScreenArea = lClientState->GetScreenArea(surfaceId);
      if (0 != lScreenArea)
      {
         lDisplayId = lScreenArea->GetDisplayId();
         lDisplayAlias = lScreenArea->GetDisplayAlias();
      }
      ETG_TRACE_USR1(("Send OnSurfaceStateChanged(%40s, %u, %u, %u, %d)",
                      lClientState->GetClientId().c_str(),
                      userData,
                      surfaceId,
                      lDisplayId,
                      surfaceState));

      // Inform dedicated client about a surface state change
      SendNotifySurfaceStateChange(lClientState->GetClientId().c_str(),
                                   userData,
                                   surfaceId,
                                   lDisplayId,
                                   lDisplayAlias,
                                   surfaceState);
   }
   else
   {
      ETG_TRACE_SYS(("NotifySurfaceStateChange (state: %d) failed: Client registration for surface ID:%u not found",
                     surfaceState,
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::NotifyError(const ServiceRequestArg& serviceRequestArg,
                                 Int32 systemErrno,
                                 Int32 presentationErrno)
{
   //SCREENBROKER_LOG_FN();
   ETG_TRACE_USR1(("Send OnError(%40s, {%u, %u, %u}, %d, %d)",
                   serviceRequestArg.ClientId().c_str(),
                   serviceRequestArg.RequestId(),
                   serviceRequestArg.AppId(),
                   serviceRequestArg.UserData(),
                   systemErrno,
                   presentationErrno));

   SendNotifyError(serviceRequestArg, systemErrno, presentationErrno);
}


// ------------------------------------------------------------------------
void ServiceAdaptor::NotifyKeyEvent(UInt32 userData,
                                    UInt32 surfaceId,
                                    UInt32 keyCode,
                                    KeyState::Enum keyState)
{
   //SCREENBROKER_LOG_FN();
   ClientState* lClientState = GetClientState(surfaceId);
   if (0 != lClientState)
   {
      ETG_TRACE_USR1(("Send OnKeyEvent(%40s, %u, %u, %u, %d)",
                      lClientState->GetClientId().c_str(),
                      userData,
                      surfaceId,
                      keyCode,
                      keyState));

      // Inform dedicated client about a key event
      SendNotifyKeyEvent(lClientState->GetClientId().c_str(),
                         userData,
                         surfaceId,
                         keyCode,
                         keyState);
   }
   else
   {
      ETG_TRACE_SYS(("NotifyKeyEvent (code: %u, state: %d) failed: Client registration for surface ID:%u not found",
                     keyCode,
                     keyState,
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::NotifyCurrentStatus(UInt32 requestId,
      UInt32 surfaceId,
      UInt32 status,
      UInt32 userData)
{
   //SCREENBROKER_LOG_FN();
   ClientState* lClientState = GetClientState(surfaceId);
   if (0 != lClientState)
   {
      ETG_TRACE_USR1(("Send NotifyCurrentStatus(%40s, %u, %u, %u, %u)",
                      lClientState->GetClientId().c_str(),
                      requestId,
                      surfaceId,
                      status,
                      userData));

      // Inform dedicated client about a key event
      SendNotifyCurrentStatus(lClientState->GetClientId().c_str(),
                              requestId,
                              surfaceId,
                              status,
                              userData);
   }
   else
   {
      ETG_TRACE_USR1(("Client registration for surface ID:%u not found"
                      "Send NotifyCurrentStatus (, %u, %u, %u, %u)", surfaceId,
                      requestId,
                      status,
                      userData,
                      surfaceId));
      // If not clientstate is available send notfication without ClientId
      SendNotifyCurrentStatus("",
                              requestId,
                              surfaceId,
                              status,
                              userData);
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::NotifyLayerStateChanged(UInt32 requestId,
      UInt32 layerId,
      LayerState::Enum layerstate)
{
   //SCREENBROKER_LOG_FN();
   ClientState* lClientState = GetClientState(layerId);
   if (0 != lClientState)
   {
      ETG_TRACE_USR1(("Send NotifyLayerStateChanged(%40s, %u, %u, %u)",
                      lClientState->GetClientId().c_str(),
                      requestId,
                      layerId,
                      layerstate));

      // Inform dedicated client about a key event
      SendNotifyLayerStateChanged(lClientState->GetClientId().c_str(),
                                  requestId,
                                  layerId,
                                  layerstate);
   }
   else
   {
      ETG_TRACE_USR1(("Client registration for Layer ID:%u not found"
                      "Send NotifyLayerStateChanged (%u, %u, %u)", layerId,
                      layerId,
                      requestId,
                      layerstate));
      // If not clientstate is available send notfication without ClientId
      SendNotifyLayerStateChanged("",
                                  requestId,
                                  layerId,
                                  layerstate);
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::NotifyCloseOnExternalTouch(bool status)
{
   //SCREENBROKER_LOG_FN();
   SendNotifyCloseOnExternalTouch(status);
}


// ------------------------------------------------------------------------
bool ServiceAdaptor::HandleClientTermination(const std::string& clientId)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ETG_TRACE_SYS(("Client '%10s': Client termination", clientId.c_str()));

   // Finally remove the related client state
   ClientStateMap::iterator it = mClientStateMap.find(clientId);
   if (mClientStateMap.end().operator != (it))
   {
      ETG_TRACE_SYS(("Client '%10s': Remove client state", (*it).first.c_str()));
      mClientStateMap.erase(it);
   }

   return true;
}


// ------------------------------------------------------------------------
ScreenArea* ServiceAdaptor::GetScreenArea(UInt32 screenAreaId)
{
   IScreenLayouter* lScreenLayouter = PluginManager::GetInstance().Plugin<IScreenLayouter>();
   ScreenArea* lScreenArea = 0;
   if (0 != lScreenLayouter)
   {
      lScreenArea = lScreenLayouter->GetScreenArea(screenAreaId);
   }
   return lScreenArea;
}


// ------------------------------------------------------------------------
ClientState& ServiceAdaptor::GetClientState(const std::string& clientId)
{
   ClientStateMap::iterator it = mClientStateMap.find(clientId);

   // If not found then create a new entry
   if (mClientStateMap.end().operator == (it))
   {
      ETG_TRACE_USR1(("Registering client state %40s", clientId.c_str()));
      mClientStateMap[clientId] = ClientState(clientId);
      return mClientStateMap[clientId];
   }

   return (*it).second;
}


// ------------------------------------------------------------------------
ClientState* ServiceAdaptor::GetClientState(UInt32 surfaceId)
{
   // Scan all client stated if they have a surface registered to a screen area of theirs
   ClientStateMap::iterator it;
   for (it = mClientStateMap.begin();
         mClientStateMap.end().operator != (it); ++it)
   {
      if (0 != it->second.GetScreenArea(surfaceId))
      {
         return &(*it).second;
      }
   }
   return 0;
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleGetDisplayId(const std::string& clientId,
                                        const RequestArg& reqArg,
                                        UInt32 screenAreaId)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ETG_TRACE_USR4(("Receive request GetDisplayId(%40s, {%u, %u, %u}, %u)",
                   clientId.c_str(),
                   reqArg.RequestId(),
                   reqArg.AppId(),
                   reqArg.UserData(),
                   screenAreaId));

   // Retrieve screen area object
   ScreenArea* lScreenArea = GetScreenArea(screenAreaId);
   if (0 != lScreenArea)
   {
      // Send response back to sender
      SendOnGetDisplayId(clientId,
                         reqArg,
                         lScreenArea->GetDisplayId());
      ETG_TRACE_USR4(("Send OnGetDisplayId(%40s, {%u, %u, %u}, %u)",
                      clientId.c_str(),
                      reqArg.RequestId(),
                      reqArg.AppId(),
                      reqArg.UserData(),
                      lScreenArea->GetDisplayId()));
   }
   else
   {
      ETG_TRACE_ERR(("Failed to get display ID of screen area %u: screen area not supported!",
                     screenAreaId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleGetPreferredDimensions(const std::string& clientId,
      const RequestArg& reqArg,
      UInt32 screenAreaId)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   ETG_TRACE_USR4(("Receive request GetPreferredDimensions(%40s, {%u, %u, %u}, %u)",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData(),
                   screenAreaId));

   // Find related screen area
   ScreenArea* lScreenArea = GetScreenArea(screenAreaId);
   if (0 != lScreenArea)
   {
      // Prepare default dimension
      DimensionArg lDefaultDimension(lScreenArea->GetWidth(), lScreenArea->GetHeight());

      // Get preferred dimension from screen broker activator plugin request
      IScreenBrokerActivator* lScreenBrokerActivator = PluginManager::GetInstance().Plugin<IScreenBrokerActivator>();

      if (0 != lScreenBrokerActivator)
      {
         DimensionArg lPreferredDimension = lScreenBrokerActivator->GetPreferredDimensions(lServiceRequestArg,
                                            lScreenArea->GetLayerId(),
                                            lDefaultDimension);

         // Send response back to sender
         DimensionArg dimArg(lPreferredDimension.Width(), lPreferredDimension.Height());
         SendOnGetPreferredDimensions(clientId,
                                      reqArg,
                                      dimArg);
         ETG_TRACE_USR4(("Send OnGetPreferredDimensions(%40s, {%u, %u, %u}, {%u, %u})",
                         clientId.c_str(),
                         lServiceRequestArg.RequestId(),
                         lServiceRequestArg.AppId(),
                         lServiceRequestArg.UserData(),
                         lPreferredDimension.Width(),
                         lPreferredDimension.Height()));
      }
      else
      {
         ETG_TRACE_ERR(("Failed to get preferred dimension of screen area %u: plugin not available!",
                        screenAreaId));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Failed to get preferred dimension of screen area %u: screen area not supported!",
                     screenAreaId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleRegisterSurface(const std::string& clientId,
      const RequestArg& reqArg,
      UInt32 screenAreaId,
      UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg.RequestId(),
                                        reqArg.AppId(),
                                        reqArg.UserData(),
                                        clientId);

   ETG_TRACE_USR4(("Receive request RegisterSurface(%40s, {%u, %u, %u}, %u, %u)",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData(),
                   screenAreaId,
                   surfaceId));

   // Register surface to client's screen area, only if not yet registered (to any client)
   ClientState* lClientState = GetClientState(surfaceId);
   if (0 == lClientState)
   {
      // Retrieve (and create) client state and register surface to given screen area
      ClientState& lNewClientState = GetClientState(clientId);
      ScreenArea* lScreenArea = GetScreenArea(screenAreaId);
      if (0 != lScreenArea)
      {
         lNewClientState.RegisterSurface(lServiceRequestArg,
                                         *lScreenArea,
                                         surfaceId);
         SendNotifySurfaceStateChange(clientId,
                                      lServiceRequestArg.UserData(),
                                      surfaceId,
                                      lScreenArea->GetDisplayId(),
                                      lScreenArea->GetDisplayAlias(),
                                      SurfaceState::Registered);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Failed to register surface %u to screen area %u: screen area not supported!",
                        clientId.c_str(),
                        surfaceId,
                        screenAreaId));
      }
   }
   else
   {
      if (lClientState->GetClientId() == clientId)
      {
         const ScreenArea* lScreenArea = lClientState->GetScreenArea(surfaceId);
         if (0 != lScreenArea)
         {
            if (lScreenArea->GetId() == screenAreaId)
            {
               ETG_TRACE_SYS(("Client '%10s': Surface %u is already registered to screen area %u!",
                              clientId.c_str(),
                              surfaceId,
                              screenAreaId));
            }
            else
            {
               ETG_TRACE_ERR(("Client '%10s': Failed to register surface %u to screen area %u: surface already registered to screen area ID:%u!",
                              clientId.c_str(),
                              surfaceId,
                              screenAreaId,
                              lScreenArea->GetId()));
            }
         }
         else
         {
            // Should never happen!!!
            ETG_TRACE_FATAL(("Client '%10s': Surface %u seems to be registered but screen area relation is missing!",
                             clientId.c_str(),
                             surfaceId));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Failed to register surface %u to screen area %u: surface already used by client '%40s'!",
                        clientId.c_str(),
                        surfaceId,
                        screenAreaId,
                        lClientState->GetClientId().c_str()));
      }
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleDeregisterSurface(const std::string& clientId,
      const RequestArg& reqArg,
      UInt32 surfaceId,
      bool sendToIlm)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg.RequestId(),
                                        reqArg.AppId(),
                                        reqArg.UserData(),
                                        clientId);

   ETG_TRACE_USR4(("Receive request DeregisterSurface(%40s, {%u, %u, %u}, %u, %d)",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData(),
                   surfaceId,
                   sendToIlm));

   ClientState* lClientState = GetClientState(surfaceId);
   if (0 != lClientState)
   {
      if (lClientState->GetClientId() == clientId)
      {
         // Propagate DeregisterSurface to PopupManagerActivator plugin (if exists)
         IPopupManagerActivator* lPopupManagerActivator =
            PluginManager::GetInstance().Plugin<IPopupManagerActivator>();
         if (0 != lPopupManagerActivator)
         {
            lPopupManagerActivator->DeregisterSurface(surfaceId);
         }

         // Propagate DeregisterSurface to ScreenBrokerActivator plugin (if exists)
         IScreenBrokerActivator* lScreenBrokerActivator =
            PluginManager::GetInstance().Plugin<IScreenBrokerActivator>();
         if (0 != lScreenBrokerActivator)
         {
            lScreenBrokerActivator->DeregisterSurface(surfaceId);
         }

         ETG_TRACE_USR1(("Deregister surface %u",
                         surfaceId));

         // Notify surface state change before deregister surface to avoid invalid clientstate
         NotifySurfaceStateChange(lServiceRequestArg.UserData(),
                                  surfaceId,
                                  SurfaceState::Deregistered);

         lClientState->DeregisterSurface(surfaceId,
                                         sendToIlm);
#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
         LayerManagerAccessor::popFromCofiguredSurfaceList(surfaceId);
#endif
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': DeregisterSurface for surface ID:%u failed, surface is registered to client %40s!",
                        clientId.c_str(),
                        surfaceId,
                        lClientState->GetClientId().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': Failed to get client state for Surface %u to deregister Surface!",
                     clientId.c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleActivateApplication(const std::string& clientId,
      const RequestArg& reqArg,
      UInt32 surfaceId,
      UInt32 entryCustomAnimationType,
      UInt32 exitCustomAnimationType)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   ETG_TRACE_USR4(("Receive request ActivateApplication(%40s, {%u, %u, %u}, %u, %u, %u)",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData(),
                   surfaceId,
                   entryCustomAnimationType,
                   exitCustomAnimationType));

   // Retrieve client state to which surface was registered
   ClientState* lClientState = GetClientState(surfaceId);

   // If surface is not registered to any client, try to perform on-demand surface registration.
   if (0 == lClientState)
   {
      OnDemandSurfaceRegistration(reqArg, clientId, surfaceId);
      // Get newly registered client state, which has the surface now registered
      lClientState = GetClientState(surfaceId);
   }

#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL
   if (!LayerManagerAccessor::isValidSurface(surfaceId))
   {
      ETG_TRACE_ERR(("Client '%10s': Failed to register surface %u on ActivateApplication",
                     clientId.c_str(),
                     surfaceId));
      return;
   }

#endif

   // Activate surface
   if (0 != lClientState)
   {
      if (lClientState->GetClientId() == clientId)
      {
         lClientState->Activate(lServiceRequestArg,
                                surfaceId,
                                entryCustomAnimationType,
                                exitCustomAnimationType);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': Application activation for surface ID:%u failed, surface is registered to client %40s! with CustomAnimationTypes (%u, %u)",
                        clientId.c_str(),
                        surfaceId,
                        lClientState->GetClientId().c_str(),
                        entryCustomAnimationType,
                        exitCustomAnimationType));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': Application activation for surface ID:%u failed, surface is not registered!",
                     clientId.c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleStartAnimation(const std::string& clientId,
      const RequestArg& reqArg,
      UInt32 surfaceId,
      UInt32 animationType,
      UInt32 hint)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   ETG_TRACE_USR4(("Receive request StartAnimation(%40s, {%u, %u, %u}, %u, %u, %u)",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData(),
                   surfaceId,
                   animationType,
                   hint));

   // Retrieve client state to which surface was registered
   ClientState* lClientState = GetClientState(surfaceId);

   /*  // If surface is not registered to any client, try to perform on-demand surface registration.
     if (0 == lClientState)
     {
        OnDemandSurfaceRegistration(reqArg, clientId, surfaceId);
        // Get newly registered client state, which has the surface now registered
        lClientState = GetClientState(surfaceId);
     }*/

   // Activate surface
   if (0 != lClientState)
   {
      if (lClientState->GetClientId() == clientId)
      {
         lClientState->TransitionAnimation(lServiceRequestArg,
                                           surfaceId,
                                           animationType,
                                           hint);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': transition Animation for surface ID:%u failed, surface is registered to client %40s!",
                        clientId.c_str(),
                        surfaceId,
                        lClientState->GetClientId().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': transition Animation for surface ID:%u failed, surface is not registered!",
                     clientId.c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleAttachandAnimateLayer(const std::string& clientId,
      const RequestArg& reqArg,
      UInt32 layerId,
      std::vector<UInt32> surfaceIdList,
      UInt32 animationType,
      UInt32 hint)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   ETG_TRACE_USR4(("Receive request AttachandAnimateLayer(%40s, {%u, %u, %u}, %u, %u, %u)",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData(),
                   layerId,
                   animationType,
                   hint));
   // Retrieve client state to which first surface was registered
   if (!surfaceIdList.empty())
   {
      UInt32 lsurfaceId = surfaceIdList.front();
      ClientState* lClientState = GetClientState(lsurfaceId);
      if (0 != lClientState)
      {
         lClientState->AttachandAnimateLayer(lServiceRequestArg, layerId, surfaceIdList, animationType, hint);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': transition Animation for Layer ID:%u Failed",
                        clientId.c_str(),
                        layerId));
      }
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleShowPopup(const std::string& clientId,
                                     const RequestArg& reqArg,
                                     UInt32 surfaceId,
                                     const PopupPresentationArg& popupPresentationArg)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   {
      char str[230];
      sprintf(str, "Receive ShowPopup(%s, {%u, %u, %u}, %u, {%u, %d, %u, %u, %u, %d, %d, %d, %d} )",
              clientId.c_str(),
              lServiceRequestArg.RequestId(),
              lServiceRequestArg.AppId(),
              lServiceRequestArg.UserData(),
              surfaceId,
              popupPresentationArg.Priority(),
              popupPresentationArg.Modality(),
              popupPresentationArg.PresentationTime(),
              popupPresentationArg.MinimumPresentationTime(),
              popupPresentationArg.ValidityPeriod(),
              popupPresentationArg.HorizontalAlignment(),
              popupPresentationArg.VerticalAlignment(),
              popupPresentationArg.Scaling(),
              popupPresentationArg.FocusPriority());
      ETG_TRACE_USR4(("%s", str));
   }

   // Retrieve client state to which surface was registered
   ClientState* lClientState = GetClientState(surfaceId);

   // If surface is not registered to any client, try to perform on-demand surface registration.
   if (0 == lClientState)
   {
      OnDemandSurfaceRegistration(reqArg, clientId, surfaceId);
      // Get newly registered client state, which has the surface now registered
      lClientState = GetClientState(surfaceId);
   }

#ifdef VARIANT_S_FTR_ENABLE_IVI_SHELL

   if (!LayerManagerAccessor::isValidSurface(surfaceId))
   {
      ETG_TRACE_ERR(("Client '%10s': Failed to register surface %u on Show Popup ",
                     clientId.c_str(),
                     surfaceId));
      return;
   }

#endif

   if (0 != lClientState)
   {
      if (lClientState->GetClientId() == clientId)
      {
         // Retrieve client state and propagate the request to popup manager
         PopupManager::GetInstance().ShowPopup(lClientState,
                                               lServiceRequestArg,
                                               surfaceId,
                                               popupPresentationArg);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': ShowPopup for surface ID:%u failed, surface is registered to client %40s!",
                        clientId.c_str(),
                        surfaceId,
                        lClientState->GetClientId().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': ShowPopup for surface ID:%u failed, surface is not registered!",
                     clientId.c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleHidePopup(const std::string& clientId,
                                     const RequestArg& reqArg,
                                     UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   ETG_TRACE_USR4(("Receive HidePopup(%40s, {%u, %u, %u}, %u)",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData(),
                   surfaceId));

   // Retrieve client state to which surface was registered
   ClientState* lClientState = GetClientState(surfaceId);

   // If surface is not registered to any client, try to perform on-demand surface registration.
   if (0 == lClientState)
   {
      OnDemandSurfaceRegistration(reqArg, clientId, surfaceId);
      // Get newly registered client state, which has the surface now registered
      lClientState = GetClientState(surfaceId);
   }

   if (0 != lClientState)
   {
      if (lClientState->GetClientId() == clientId)
      {
         // Retrieve client state and propagate the request to popup manager
         PopupManager::GetInstance().HidePopup(lClientState,
                                               lServiceRequestArg,
                                               surfaceId);
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': HidePopup for surface ID:%u failed, surface is registered to client %40s!",
                        clientId.c_str(),
                        surfaceId,
                        lClientState->GetClientId().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': HidePopup for surface ID:%u failed, surface is not registered!",
                     clientId.c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleCloseOnExternalTouch(const std::string& clientId,
      const RequestArg& reqArg) const
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   ETG_TRACE_USR4(("Receive CloseOnExternalTouch (%40s, {%u, %u, %u})",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData()));

   // Propagate close on external touch request to PopupManagerActivator plugin (if exists)
   IPopupManagerActivator* lPopupManagerActivator = PluginManager::GetInstance().Plugin<IPopupManagerActivator>();
   if (0 != lPopupManagerActivator)
   {
      lPopupManagerActivator->CloseOnExternalTouch(lServiceRequestArg);
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleSetPopupFilter(const std::string& clientId,
      const RequestArg& reqArg,
      bool disableAll,
      const PopupPresentationArg& popupPresentationArg) const
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   {
      char str[230];
      sprintf(str, "Receive SetPopupFilter(%s, {%u, %u, %u}, %u, {%u, %d, %u, %u, %u, %d, %d, %d, %d} )",
              clientId.c_str(),
              lServiceRequestArg.RequestId(),
              lServiceRequestArg.AppId(),
              lServiceRequestArg.UserData(),
              disableAll,
              popupPresentationArg.Priority(),
              popupPresentationArg.Modality(),
              popupPresentationArg.PresentationTime(),
              popupPresentationArg.MinimumPresentationTime(),
              popupPresentationArg.ValidityPeriod(),
              popupPresentationArg.HorizontalAlignment(),
              popupPresentationArg.VerticalAlignment(),
              popupPresentationArg.Scaling(),
              popupPresentationArg.FocusPriority());
      ETG_TRACE_USR4(("%s", str));
   }

   // Propagate set filter request to PopupManagerActivator plugin (if exists)
   IPopupManagerActivator* lPopupManagerActivator = PluginManager::GetInstance().Plugin<IPopupManagerActivator>();
   if (0 != lPopupManagerActivator)
   {
      lPopupManagerActivator->SetPopupFilter(lServiceRequestArg, disableAll, popupPresentationArg);
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleClearPopupFilter(const std::string& clientId,
      const RequestArg& reqArg) const
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ServiceRequestArg lServiceRequestArg(reqArg,
                                        clientId);

   ETG_TRACE_USR4(("Receive ClearPopupFilter(%40s, {%u, %u, %u})",
                   clientId.c_str(),
                   lServiceRequestArg.RequestId(),
                   lServiceRequestArg.AppId(),
                   lServiceRequestArg.UserData()));

   // Propagate clear filter request to PopupManagerActivator plugin (if exists)
   IPopupManagerActivator* lPopupManagerActivator = PluginManager::GetInstance().Plugin<IPopupManagerActivator>();
   if (0 != lPopupManagerActivator)
   {
      lPopupManagerActivator->ClearPopupFilter(lServiceRequestArg);
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleKeyFeedback(const std::string& clientId,
                                       UInt32 userData,
                                       UInt32 keyCode,
                                       KeyState::Enum keyState,
                                       bool consumed) const
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ETG_TRACE_USR4(("Receive KeyFeedback(%40s, %u, %u, %d, %d)",
                   clientId.c_str(),
                   userData,
                   keyCode,
                   keyState,
                   consumed));

   // Propagate key feedback to KeyHandler plugin (if exists)
   IKeyHandler* lKeyHandler = PluginManager::GetInstance().Plugin<IKeyHandler>();
   if (0 != lKeyHandler)
   {
      lKeyHandler->KeyFeedback(userData,
                               keyCode,
                               keyState,
                               consumed);
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleAction(const std::string& clientId,
                                  UInt32 actionId,
                                  UInt32 actionData) const
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ETG_TRACE_USR4(("Receive HandleAction(%40s, %u, %u)",
                   clientId.c_str(),
                   actionId,
                   actionData));

   // Propagate current status request to respective plugin (if exists)
   for (Int i = static_cast<Int>(PluginId::First);
         i < static_cast<Int>(PluginId::Count);
         ++i)
   {
      IScreenBrokerPlugin* lPlugin =
         Internal::PluginManager::GetInstance().GetPlugin(static_cast<PluginId::Enum>(i));
      if (0 != lPlugin)
      {
         lPlugin->Action(actionId, actionData);
      }
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleSetInputFocus(const std::string& clientId,
      UInt32 surfaceId,
      bool keyboardFocus,
      bool pointerFocus,
      Int32 priority,
      UInt32 focusData)
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ETG_TRACE_USR4(("Receive HandleSetInputFocus(%40s, %u, %d, %d, %d, %u)",
                   clientId.c_str(),
                   surfaceId,
                   keyboardFocus,
                   pointerFocus,
                   priority,
                   focusData));

   // Retrieve client state to which surface was registered
   ClientState* lClientState = GetClientState(surfaceId);

   if (0 != lClientState)
   {
      if (lClientState->GetClientId() == clientId)
      {
         // Propagate set input focus to PopupManagerActivator plugin (if exists)
         IPopupManagerActivator* lPopupManagerActivator =
            PluginManager::GetInstance().Plugin<IPopupManagerActivator>();
         if (0 != lPopupManagerActivator)
         {
            lPopupManagerActivator->SetInputFocus(surfaceId, keyboardFocus, pointerFocus, priority, focusData);
         }

         // Propagate set input focus to ScreenBrokerActivator plugin (if exists)
         IScreenBrokerActivator* lScreenBrokerActivator =
            PluginManager::GetInstance().Plugin<IScreenBrokerActivator>();
         if (0 != lScreenBrokerActivator)
         {
            lScreenBrokerActivator->SetInputFocus(surfaceId, keyboardFocus, pointerFocus, priority, focusData);
         }
      }
      else
      {
         ETG_TRACE_ERR(("Client '%10s': SetInputFocus for surface ID:%u failed, surface is registered to client %40s!",
                        clientId.c_str(),
                        surfaceId,
                        lClientState->GetClientId().c_str()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': Failed to get client state for Surface %u to set input focus!",
                     clientId.c_str(),
                     surfaceId));
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::HandleRequestCurrentStatus(const std::string& clientId,
      UInt32 requestId) const
{
   //SCREENBROKER_LOG_FN();
   // Synchronize (execution scope) with screen broker service
   SCREENBROKER_SYNCHRONIZE();

   ETG_TRACE_USR4(("Receive HandleRequestCurrentStatus(%40s, %u)",
                   clientId.c_str(),
                   requestId));

   // Propagate current status request to respective plugin (if exists)
   for (Int i = static_cast<Int>(PluginId::First);
         i < static_cast<Int>(PluginId::Count);
         ++i)
   {
      IScreenBrokerPlugin* lPlugin =
         Internal::PluginManager::GetInstance().GetPlugin(static_cast<PluginId::Enum>(i));
      if (0 != lPlugin)
      {
         lPlugin->RequestCurrentStatus(requestId);
      }
   }
}


// ------------------------------------------------------------------------
void ServiceAdaptor::OnDemandSurfaceRegistration(const RequestArg& reqArg,
      const std::string& clientId,
      UInt32 surfaceId)
{
   //SCREENBROKER_LOG_FN();
   StaticSurfaceContextMap::iterator it = mStaticSurfaceContextMap.find(surfaceId);
   if (mStaticSurfaceContextMap.end().operator != (it))
   {
      RequestArg lRegArg(reqArg.RequestId(), reqArg.AppId(), (*it).second.Hint());
      HandleRegisterSurface(clientId,
                            lRegArg,
                            (*it).second.LayerId(),
                            surfaceId);
   }
   else
   {
      ETG_TRACE_ERR(("Client '%10s': On demand registration for surface ID:%u failed, static layer mapping for surface is missing!",
                     clientId.c_str(),
                     surfaceId));
   }
}


}
}
